#include <locations.h>
#include <math.h>
#include "distance_tracker.h"
#include "acceleration_sensor.h"
#include "$(appName).h"
#include "view.h"

struct distdata {
	location_manager_h location_manager;
	double total_distance;
	double prev_latitude;
	double prev_longitude;
} s_tracker_data = {
	.location_manager = NULL,
	.total_distance = 0.0,
	.prev_latitude = LAT_UNINITIATED,
	.prev_longitude = LONG_UNINITIATED
};

static void
__pos_updated_cb(double latitude,
					double longitude,
					double altitude,
					time_t timestamp,
					void *data)
{
	int steps_count;
	double distance = 0;

	/* If callback is called for the first time, set prev values */
	if (fabs(s_tracker_data.prev_latitude - LAT_UNINITIATED) < DOUBLE_COMPARIZON_THRESHOLD &&
		fabs(s_tracker_data.prev_longitude - LONG_UNINITIATED) < DOUBLE_COMPARIZON_THRESHOLD) {
		s_tracker_data.prev_latitude = latitude;
		s_tracker_data.prev_longitude = longitude;
		return;
	}

	/* Calculate distance between previous and current location data and
	 * update view */
	int ret = location_manager_get_distance(
									latitude,
									longitude,
									s_tracker_data.prev_latitude,
									s_tracker_data.prev_longitude, &distance);
	if (ret != LOCATIONS_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get distance");
		return;
	}

	s_tracker_data.total_distance += distance;
	view_set_total_distance(s_tracker_data.total_distance);

	/* Set new prev values */
	s_tracker_data.prev_latitude = latitude;
	s_tracker_data.prev_longitude = longitude;

	/* Calculate and set step length if number of steps is greater than 0 */
	steps_count = acceleration_sensor_get_steps_count();

	if (steps_count)
		view_set_step_length(s_tracker_data.total_distance/steps_count);
}

static void
__gps_status_changed_cb(location_method_e method, bool enable, void *data)
{
	if (method == LOCATIONS_METHOD_GPS) {
		view_set_gps_ok_text(enable);
		if (enable) {
			if (!s_tracker_data.location_manager)
				distance_tracker_init();

			distance_tracker_start();
		} else {
			distance_tracker_stop();
		}
	}
}

bool
distance_tracker_check_gps(void)
{
	bool gps_enabled = false;

	/* Check if GPS is enabled and set callback for GPS status updates */
	int ret = location_manager_is_enabled_method(LOCATIONS_METHOD_GPS,
													&gps_enabled);
	if (ret != LOCATIONS_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get GPS status");
		return false;
	}

	ret = location_manager_set_setting_changed_cb(LOCATIONS_METHOD_GPS,
													__gps_status_changed_cb,
													NULL);
	if (ret != LOCATIONS_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to register callback for GPS status updates");
		return false;
	}

	if (!gps_enabled) {
		dlog_print(DLOG_ERROR, LOG_TAG, "GPS not enabled");
		return false;
	}

	view_set_gps_ok_text(gps_enabled);

	return true;
}

bool
distance_tracker_init(void)
{
	/* Create location manager and set callback for position updates */
	int ret = location_manager_create(LOCATIONS_METHOD_GPS,
										&s_tracker_data.location_manager);
	if (ret != LOCATIONS_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create location manager");
		return false;
	}

	ret = location_manager_set_position_updated_cb(
												s_tracker_data.location_manager,
												__pos_updated_cb,
												4,
												NULL);
	if (ret != LOCATIONS_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to register callback for position update");
		distance_tracker_destroy();
		return false;
	}

	return true;
}

void
distance_tracker_destroy(void)
{
	if (s_tracker_data.location_manager) {
		int ret = location_manager_destroy(s_tracker_data.location_manager);
		if (ret != LOCATIONS_ERROR_NONE)
			dlog_print(DLOG_ERROR, LOG_TAG, "Failed to destroy location manager");
	}
}

void
distance_tracker_start(void)
{
	if (!s_tracker_data.location_manager &&
		!distance_tracker_init()) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Location manager not initialized");
		return;
	}

	int ret = location_manager_start(s_tracker_data.location_manager);
	if (ret != LOCATIONS_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to start location manager");
}

void
distance_tracker_stop(void)
{
	if (s_tracker_data.location_manager) {
		int ret = location_manager_stop(s_tracker_data.location_manager);
		if (ret != LOCATIONS_ERROR_NONE)
			dlog_print(DLOG_ERROR, LOG_TAG, "Failed to stop location manager");
	}
}
