#include <tizen.h>
#include <activity_recognition.h>
#include "$(appName).h"
#include "acceleration_sensor.h"
#include "distance_tracker.h"
#include "view.h"

struct infodata {
	activity_h *stationary_handle;
	activity_h *walk_handle;
	activity_h *run_handle;
	activity_type_e activity_type;
} s_pedometer_info = {
	.stationary_handle = NULL,
	.walk_handle = NULL,
	.run_handle = NULL,
	.activity_type = ACTIVITY_STATIONARY
};

static void
__activity_cb(activity_type_e type,
				const activity_data_h data,
				double timestamp,
				activity_error_e error,
				void *user_data)
{
	s_pedometer_info.activity_type = type;

	if (type == ACTIVITY_WALK) {
		view_set_activity_type(WALK_STR);
	} else if (type == ACTIVITY_RUN) {
		view_set_activity_type(RUN_STR);
	} else if (type == ACTIVITY_STATIONARY) {
		view_set_activity_type(STATIONARY_STR);
	}
}

static activity_h *
__init_activity(activity_type_e type)
{
	activity_h *handle = NULL;
	bool supported = false;

	if (activity_is_supported(type, &supported) != ACTIVITY_ERROR_NONE ||
		!supported) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Activity not supported on current device: %d", type);
		return NULL;
	}

	if (activity_create(handle) != ACTIVITY_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to initialize activity handle: %d", type);
		return NULL;
	}

	if (activity_start_recognition(*handle,
										type,
										__activity_cb,
										NULL) != ACTIVITY_ERROR_NONE) {
		dlog_print(DLOG_INFO, LOG_TAG, "Failed to start activity recognition: %d", type);
		activity_release(*handle);
		return NULL;
	}

	return handle;
}

static void
__init_acitivity_monitor(void)
{
	s_pedometer_info.stationary_handle = __init_activity(ACTIVITY_STATIONARY);
	s_pedometer_info.walk_handle = __init_activity(ACTIVITY_WALK);
	s_pedometer_info.run_handle = __init_activity(ACTIVITY_RUN);
}

static void
__release_activity_monitor(void)
{
	if (s_pedometer_info.stationary_handle) {
		activity_stop_recognition(*s_pedometer_info.stationary_handle);
		activity_release(*s_pedometer_info.stationary_handle);
	}

	if (s_pedometer_info.walk_handle) {
		activity_stop_recognition(*s_pedometer_info.walk_handle);
		activity_release(*s_pedometer_info.walk_handle);
	}

	if (s_pedometer_info.run_handle) {
		activity_stop_recognition(*s_pedometer_info.run_handle);
		activity_release(*s_pedometer_info.run_handle);
	}
}

static bool
__create_app(void *data)
{
	if (!view_create_base_gui())
		return false;

	__init_acitivity_monitor();

	if (distance_tracker_check_gps())
		distance_tracker_init();

	return acceleration_sensor_init_handle();
}

static void
__control_app(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void
__pause_app(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
__resume_app(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
__terminate_app(void *data)
{
	/* Release all resources. */
	__release_activity_monitor();
	distance_tracker_destroy();
	acceleration_sensor_release_handle();
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/* APP_EVENT_LANGUAGE_CHANGED */
	return;
}

static void
ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/* APP_EVENT_DEVICE_ORIENTATION_CHANGED */
	return;
}

static void
ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/* APP_EVENT_REGION_FORMAT_CHANGED */
}

static void
ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/* APP_EVENT_LOW_BATTERY */
}

static void
ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}

int
main(int argc, char *argv[])
{
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = __create_app;
	event_callback.terminate = __terminate_app;
	event_callback.pause = __pause_app;
	event_callback.resume = __resume_app;
	event_callback.app_control = __control_app;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY],
								APP_EVENT_LOW_BATTERY,
								ui_app_low_battery,
								NULL);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY],
								APP_EVENT_LOW_MEMORY,
								ui_app_low_memory,
								NULL);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED],
								APP_EVENT_DEVICE_ORIENTATION_CHANGED,
								ui_app_orient_changed,
								NULL);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED],
								APP_EVENT_LANGUAGE_CHANGED,
								ui_app_lang_changed,
								NULL);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED],
								APP_EVENT_REGION_FORMAT_CHANGED,
								ui_app_region_changed,
								NULL);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, NULL);
	if (ret != APP_ERROR_NONE)
		dlog_print(DLOG_ERROR,
						LOG_TAG,
						"ui_app_main() is failed. err = %d",
						ret);

	return ret;
}
