#include "view.h"
#include "acceleration_sensor.h"
#include "distance_tracker.h"

struct viewdata {
	Evas_Object *win;
	Evas_Object *layout;
	Evas_Object *conform;
} s_view_info = {
	.win = NULL,
	.layout = NULL,
	.conform = NULL
};

static void
__delete_win_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

static void
__layout_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	/* Let window go to hide state. */
	elm_win_lower(s_view_info.win);
}

static void
__get_app_resource(const char *edj_file_in, char *edj_path_out, int edj_path_max)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, edj_path_max, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

static void
__start_cb(void *data, Evas_Object *obj, void *event)
{
	acceleration_sensor_start();
	distance_tracker_start();
}

static void
__stop_cb(void *data, Evas_Object *obj, void *event)
{
	acceleration_sensor_stop();
	distance_tracker_stop();
}

static Evas_Object *
__create_button(Evas_Object *parent, char *btn_text, Evas_Smart_Cb func)
{
	Evas_Object *btn = elm_button_add(parent);
	if (!btn) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create button");
		return NULL;
	}

	elm_object_text_set(btn, btn_text);
	evas_object_smart_callback_add(btn, "clicked", func, NULL);
	evas_object_show(btn);

	return btn;
}

static Evas_Object *
__create_base_layout(Evas_Object *parent)
{
	char edj_path[PATH_MAX] = {0, };
	Evas_Object *layout;

	__get_app_resource(EDJ_FILE, edj_path, (int)PATH_MAX);

	layout = elm_layout_add(parent);
	if (!layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create layout");
		return NULL;
	}

	if (!elm_layout_file_set(layout, edj_path, GRP_MAIN)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to set layout file");
		evas_object_del(layout);
		return NULL;
	}

	/* Initialize text parts */
	elm_object_part_text_set(layout, PART_GPS_STATUS, GPS_NOT_DETECTED);
	elm_object_part_text_set(layout, PART_ACTIVITY_TEXT, NOT_AVAILABLE);
	elm_object_part_text_set(layout, PART_STEPS_TEXT, STEPS_0);
	elm_object_part_text_set(layout, PART_STEP_LENGTH_TEXT, NOT_AVAILABLE);
	elm_object_part_text_set(layout, PART_DISTANCE_TEXT, NOT_AVAILABLE);

	return layout;
}

static Evas_Object *
__create_conformant(Evas_Object *parent)
{
	Evas_Object *conform = elm_conformant_add(parent);
	if (!conform) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create conformant");
		return NULL;
	}

	elm_win_indicator_mode_set(parent, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(parent, ELM_WIN_INDICATOR_OPAQUE);
	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(parent, conform);
	evas_object_show(conform);

	return conform;
}

void
view_set_gps_ok_text(bool gps_ok)
{
	if (gps_ok) {
		elm_object_part_text_set(s_view_info.layout,
									PART_GPS_STATUS,
									GPS_OK_TEXT);
	} else {
		elm_object_part_text_set(s_view_info.layout,
									PART_GPS_STATUS,
									GPS_NOT_DETECTED);
	}
}

void
view_set_activity_type(char *type)
{
	elm_object_part_text_set(s_view_info.layout, PART_ACTIVITY_TEXT, type);
}

void
view_set_steps_count(int count)
{
	char steps[BUF_MAX] = {0, };
	snprintf(steps, BUF_MAX, "%d", count);
	elm_object_part_text_set(s_view_info.layout, PART_STEPS_TEXT, steps);
}

void
view_set_step_length(double length)
{
	char len[BUF_MAX] = {0, };
	snprintf(len, BUF_MAX, "%gm", length);
	elm_object_part_text_set(s_view_info.layout, PART_STEP_LENGTH_TEXT, len);
}

void
view_set_total_distance(double distance)
{
	char dist[BUF_MAX] = {0, };
	snprintf(dist, BUF_MAX, "%gm", distance);
	elm_object_part_text_set(s_view_info.layout, PART_DISTANCE_TEXT, dist);
}

bool
view_create_base_gui(void)
{
	Evas_Object *start_button, *stop_button;

	/* Window */
	s_view_info.win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	if (!s_view_info.win) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create window");
		return false;
	}

	elm_win_conformant_set(s_view_info.win, EINA_TRUE);
	elm_win_autodel_set(s_view_info.win, EINA_TRUE);
	evas_object_smart_callback_add(s_view_info.win,
									"delete,request",
									__delete_win_request_cb,
									NULL);

	/* Conformant */
	s_view_info.conform = __create_conformant(s_view_info.win);
	if (!s_view_info.conform) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create conformant");
		evas_object_del(s_view_info.win);
		return false;
	}

	/* Base Layout */
	s_view_info.layout = __create_base_layout(s_view_info.win);
	if (!s_view_info.layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create base layout");
		evas_object_del(s_view_info.win);
		return false;
	}

	/* Initialize buttons */
	start_button = __create_button(s_view_info.win, BTN_START_TEXT, __start_cb);
	stop_button = __create_button(s_view_info.win, BTN_STOP_TEXT, __stop_cb);
	if (!start_button || !stop_button) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create button");
		evas_object_del(s_view_info.win);
		return false;
	}

	elm_object_part_content_set(s_view_info.layout,
									PART_START_BTN,
									start_button);
	elm_object_part_content_set(s_view_info.layout,
									PART_STOP_BTN,
									stop_button);

	evas_object_size_hint_weight_set(s_view_info.layout,
										EVAS_HINT_EXPAND,
										EVAS_HINT_EXPAND);
	eext_object_event_callback_add(s_view_info.layout,
										EEXT_CALLBACK_BACK,
										__layout_back_cb,
										NULL);
	elm_object_content_set(s_view_info.conform, s_view_info.layout);

	/* Show window after base gui is set up */
	evas_object_show(s_view_info.win);

	return true;
}
