#include <stdlib.h>
#include "$(appName).h"
#include "controller.h"

#define BOOL_STR_TRUE "TRUE"
#define BOOL_STR_FALSE "FALSE"
#define TRESHOLD 0.0001

static bool __string_to_int(const char *string, int *value)
{
	if (!value) {
		controller_log(DLOG_INFO, "Function string_to_int() failed");
		return false;
	}

	*value = atoi(string);

	if (*value == 0)
		return (strlen(string) == 1 && string[0] == '0');

	return true;
}

static bool __string_to_double(const char *string, double *value)
{
	if (!value) {
		controller_log(DLOG_INFO, "Function __string_to_double() failed");
		return false;
	}

	*value = atof(string);

	if (fabs(*value) < TRESHOLD)
		return ((strlen(string) == 1 && string[0] == '0') ||
				(strlen(string) == 3 && !strncmp(string, "0.0", 3)));

	return true;
}

static bool __string_to_bool(const char *string, bool *value)
{
	if (!value) {
		controller_log(DLOG_INFO, "Function __string_to_bool() failed");
		return false;
	}

	if (controller_check_same_string((char *)string, BOOL_STR_TRUE)) {
		*value = true;
	} else if (controller_check_same_string((char *)string, BOOL_STR_FALSE)) {
		*value = false;
	} else {
		return false;
	}

	return true;
}

void __decompose_property_items_enum_key_type(const char *items_enum,
												Eina_List **list)
{
	if (!items_enum)
		return;

	key_value_t *key_val = NULL;
	int i = 0;
	const char *substr = strtok((char *)items_enum, ":;");

	while (substr != NULL) {
		if (i % 2 == 0) {
			key_val = (key_value_t *)malloc(sizeof(key_value_t));
			key_val->key = strdup(substr);
		} else {
			if (__string_to_int(substr, (int *)&key_val->value_type) &&
					key_val->value_type >= PREF_INTEGER && key_val->value_type < PREF_MAX) {

				*list = eina_list_append(*list, (void *)key_val);

			} else {

				free(key_val->key);
				free(key_val);
			}
		}

		substr = strtok(NULL, ":;");
		i++;
	}
}

bool __compose_property_items_enum_key_type(char *items_enum,
											const char *key_name,
											pref_value_type_t value_type,
											char **items_enum_new)
{
	*items_enum_new = NULL;

	Eina_List *list = NULL;
	__decompose_property_items_enum_key_type(items_enum, &list);

	Eina_Strbuf *items_enum_buff = eina_strbuf_new();
	if (!items_enum_buff) {
		controller_remove_key_value_list(list);
		controller_log(DLOG_ERROR, "Function eina_strbuf_new() failed");
		return false;
	}

	char *key = strdup(key_name);
	eina_str_tolower(&key);

	Eina_List *it = NULL;
	key_value_t *key_val = NULL;

	EINA_LIST_FOREACH(list, it, key_val) {
		if (!key_val)
			continue;

		if (strlen(key_val->key) == strlen(key) &&
			!strncmp(key_val->key, key, strlen(key_val->key)))
			continue;

		eina_strbuf_append_printf(items_enum_buff,
									"%s:%d;",
									key_val->key,
									(int)key_val->value_type);
	}

	char value_type_str[3] = {0,};
	snprintf(value_type_str, sizeof(value_type_str), "%d", (int)value_type);

	eina_strbuf_append_printf(items_enum_buff, "%s:%s;", key, value_type_str);

	*items_enum_new = strdup(eina_strbuf_string_get(items_enum_buff));

	free(key);
	eina_strbuf_free(items_enum_buff);
	controller_remove_key_value_list(list);

	return true;
}

void __preference_read_valid_cb(key_value_t *key_value)
{
	view_update_genlist_item(key_value);
}

bool controller_initialize(viewdata_s *vd, modeldata_s *md)
{
	Eina_List *pref_types = NULL;

	if (!controller_get_property_items_enum(&pref_types))
		return false;

	if (!view_create_base_gui(vd))
		return false;

	model_create(md, pref_types, __preference_read_valid_cb);

	controller_remove_key_value_list(pref_types);

	return true;
}

bool controller_set_preference(const char *key,
								const char *value,
								pref_value_type_t type)
{
	int int_value = 0;
	double double_value = 0.0;
	bool bool_value = false;
	bool ret = false;

	switch (type) {
	case PREF_INTEGER:
		if (__string_to_int(value, &int_value))
			ret = model_set_preference_int(key, int_value);
		break;
	case PREF_DOUBLE:
		if (__string_to_double(value, &double_value))
			ret = model_set_preference_double(key, double_value);
		break;
	case PREF_BOOL:
		if (__string_to_bool(value, &bool_value))
			ret = model_set_preference_bool(key, bool_value);
		break;
	case PREF_STRING:
		ret = model_set_preference_string(key, value);
		break;
	default:
		controller_log(DLOG_WARN, "Unknown value type");
		return false;
	}

	if (!ret) {
		controller_log(DLOG_ERROR, "Function model_preference_*_set() failed");
		return false;
	}

	return true;
}

bool controller_get_preference(const char *key,
								pref_value_type_t type,
								void **value)
{
	int int_value = 0;
	int size_val = 0;
	double dbl_value = 0.0;
	bool bool_value = false;
	bool ret = false;
	char *str_value = NULL;
	void *ptr_val = NULL;

	switch (type) {
	case PREF_INTEGER:
		ret = model_get_preference_int(key, &int_value);
		ptr_val = (void *)&int_value;
		size_val = sizeof(int);
		break;
	case PREF_DOUBLE:
		ret = model_get_preference_double(key, &dbl_value);
		ptr_val = (void *)&dbl_value;
		size_val = sizeof(double);
		break;
	case PREF_BOOL:
		ret = model_get_preference_bool(key, &bool_value);
		ptr_val = (void *)&bool_value;
		size_val = sizeof(bool);
		break;
	case PREF_STRING:
		ret = model_get_preference_string(key, &str_value);
		ptr_val = (void *)str_value;
		size_val = strlen(str_value) + 1;
		break;
	default:
		controller_log(DLOG_WARN, "Unknown value type");
		return false;
	}

	if (!ret) {
		controller_log(DLOG_ERROR, "Function model_preference_*_get() failed");
		return false;
	}

	if (size_val > 0) {
		*value = (void *)malloc(size_val);
		memcpy(*value, ptr_val, size_val);
	}

	return true;
}

bool controller_add_property_items_enum(const char *key_name,
											pref_value_type_t value_type)
{
	bool key_exists = model_check_preference_exists(PROPERTY_ITEMS_ENUM_KEY);
	char *items_enum_value = NULL;

	if (key_exists)
		if (!model_get_preference_string(PROPERTY_ITEMS_ENUM_KEY,
										&items_enum_value))
			return false;

	char *items_enum_new = NULL;
	bool ret = __compose_property_items_enum_key_type(items_enum_value,
														key_name,
														value_type,
														&items_enum_new);

	if (items_enum_value)
		free(items_enum_value);

	if (!ret)
		return false;

	if (!model_set_preference_string(PROPERTY_ITEMS_ENUM_KEY, items_enum_new)) {
		free(items_enum_new);
		return false;
	}

	free(items_enum_new);
	return true;
}

bool controller_get_property_items_enum(Eina_List **properties_enum)
{
	bool key_exists = model_check_preference_exists(PROPERTY_ITEMS_ENUM_KEY);
	char *items_enum_value = NULL;

	if (key_exists)
		if (!model_get_preference_string(PROPERTY_ITEMS_ENUM_KEY,
										&items_enum_value))
			return false;

	__decompose_property_items_enum_key_type(items_enum_value, properties_enum);

	if (items_enum_value)
		free(items_enum_value);

	return true;
}

void controller_remove_key_value(key_value_t *key_value)
{
	if (!key_value)
		return;

	if (key_value->key)
		free(key_value->key);

	free(key_value);
}

void controller_remove_key_value_list(Eina_List *list)
{
	if (!list)
		return;

	key_value_t *key_val = NULL;

	EINA_LIST_FREE(list, key_val)
		controller_remove_key_value(key_val);
}

void controller_log(log_priority pri, const char *format, ...)
{
	char buff[512] = {0,};
	va_list vl;

	va_start(vl, format);
	vsnprintf(buff, sizeof(buff), format, vl);
	va_end(vl);

	dlog_print(pri, LOG_TAG, buff);
}

bool controller_check_same_string(char *str1, char *str2)
{
	if (!str1 || !str2)
		return false;

	char *s1 = strdup(str1);
	char *s2 = strdup(str2);

	eina_str_tolower(&s1);
	eina_str_tolower(&s2);

	bool comp = (bool)(strlen(s1) == strlen(s2) &&
						!strncmp(s1, s2, strlen(s1)));

	free(s1);
	free(s2);

	return comp;
}

bool controller_remove_preference(const char *key)
{
	if (!model_check_preference_exists(key)) {
		controller_log(DLOG_WARN, "Preference '%s' does not exist.", key);
		return false;
	}

	return model_remove_preference(key);
}
