#include <stddef.h>
#include <app_preference.h>
#include <app_common.h>
#include "model.h"
#include "controller.h"

typedef struct _pref_foreach_data {
	Eina_List *pref_types;
	preference_read_cb func_cb;
} pref_foreach_data_s;


modeldata_s *modeldata = NULL;


static void __preference_item_changed_cb(const char *key, void *user_data)
{
	controller_log(DLOG_INFO, "Preferences changed: %s", key);
}

static bool __preference_item_enum_cb(const char *key, void *user_data)
{
	pref_foreach_data_s *data = (pref_foreach_data_s *)user_data;
	Eina_List *it = NULL;
	key_value_t *key_val = NULL;
	int val_type = -1;

	EINA_LIST_FOREACH(data->pref_types, it, key_val) {
		if (!key_val)
			continue;

		if (!controller_check_same_string((char *)key, key_val->key))
			continue;

		val_type = (int)key_val->value_type;
		break;
	}

	if ((pref_value_type_t)val_type < PREF_INTEGER ||
		(pref_value_type_t)val_type >= PREF_MAX)
		return true;	/* continue preferences enumeration */

	controller_log(DLOG_INFO, "Preferences enum: %s, type: %d", key, val_type);

	if (data->func_cb) {
		key_value_t *kv = (key_value_t *)malloc(sizeof(key_value_t));
		kv->key = strdup(key_val->key);
		kv->value = NULL;
		kv->value_type = key_val->value_type;

		if (controller_get_preference(key_val->key,
										key_val->value_type,
										&kv->value)) {
			data->func_cb(kv);
		} else {
			free(kv->key);
			free(kv);
		}
	}

	int ret = preference_set_changed_cb(key,
										__preference_item_changed_cb,
										NULL);
	if (ret != PREFERENCE_ERROR_NONE)
		controller_log(DLOG_ERROR,
					"Function preference_set_changed_cb() failed with error %d",
					ret);

	return true;
}

void model_create(modeldata_s *md,
					Eina_List *pref_types,
					preference_read_cb func_cb)
{
	pref_foreach_data_s pref_foreach_data = {
		.pref_types = pref_types,
		.func_cb = func_cb,
	};

	modeldata = md;

	int ret = preference_foreach_item(__preference_item_enum_cb,
										(void *)&pref_foreach_data);
	if (ret != PREFERENCE_ERROR_NONE)
		controller_log(DLOG_ERROR,
					"Function preference_foreach_item() failed with error %d",
					ret);
}

void model_set_preference_type_selected(pref_value_type_t pref_value_type)
{
	modeldata->selected_pref_value_type = pref_value_type;
}

pref_value_type_t model_get_preference_type_selected(void)
{
	return modeldata->selected_pref_value_type;
}

bool model_remove_preferences(void)
{
	int ret = preference_remove_all();
	if (ret != PREFERENCE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
						"Function preference_remove_all() failed with error %d",
						ret);
		return false;
	}

	controller_log(DLOG_INFO, "All preferences removed successfully");

	return true;
}

bool model_remove_preference(const char *key)
{
	int ret = preference_remove(key);
	if (ret != PREFERENCE_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function preference_remove() failed.");
		return false;
	}

	return true;
}

bool model_set_preference_int(const char *key, int value)
{
	int ret = preference_set_int(key, value);
	if (ret != PREFERENCE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
						"Function preference_set_int() failed with error %d",
						ret);
		return false;
	}

	return true;
}

bool model_get_preference_int(const char *key, int *value)
{
	int ret = preference_get_int(key, value);
	if (ret != PREFERENCE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
						"Function preference_get_int() failed with error %d",
						ret);
		return false;
	}

	return true;
}

bool model_set_preference_double(const char *key, double value)
{
	int ret = preference_set_double(key, value);
	if (ret != PREFERENCE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
						"Function preference_set_double() failed with error %d",
						ret);
		return false;
	}

	return true;
}

bool model_get_preference_double(const char *key, double *value)
{
	int ret = preference_get_double(key, value);
	if (ret != PREFERENCE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
						"Function preference_get_double() failed with error %d",
						ret);
		return false;
	}

	return true;
}

bool model_set_preference_bool(const char *key, bool value)
{
	int ret = preference_set_boolean(key, value);
	if (ret != PREFERENCE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
					"Function preference_set_boolean() failed with error %d",
					ret);
		return false;
	}

	return true;
}

bool model_get_preference_bool(const char *key, bool *value)
{
	int ret = preference_get_boolean(key, value);
	if (ret != PREFERENCE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
					"Function preference_get_boolean() failed with error %d",
					ret);
		return false;
	}

	return true;
}

bool model_set_preference_string(const char *key, const char *value)
{
	int ret = preference_set_string(key, value);
	if (ret != PREFERENCE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
					"Function preference_set_string() failed with error %d",
					ret);
		return false;
	}

	return true;
}

bool model_get_preference_string(const char *key, char **value)
{
	int ret = preference_get_string(key, value);
	if (ret != PREFERENCE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
						"Function preference_get_string() failed with error %d",
						ret);
		return false;
	}

	return (*value != NULL);
}

bool model_check_preference_exists(const char *key)
{
	bool key_exists = false;
	int ret = preference_is_existing(key, &key_exists);
	if (ret != PREFERENCE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
					"Function preference_is_existing() failed with error %d",
					ret);
		return false;
	}

	return key_exists;
}