#include "$(appName).h"
#include "view.h"
#include "controller.h"
#include "model.h"
#include "defines.h"

#define REMOVE_ALL_PREFS_CAPTION "Remove all"
#define UPDATE_PREF_CAPTION "Update"
#define ICON_DELETE_FILE_NAME "icon_delete.png"
#define DELETE_IMAGE_SIZE_MIN 25


typedef struct _preftype {
	char *caption;
	pref_value_type_t preftype;
} preftype_s;

static viewdata_s *viewdata = NULL;

preftype_s preftypes[PREF_MAX] = {
	{ .caption = "Integer", .preftype = PREF_INTEGER, },
	{ .caption = "Double", .preftype = PREF_DOUBLE, },
	{ .caption = "Bool", .preftype = PREF_BOOL, },
	{ .caption = "String", .preftype = PREF_STRING, },
};


static char *__get_image_file_path(const char *file_name)
{
	char *res_path = app_get_resource_path();
	if (!res_path) {
		controller_log(DLOG_ERROR, "app_get_resource_path() failed.");
		return NULL;
	}

	char file_path[PATH_MAX] = {0,};
	snprintf(file_path, sizeof(file_path), "%simages/%s", res_path, file_name);

	free(res_path);

	return strdup(file_path);
}

static bool __find_genlist_item(const char *key, Elm_Object_Item **item)
{
	key_value_t *kv = NULL;
	Eina_List *l;

	EINA_LIST_FOREACH(viewdata->genlist_items, l, *item) {
		if (!(*item))
			continue;

		kv = elm_object_item_data_get(*item);
		if (!kv)
			continue;

		if (controller_check_same_string(kv->key, (char *)key))
			return true;
	}

	return false;
}

static void __delete_win_request_cb(void *data,
									Evas_Object *obj,
									void *event_info)
{
	ui_app_exit();
}

static void __layout_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = data;
	/* Let window go to hide state. */
	elm_win_lower(vd->win);
}

static void __get_app_resource(const char *edj_file_in,
								char *edj_path_out,
								int edj_path_max)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, edj_path_max, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

static void __preference_type_selected_cb(void *data,
										Evas_Object *obj,
										void *event_info)
{

	if(elm_radio_selected_object_get(obj) != obj)
		return;

	preftype_s *preftype = (preftype_s *)data;

	elm_object_part_text_set(viewdata->pref_edit_panel_type_selector,
								"default",
								preftype->caption);

	model_set_preference_type_selected(preftype->preftype);
}

static void __remove_all_click_cb(void *data, Evas_Object *obj, void *event_info)
{
	if (!model_remove_preferences())
		return;

	elm_genlist_clear(viewdata->pref_list_panel_list);
}

static void __update_click_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = (viewdata_s *)data;

	char *key = strdup(elm_entry_entry_get(vd->pref_edit_panel_key_entry));
	if (!key || strlen(key) == 0) {
		controller_log(DLOG_WARN, "Key name field is empty");
		free(key);
		return;
	}

	eina_str_tolower(&key);

	const char *value = elm_entry_entry_get(vd->pref_edit_panel_value_entry);
	if (!value || strlen(value) == 0) {
		free(key);
		controller_log(DLOG_WARN, "Value field is empty");
		return;
	}

	pref_value_type_t type = model_get_preference_type_selected();

	controller_add_property_items_enum(key, type);

	if (controller_set_preference(key, value, type)) {
		controller_log(DLOG_INFO,
						"Preference '%s' value set successfully",
						key);

		void *key_value = NULL;

		if (controller_get_preference(key, type, &key_value)) {
			key_value_t *kv = (key_value_t *)malloc(sizeof(key_value_t));
			kv->key = strdup(key);
			kv->value = key_value;
			kv->value_type = type;

			view_update_genlist_item(kv);
		}
	} else {
		controller_log(DLOG_WARN, "Preference '%s' value set failed", key);
	}

	free(key);
}

static void __genlist_item_image_clicked_cb(void *data,
											Evas_Object *obj,
											void *event_info)
{
	key_value_t *key_val = (key_value_t *)data;

	if (controller_remove_preference(key_val->key)) {
		controller_log(DLOG_INFO, "Preference removed: %s", key_val->key);

		Elm_Object_Item *item = NULL;

		if (__find_genlist_item(key_val->key, &item))
			elm_object_item_del(item);
	}
}

static Evas_Object *__create_window(void)
{
	Evas_Object *win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	if (!win) {
		controller_log(DLOG_ERROR,
						"Function elm_win_util_standard_add() failed");
		return NULL;
	}

	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_autodel_set(win, EINA_TRUE);
	elm_win_indicator_mode_set(win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(win, ELM_WIN_INDICATOR_OPAQUE);

	evas_object_smart_callback_add(win,
									"delete,request",
									__delete_win_request_cb,
									NULL);

	return win;
}

static Evas_Object *__create_conformant(Evas_Object *parent)
{
	Evas_Object *conform = elm_conformant_add(parent);
	if (!conform) {
		controller_log(DLOG_ERROR, "Function elm_conformant_add() failed");
		return NULL;
	}

	evas_object_size_hint_weight_set(conform,
										EVAS_HINT_EXPAND,
										EVAS_HINT_EXPAND);
	elm_win_resize_object_add(parent, conform);
	evas_object_show(conform);

	return conform;
}

static Evas_Object *__create_layout(Evas_Object *parent,
									const char *edj_file_name,
									const char *edj_group_name,
									const char *target_part_name)
{
	char edj_path[PATH_MAX] = {0, };

	__get_app_resource(edj_file_name, edj_path, (int)PATH_MAX);
	Evas_Object *layout = elm_layout_add(parent);
	if (!layout) {
		controller_log(DLOG_ERROR, "Function elm_layout_add() failed");
		return NULL;
	}

	elm_layout_file_set(layout, edj_path, edj_group_name);

	if (target_part_name) {
		elm_object_part_content_set(parent, target_part_name, layout);
	} else {
		elm_object_content_set(parent, layout);
	}

	return layout;
}

static Evas_Object *__create_entry(Evas_Object *parent,
									const char *target_part_name)
{
	Evas_Object *entry = elm_entry_add(parent);
	if (!entry) {
		controller_log(DLOG_ERROR, "elm_entry_add() failed.");
		return NULL;
	}

	elm_entry_line_wrap_set(entry, ELM_WRAP_NONE);
	elm_entry_single_line_set(entry, EINA_TRUE);
	evas_object_size_hint_weight_set(entry, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(entry, EVAS_HINT_FILL, 0.5);
	elm_object_part_content_set(parent, target_part_name, entry);
	evas_object_show(entry);

	return entry;
}

static Evas_Object *__create_radio_group(Evas_Object *parent,
									const char *target_part_name)
{
	Evas_Object *group = NULL;
	Evas_Object *radio = NULL;
	Evas_Object *box = elm_box_add(parent);

	if(!box)
		return NULL;

	elm_box_horizontal_set(box, EINA_FALSE);
	evas_object_show(box);

	for (int i = 0; i < PREF_MAX; i++) {
		if(!group) {
			group = radio = elm_radio_add(box);
		} else {
			radio = elm_radio_add(group);
			elm_radio_group_add(radio, group);
		}
		if(!radio)
			continue;

		char buff[256] = {0,};
		snprintf(buff, sizeof(buff), "<font color=#e1e1e1>%s</font>", preftypes[i].caption);
		elm_object_text_set(radio, buff);

		evas_object_smart_callback_add(radio, "changed", __preference_type_selected_cb, (void *)&preftypes[i]);
		elm_radio_state_value_set(radio, i);
		evas_object_size_hint_align_set(radio, EVAS_HINT_FILL, EVAS_HINT_FILL);
		elm_box_pack_end(box, radio);
		evas_object_show(radio);
	}

	elm_object_part_content_set(parent, target_part_name, box);

	return box;
}

static Evas_Object *__create_button(Evas_Object *parent,
									const char *target_part_name,
									const char *caption,
									Evas_Smart_Cb on_click_cb,
									void *data)
{
	Evas_Object *button = elm_button_add(parent);
	if (!button) {
		controller_log(DLOG_ERROR, "elm_button_add() failed.");
		return NULL;
	}

	elm_object_part_content_set(parent, target_part_name, button);
	elm_object_text_set(button, caption);

	evas_object_smart_callback_add(button, "clicked", on_click_cb, data);

	evas_object_show(button);

	return button;
}

static Evas_Object *__create_genlist(Evas_Object *parent,
									const char *target_part_name,
									void *data)
{
	Evas_Object *list = elm_genlist_add(parent);
	if (!list) {
		controller_log(DLOG_ERROR, "elm_genlist_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(list, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(parent, target_part_name, list);
	evas_object_show(list);

	return list;
}

static bool __create_main_layout(viewdata_s *vd)
{
	vd->win = __create_window();
	if (!vd->win)
		return false;

	vd->conform = __create_conformant(vd->win);
	if (!vd->conform) {
		evas_object_del(vd->win);
		return false;
	}

	vd->main_layout = __create_layout(vd->conform,
									EDJ_FILE_NAME_MAIN,
									GROUP_MAIN,
									NULL);
	if (!vd->main_layout) {
		evas_object_del(vd->win);
		return false;
	}

	return true;
}

static bool __create_pref_edit_panel_layout(viewdata_s *vd)
{
	vd->pref_edit_panel_layout = __create_layout(vd->main_layout,
												EDJ_FILE_NAME_PREF_EDIT_PANEL,
												GROUP_PREF_EDIT_PANEL,
												PART_MAIN_PREF_EDIT_PANEL);
	if (!vd->pref_edit_panel_layout) {
		evas_object_del(vd->win);
		return false;
	}

	vd->pref_edit_panel_key_entry = __create_entry(vd->pref_edit_panel_layout,
										PART_PREF_EDIT_PANEL_KEY_PANEL_ENTRY);
	if (!vd->pref_edit_panel_key_entry)
		return false;

	vd->pref_edit_panel_value_entry = __create_entry(vd->pref_edit_panel_layout,
										PART_PREF_EDIT_PANEL_VALUE_PANEL_ENTRY);
	if (!vd->pref_edit_panel_value_entry)
		return false;

	vd->pref_edit_panel_type_selector = __create_radio_group(
										vd->pref_edit_panel_layout,
										PART_PREF_EDIT_PANEL_TYPE_PANEL_ENTRY);
	if (!vd->pref_edit_panel_type_selector)
		return false;

	return true;
}

static bool __create_pref_buttons_panel_layout(viewdata_s *vd)
{
	vd->pref_buttons_panel_layout = __create_layout(vd->main_layout,
											EDJ_FILE_NAME_PREF_BUTTONS_PANEL,
											GROUP_PREF_BUTTONS_PANEL,
											PART_MAIN_PREF_BUTTONS_PANEL);
	if (!vd->pref_buttons_panel_layout) {
		evas_object_del(vd->win);
		return false;
	}

	vd->pref_buttons_panel_left_button = __create_button(
											vd->pref_buttons_panel_layout,
											PART_PREF_BUTTONS_PANEL_LEFT_BUTTON,
											REMOVE_ALL_PREFS_CAPTION,
											__remove_all_click_cb,
											NULL);
	if (!vd->pref_buttons_panel_left_button) {
		evas_object_del(vd->win);
		return false;
	}

	vd->pref_buttons_panel_right_button = __create_button(
										vd->pref_buttons_panel_layout,
										PART_PREF_BUTTONS_PANEL_RIGHT_BUTTON,
										UPDATE_PREF_CAPTION,
										__update_click_cb,
										(void *)vd);
	if (!vd->pref_buttons_panel_right_button) {
		evas_object_del(vd->win);
		return false;
	}

	return true;
}

static bool __create_pref_list_panel_layout(viewdata_s *vd)
{
	vd->pref_list_panel_list = __create_genlist(vd->main_layout,
												PART_MAIN_PREF_LIST_PANEL,
												(void *)vd);
	if (!vd->pref_list_panel_list) {
		evas_object_del(vd->win);
		return false;
	}

	return true;
}

bool view_create_base_gui(viewdata_s *vd)
{
	viewdata = vd;

	if (!__create_main_layout(vd))
		return false;

	if (!__create_pref_edit_panel_layout(vd))
		return false;

	if (!__create_pref_buttons_panel_layout(vd))
		return false;

	if (!__create_pref_list_panel_layout(vd))
		return false;

	eext_object_event_callback_add(vd->main_layout,
									EEXT_CALLBACK_BACK,
									__layout_back_cb,
									vd);

	evas_object_show(vd->win);

	return true;
}


static char *__get_genlist_item_label(void *data,
									Evas_Object *obj,
									const char *part)
{
	key_value_t *key_val = (key_value_t *)data;

	char buff_main[256] = {0,};
	char *buff_sub = NULL;

	switch (key_val->value_type) {
	case PREF_INTEGER:
		snprintf(buff_main,
					sizeof(buff_main),
					"%s = %d",
					key_val->key,
					*((int *)key_val->value));
		buff_sub = "type: INTEGER";
		break;
	case PREF_DOUBLE:
		snprintf(buff_main,
					sizeof(buff_main),
					"%s = %.3f",
					key_val->key,
					*((double *)key_val->value));
		buff_sub = "type: DOUBLE";
		break;
	case PREF_BOOL:
		if (*((bool *)key_val->value)) {
			snprintf(buff_main, sizeof(buff_main), "%s = TRUE", key_val->key);
		} else {
			snprintf(buff_main, sizeof(buff_main), "%s = FALSE", key_val->key);
		}
		buff_sub = "type: BOOLEAN";
		break;
	case PREF_STRING:
		snprintf(buff_main,
					sizeof(buff_main),
					"%s = %s",
					key_val->key,
					(char *)key_val->value);
		buff_sub = "type: STRING";
		break;
	default:
		snprintf(buff_main,
					sizeof(buff_main),
					"%s = ???",
					key_val->key);
		buff_sub = "type: UNKNOWN";
	}

	if (controller_check_same_string((char *)part, "elm.text")) {
		return strdup(buff_main);
	} else if (controller_check_same_string((char *)part, "elm.text.sub")) {
		return strdup(buff_sub);
	}

	return NULL;
}

static Evas_Object *__get_genlist_item_content(void *data,
												Evas_Object *obj,
												const char *part)
{
	if (!strcmp(part, "elm.swallow.icon")) {
		Evas_Object *image = elm_image_add(obj);
		if (!image) {
			controller_log(DLOG_ERROR, "Function elm_image_add() failed.");
			return NULL;
		}

		char *file_path = __get_image_file_path(ICON_DELETE_FILE_NAME);
		if (!elm_image_file_set(image, file_path, NULL)) {
			free(file_path);
			evas_object_del(image);
			controller_log(DLOG_ERROR, "Function elm_image_file_set() failed.");
			return NULL;
		}

		if (file_path)
			free(file_path);

		evas_object_size_hint_min_set(image,
										DELETE_IMAGE_SIZE_MIN,
										DELETE_IMAGE_SIZE_MIN);
		evas_object_size_hint_weight_set(image,
										EVAS_HINT_EXPAND,
										EVAS_HINT_EXPAND);

		evas_object_smart_callback_add(image,
										"clicked",
										__genlist_item_image_clicked_cb,
										data);

		return image;
	}

	return NULL;
}

static void __del_genlist_item(void *data, Evas_Object *obj)
{
	key_value_t *key_val = (key_value_t *)data;

	controller_remove_key_value(key_val);
}

bool view_add_genlist_item(key_value_t *key_value)
{
	if (!viewdata->pref_list_panel_list) {
		controller_log(DLOG_ERROR,
						"Function property_items_enum_all_cb() failed.");
		return false;
	}

	Elm_Genlist_Item_Class *itc = elm_genlist_item_class_new();
	if (!itc) {
		controller_log(DLOG_ERROR,
						"Function elm_genlist_item_class_new() failed.");
		return false;
	}

	itc->item_style = "double_label";
	itc->func.text_get = __get_genlist_item_label;
	itc->func.content_get = __get_genlist_item_content;
	itc->func.state_get = NULL;
	itc->func.del = __del_genlist_item;

	Elm_Object_Item *item = elm_genlist_item_append(
								viewdata->pref_list_panel_list,
								itc,
								(void *)key_value,
								NULL,
								ELM_GENLIST_ITEM_NONE,
								NULL,
								NULL);
	if (!item) {
		controller_log(DLOG_ERROR,
						"Function elm_genlist_item_append() failed.");
		return false;
	}

	viewdata->genlist_items = eina_list_append(viewdata->genlist_items, item);

	return true;
}

bool view_update_genlist_item(key_value_t *key_value)
{
	Elm_Object_Item *item = NULL;

	if (__find_genlist_item(key_value->key, &item)) {
		key_value_t *item_key_value = elm_object_item_data_get(item);

		if (!item_key_value)
			return false;

		controller_remove_key_value(item_key_value);
		elm_object_item_data_set(item, (void *)key_value);
		elm_genlist_item_update(item);

		return true;
	}

	return view_add_genlist_item(key_value);
}
