/*
 * Samsung API
 * Copyright (c) 2009-2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <tizen.h>

#include "main.h"
#include "log.h"

void get_resource_path(appdata_s *ad)
{
	static char *res_path_buff = NULL;

	if (res_path_buff == NULL) {
		res_path_buff = app_get_resource_path();
	}

	snprintf(ad->puzzle_edj_path, MAX_PATH_LEN, "%s%s", res_path_buff, PUZZLE_EDJ);
	snprintf(ad->smile_img_path, MAX_PATH_LEN, "%s%s", res_path_buff, SMILE_IMG);
	snprintf(ad->correct_img_path, MAX_PATH_LEN, "%s%s", res_path_buff, CORRECT_IMG);

	_D("Puzzle edj path: %s, Smile img path: %s", ad->puzzle_edj_path, ad->smile_img_path);

	free(res_path_buff);
}

static void win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	ui_app_exit();
}

static void layout_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	ret_if(!data);

	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_lower(ad->win);
}

static Evas_Object *_win_create(appdata_s *ad)
{
	Evas_Object *win = NULL;

	retv_if(!ad, NULL);

	win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_autodel_set(win, EINA_TRUE);

	/*if (elm_win_wm_rotation_supported_get(win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(win, (const int *)(&rots), 4);
	}*/

	evas_object_smart_callback_add(win, "delete, request", win_delete_request_cb, NULL);

	elm_win_screen_size_get(win, NULL, NULL, &ad->screen_width, &ad->screen_height);
	_D("Screen width: %d, height: %d", ad->screen_width, ad->screen_height);

	evas_object_show(win);

	return win;
}

static Evas_Object *_conformant_create(appdata_s *ad)
{
	Evas_Object *conform = NULL;

	retv_if(!ad, NULL);

	conform = elm_conformant_add(ad->win);
	retv_if(!conform, NULL);

	elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_HIDE);
	elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(ad->win, conform);
	evas_object_show(conform);

	return conform;
}

static Evas_Object *_layout_create(appdata_s *ad)
{
	_D("Layout create");
	Evas_Object *layout = NULL;
	Evas_Object *menu = NULL;
	Evas_Object *dir_btn = NULL;

	retv_if(!ad, NULL);

	layout = elm_layout_add(ad->win);
	retv_if(!layout, NULL);

	get_resource_path(ad);

	elm_layout_file_set(layout, ad->puzzle_edj_path, GRP_MAIN);
	elm_object_part_text_set(layout, "txt_title", "Hello EFL");
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	eext_object_event_callback_add(layout, EEXT_CALLBACK_BACK, layout_back_cb, ad);

	elm_object_content_set(ad->conform, layout);
	ad->layout = layout;

	ad->e = evas_object_evas_get(layout);
	retv_if(!ad->e, NULL);

	menu = menu_create(ad, layout);
	retv_if(!menu, NULL);

	dir_btn = dir_btn_create(ad, layout);
	retv_if(!dir_btn, NULL);

	puzzle_create(ad);

	return layout;
}

static bool app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
	   Initialize UI resources and application's data
	   If this function returns true, the main loop of application starts
	   If this function returns false, the application is terminated */
	appdata_s *ad = data;

	/* Window */
	ad->win = _win_create(ad);
	retv_if(!ad->win, false);

	/* Conformant */
	ad->conform = _conformant_create(ad);
	retv_if(!ad->conform, false);

	/* Base Layout */
	_layout_create(ad);
	retv_if(!ad->layout, false);

	return true;
}

static void app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void app_terminate(void *data)
{
	/* Release all resources. */
}

static void ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

static void ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_DEVICE_ORIENTATION_CHANGED*/
	return;
}

static void ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_REGION_FORMAT_CHANGED*/
}

static void ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_BATTERY*/
}

static void ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}

int main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_orient_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_changed, &ad);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() is failed. err = %d", ret);
	}

	return ret;
}
