#include <Elementary.h>

#include "index.h"
#include "page_scroller.h"
#include "$(appName).h"
#include "conf.h"

static int current_page = -1;
static int current_region = -1;

static Evas_Object *__create_box(Evas_Object *parent);
static Evas_Object *__create_indice(Evas_Object *parent);
static Evas_Object *__get_indice(Evas_Object *index, int c_page);
static Eina_Bool __init_position_set_cb(void *data);
static void __page_area_changed_cb(void *data, Evas_Object *obj,
	void *event_info);
static void __page_current_changed_start_cb(void *data, Evas_Object *obj,
	void *event_info);
static void __page_current_changed_stop_cb(void *data, Evas_Object *obj,
	void *event_info);
static void __rotate_indice(Evas_Object *box, int n, double angle);


Evas_Object *
index_create(Evas_Object *page_scroller, cb_data_t *init_data)
{
	Evas_Object *box = NULL;
	int page_count = -1;
	int i = 0;

	if (!page_scroller) {
		dlog_print(DLOG_ERROR, LOG_TAG, "invalid parameter : page_scroller = NULL");
		return NULL;
	}

	page_count = page_scroller_get_page_count(page_scroller);
	if (page_count < 0) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to obtain pages count");
		return NULL;
	}

	box = __create_box(page_scroller);
	if (!box) {
		dlog_print(DLOG_ERROR, LOG_TAG, "box creation failed");
		return NULL;
	}

	for (i = 0 ; i < page_count; i++)
		elm_box_pack_end(box, __create_indice(box));

	init_data = (cb_data_t *) malloc(sizeof(cb_data_t));
	if (!init_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to allocate init data");
		evas_object_del(box);
		return NULL;
	}

	init_data->box = box;
	init_data->scroller = page_scroller;

	ecore_timer_add(0.2, __init_position_set_cb, init_data);

	evas_object_smart_callback_add(page_scroller, "scroll",
		__page_area_changed_cb, init_data);
	evas_object_smart_callback_add(page_scroller, "scroll,drag,start",
		__page_current_changed_start_cb, init_data);
	evas_object_smart_callback_add(page_scroller, "scroll,anim,stop",
		__page_current_changed_stop_cb, init_data);

	return box;
}

static Evas_Object *
__create_box(Evas_Object *parent)
{
	Evas_Object *box = NULL;

	if (!parent) {
		dlog_print(DLOG_ERROR, LOG_TAG, "invalid parameter : parent is NULL");
		return NULL;
	}

	box = elm_box_add(parent);
	if (!box) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create elm_box widget");
		return NULL;
	}

	evas_object_size_hint_weight_set(box, EVAS_HINT_EXPAND,
		EVAS_HINT_EXPAND);

	elm_box_horizontal_set(box, EINA_TRUE);
	elm_box_homogeneous_set(box, EINA_TRUE);
	elm_box_align_set(box, 0.5, 0.5);

	evas_object_show(box);
	return box;
}

static Evas_Object *
__create_indice(Evas_Object *parent)
{
	Evas_Object *indice = NULL;
	char edje_path[PATH_MAX] = {0, };

	if (!parent) {
		dlog_print(DLOG_ERROR, LOG_TAG, "invalid parameter : parent is NULL");
		return NULL;
	}

	indice = elm_layout_add(parent);
	if (!indice) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create indice layout");
		return NULL;
	}

	get_app_resource(EDJ_FILE, edje_path, PATH_MAX);

	if (!elm_layout_file_set(indice, edje_path, GROUP_INDICE)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to set edje file");
		evas_object_del(indice);
		return NULL;
	}

	evas_object_show(indice);
	return indice;
}

static Evas_Object *
__get_indice(Evas_Object *index, int c_page)
{
	Eina_List *box_content = NULL;

	if (!index || c_page < 0) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Invalid parameter : index is NULL | c_page < 0");
		return NULL;
	}

	box_content = elm_box_children_get(index);
	if (!box_content) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to get index content");
		return NULL;
	}

	return (Evas_Object *) eina_list_nth(box_content, c_page);
}

static Eina_Bool
__init_position_set_cb(void *data)
{
	cb_data_t *init_data = (cb_data_t *) data;
	Evas_Object *page_scroller = NULL;
	Evas_Object *index = NULL;

	int c_page = -1;
	int i = 0;

	if (!init_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "invalid cb data!");
		return EINA_FALSE;
	}

	page_scroller = init_data->scroller;
	index = init_data->box;

	if (!page_scroller || !index) {
		dlog_print(DLOG_ERROR, LOG_TAG, "invalid parameter : page_scroller | index = NULL");
		return EINA_FALSE;
	}

	c_page = page_scroller_get_current_page_number(page_scroller);

	for (i = 0; i < SCROLLER_PAGES; i++)
		if (i == c_page)
			__rotate_indice(index, i, CUR_ANGLE);
		else
			__rotate_indice(index, i, DEF_ANGLE);

	return EINA_FALSE;
}

static void
__page_area_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	int x = 0;
	int w = 0;
	double angle = 0.0;
	int next_page = -1;
	cb_data_t *cb_data = (cb_data_t *) data;

	if (!cb_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "callback data is null");
		return;
	}

	elm_scroller_region_get(obj, &x, NULL, &w, NULL);

	angle = (double) (x - current_region)/w * 90.0;
	if (abs(x - current_region) <= w) {
		next_page = x > current_region ? (current_page + 1)
			% SCROLLER_PAGES : current_page - 1;
	} else {
		if (x > current_region) {
			next_page = current_page - 1;
		} else {
			next_page = 0;
		}
	}

	if (next_page < 0)
		next_page = SCROLLER_MAX_IDX;

	__rotate_indice(cb_data->box, current_page, 90 + angle);
	__rotate_indice(cb_data->box, next_page, 360 + angle);
}

static void
__page_current_changed_start_cb(void *data, Evas_Object *obj, void *event_info)
{
	current_page = page_scroller_get_current_page_number(obj);
	elm_scroller_region_get(obj, &current_region, NULL, NULL, NULL);
}

static void
__page_current_changed_stop_cb(void *data, Evas_Object *obj, void *event_info)
{
	current_page = -1;
	current_region = -1;

	__init_position_set_cb(data);
}


static void
__rotate_indice(Evas_Object *box, int n, double angle)
{
	Evas_Object *indice = NULL;
	Evas_Map *indice_map = NULL;
	int x = -1;
	int y = -1;
	int w = -1;
	int h = -1;

	if (!box || n < 0 || angle < 0) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Invalid params");
		return;
	}

	indice = __get_indice(box, n);
	if (!indice) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get indice");
		return;
	}

	evas_object_geometry_get(indice, &x, &y, &w, &h);
	indice_map = evas_map_new(TRANSFORMATION_POINTS_CNT);

	evas_map_util_points_populate_from_object(indice_map, indice);
	evas_map_util_rotate(indice_map, angle, x + w / 2, y + w / 2);
	evas_object_map_set(indice, indice_map);
	evas_object_map_enable_set(indice, EINA_TRUE);
}
