/*
 * Copyright (c) 2014-2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#include "main-app.h"
#include "utils/logger.h"
#include "view/sensor-list-view.h"
#include "view/window.h"

#include <app.h>
#include <Elementary.h>
#include <efl_extension.h>

struct _app_data {
	window *win;
	Evas_Object *navi;
};

static Evas_Object *_app_navi_add(app_data *app);
static bool _app_create_cb(void *user_data);
static void _app_terminate_cb(void *user_data);
static void _app_navi_back_cb(void *data, Evas_Object *obj, void *event_info);

app_data *app_create()
{
	return calloc(1, sizeof(app_data));
}

void app_destroy(app_data *app)
{
	free(app);
}

int app_run(app_data *app, int argc, char **argv)
{
	RETVM_IF(!app, -1, "app is NULL");
	ui_app_lifecycle_callback_s cbs = {
		.create = _app_create_cb,
		.terminate = _app_terminate_cb,
		NULL
	};

	return ui_app_main(argc, argv, &cbs, app);
}

static Evas_Object *_app_navi_add(app_data *app)
{
	Evas_Object *navi = elm_naviframe_add(window_layout_get(app->win));
	RETVM_IF(!navi, NULL, "elm_naviframe_add() failed");
	elm_naviframe_prev_btn_auto_pushed_set(navi, EINA_FALSE);
	eext_object_event_callback_add(navi, EEXT_CALLBACK_BACK, _app_navi_back_cb, app);
	window_content_set(app->win, navi);
	return navi;
}

static bool _app_create_cb(void *user_data)
{
	RETVM_IF(!user_data, false, "user_data is NULL");
	app_data *app = user_data;

	app->win = window_create();
	RETVM_IF(!app->win, false, "window_create() failed");

	app->navi = _app_navi_add(app);
	if (!app->navi) {
		ERR("_app_navi_add() failed");
		window_destroy(app->win);
		return false;
	}

	Evas_Object *view = sensor_list_view_create(app->navi);
	if (!view) {
		ERR("sensor_list_view_create() failed");
		window_destroy(app->win);
		return false;
	}

	return true;
}

static void _app_terminate_cb(void *user_data)
{
	RETM_IF(!user_data, "user_data is NULL");
	app_data *app = user_data;

	if (app->win) {
		window_destroy(app->win);
	}
}

static void _app_navi_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is NULL");
	app_data *app = data;
	if (elm_naviframe_top_item_get(app->navi) == elm_naviframe_bottom_item_get(app->navi)) {
		window_lower(app->win);
	} else {
		elm_naviframe_item_pop(app->navi);
	}
}
