/*
 * Copyright (c) 2014-2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#include "model/sensor-list.h"
#include "model/sensor-magnetic.h"
#include "utils/logger.h"

static const char *axes[] = { "X", "Y", "Z" };
static const char *tilts[] = { "Yaw", "Pitch", "Roll" };
static const char *level = "Level";
static const char *distance = "Distance";

static sensor_info sensors[] = {
	{
		.type = SENSOR_ACCELEROMETER,
		.units = SENSOR_UNIT_METRE_PER_SECOND_SQUARED,
		.name = "Acceleration",
		.value_names = axes,
		.value_count = 3,
		.axes = { -1, 1, 1 }
	},
	{
		.type = SENSOR_LIGHT,
		.units = SENSOR_UNIT_LUX,
		.name = "Ambient Light",
		.value_names = &level,
		.value_count = 1
	},
	{
		.type = SENSOR_ORIENTATION,
		.units = SENSOR_UNIT_DEGREE,
		.name = "Device Orientation",
		.value_names = tilts,
		.value_count = 3
	},
	{
		.type = SENSOR_GRAVITY,
		.units = SENSOR_UNIT_METRE_PER_SECOND_SQUARED,
		.name = "Gravity",
		.value_names = axes,
		.value_count = 3,
		.axes = { -1, 1, 1 }
	},
	{
		.type = SENSOR_GYROSCOPE,
		.units = SENSOR_UNIT_DEGREE_PER_SECOND,
		.name = "Gyro",
		.value_names = axes,
		.value_count = 3
	},
	{
		.type = SENSOR_MAGNETIC,
		.units = SENSOR_UNIT_MICRO_TESLA,
		.name = "Magnetic",
		.value_names = axes,
		.value_count = 3,
		.extra_values = &sensor_magnetic_strength_value,
		.extra_value_count = 1,
		.axes = { 1, -1, 1 }
	},
	{
		.type = SENSOR_PROXIMITY,
		.units = SENSOR_UNIT_CENTIMETER,
		.name = "Proximity",
		.value_names = &distance,
		.value_count = 1
	},
	{
		.type = SENSOR_LINEAR_ACCELERATION,
		.units = SENSOR_UNIT_METRE_PER_SECOND_SQUARED,
		.name = "User Acceleration",
		.value_names = axes,
		.value_count = 3,
		.axes = { -1, 1, 1 }
	}
};

static unsigned sensor_count = sizeof(sensors) / sizeof(sensors[0]);

static const char *_sensor_units_str_get(sensor_unit_e units);
static const char *_sensor_value_format_get(float resolution);

void sensor_list_init()
{
	sensor_info *info = sensors;
	sensor_info *end = info + sensor_count;
	for (; info != end; ++info) {
		float resolution = 1.0;

		sensor_get_default_sensor(info->type, &info->sensor);
		sensor_get_min_range(info->sensor, &info->value_min);
		sensor_get_max_range(info->sensor, &info->value_max);
		sensor_get_resolution(info->sensor, &resolution);

		info->units_str = _sensor_units_str_get(info->units);
		info->value_range = info->value_max - info->value_min;
		info->value_format = _sensor_value_format_get(resolution);
	}
}

const sensor_info *sensor_list_get(unsigned *count)
{
	if (count) {
		*count = sensor_count;
	}

	return sensors;
}

static const char *_sensor_units_str_get(sensor_unit_e units)
{
	static const char *units_str[] = {
		"m/s²", /* SENSOR_UNIT_METRE_PER_SECOND_SQUARED */
		"μT",   /* SENSOR_UNIT_MICRO_TESLA */
		"°",    /* SENSOR_UNIT_DEGREE */
		"lx",   /* SENSOR_UNIT_LUX */
		"cm",   /* SENSOR_UNIT_CENTIMETER */
		"°/s"   /* SENSOR_UNIT_DEGREE_PER_SECOND */
	};

	if (units <= (sizeof(units_str) / sizeof(units_str[0]))) {
		return units_str[units];
	}

	return NULL;
}

static const char *_sensor_value_format_get(float resolution)
{
	if (resolution >= 1.0) {
		return "%.0f";
	} else if (resolution >= 0.1) {
		return "%.1f";
	} else {
		return "%.2f";
	}
}
