/*
 * Copyright (c) 2014-2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#include "view/sensor-angle-chart.h"
#include "sensor-data-chart-private.h"

#include <math.h>
#include "utils/color-utils.h"
#include "utils/logger.h"

static const int circle_padding = 35;

static void _angle_chart_draw_cb(data_chart *chart, float value_range,
		const int *axes, const float *values, const int *colors, unsigned value_count);
static void _angle_chart_draw_segment(data_chart *chart, double x, double y,
		double radius, float value, int color);

Evas_Object *sensor_angle_chart_create(Evas_Object *parent)
{
	Evas_Object *image = sensor_data_chart_create(parent);
	data_chart *chart = evas_object_data_get(image, chart_data_key);
	if (!chart) {
		ERR("chart is NULL");
		evas_object_del(image);
		return NULL;
	}

	chart->draw_cb = _angle_chart_draw_cb;
	return image;
}

static void _angle_chart_draw_cb(data_chart *chart, float value_range,
		const int *axes, const float *values, const int *colors, unsigned value_count)
{
	int circle_diameter = (chart->width - circle_padding * (value_count + 1)) / (value_count);

	/* distance between sibling circles centers */
	int circle_distance = circle_diameter + circle_padding;

	/* distance between first and last circle centers */
	int width = (value_count - 1) * circle_distance;

	double x = (chart->width - width) / 2;
	double y = chart->height / 2;

	const int *color = colors;
	const float *value = values;
	const float *end = value + value_count;
	for (; value != end; ++value, ++color) {
		_angle_chart_draw_segment(chart, x, y, circle_diameter / 2, *value, *color);
		x += circle_distance;
	}
}

static void _angle_chart_draw_segment(data_chart *chart, double x, double y,
		double radius, float value, int color)
{
	cairo_set_source_rgb(chart->cairo, FRGB(color));
	cairo_set_line_width(chart->cairo, 1.0);
	cairo_move_to(chart->cairo, x, y);
	cairo_line_to(chart->cairo, x + radius, y);

	double angle = value * (M_PI / 180.0);
	if (angle > 0) {
		cairo_arc(chart->cairo, x, y, radius, 0.0, angle);
	} else {
		cairo_arc_negative(chart->cairo, x, y, radius, 0.0, angle);
	}

	cairo_line_to(chart->cairo, x, y);
	cairo_fill(chart->cairo);
}
