/*
 * Copyright (c) 2014-2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#include "sensor-data-chart-private.h"
#include "utils/logger.h"

const char *chart_data_key = "chart_data";

static void _data_chart_destroy_cb(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void _data_chart_resize_cb(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void _data_chart_clear(data_chart *chart);
static void _data_chart_flush(data_chart *chart);

Evas_Object *sensor_data_chart_create(Evas_Object *parent)
{
	data_chart *chart = calloc(1, sizeof(data_chart));
	RETVM_IF(!chart, NULL, "calloc() failed");

	Evas *evas = evas_object_evas_get(parent);
	chart->image = evas_object_image_add(evas);
	if (!chart->image) {
		ERR("evas_object_image_add() failed");
		free(chart);
		return NULL;
	}

	evas_object_image_colorspace_set(chart->image, EVAS_COLORSPACE_ARGB8888);
	evas_object_data_set(chart->image, chart_data_key, chart);
	evas_object_event_callback_add(chart->image, EVAS_CALLBACK_FREE, _data_chart_destroy_cb, chart);
	evas_object_event_callback_add(chart->image, EVAS_CALLBACK_RESIZE, _data_chart_resize_cb, chart);

	return chart->image;
}

static void _data_chart_destroy_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is NULL");
	data_chart *chart = data;
	cairo_destroy(chart->cairo);
	cairo_surface_destroy(chart->surface);
	free(chart);
}

static void _data_chart_resize_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is NULL");
	data_chart *chart = data;

	evas_object_geometry_get(chart->image, NULL, NULL, &chart->width, &chart->height);
	RETM_IF(0 == chart->width, "width is 0");
	RETM_IF(0 == chart->height, "height is 0");

	evas_object_image_size_set(chart->image, chart->width, chart->height);
	evas_object_image_fill_set(chart->image, 0, 0, chart->width, chart->height);

	if (chart->surface) {
		cairo_destroy(chart->cairo);
		cairo_surface_destroy(chart->surface);
	}

	chart->surface = cairo_image_surface_create(CAIRO_FORMAT_ARGB32,
			chart->width, chart->height);
	chart->cairo = cairo_create(chart->surface);

	_data_chart_clear(chart);
	_data_chart_flush(chart);
}

static void _data_chart_clear(data_chart *chart)
{
	cairo_set_source_rgb(chart->cairo, 1.0, 1.0, 1.0);
	cairo_paint(chart->cairo);
}

static void _data_chart_flush(data_chart *chart)
{
	evas_object_image_data_set(chart->image,
			cairo_image_surface_get_data(chart->surface));
	evas_object_image_data_update_add(chart->image, 0, 0, chart->width, chart->height);
}

void sensor_data_chart_update(Evas_Object *image, float value_range,
		const int *axes, const float *values, const int *colors, unsigned count)
{
	RETM_IF(!values, "values is NULL");
	RETM_IF(!colors, "colors is NULL");

	data_chart *chart = evas_object_data_get(image, chart_data_key);
	RETM_IF(!chart, "chart is NULL");
	RETM_IF(!chart->cairo, "cairo is NULL");

	if (chart->draw_cb) {
		_data_chart_clear(chart);
		chart->draw_cb(chart, value_range, axes, values, colors, count);
		_data_chart_flush(chart);
	}
}
