/*
 * Copyright (c) 2014-2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#include "view/sensor-data-view.h"

#include "utils/color-utils.h"
#include "utils/logger.h"
#include "view/sensor-angle-chart.h"
#include "view/sensor-vector-chart.h"

#include <Elementary.h>
#include <stdbool.h>

#define GROUP_TITLE_LEN 64
#define ITEM_TEXT_LEN 32
#define SENSOR_INTERVAL 100

#define STYLE_ITEM_TITLE "groupindex"
#define STYLE_ITEM_VALUE "type1"

#define PART_TITLE "elm.text.main"
#define PART_LABEL "elm.text"
#define PART_VALUE "elm.text.end"
#define PART_ICON  "elm.swallow.icon"

static const int data_view_item_colors[] = {
	0xffa050,
	0x50a0ff,
	0xa0ff50
};

typedef struct {
	Elm_Object_Item *obj_item;
	int color;
	const char *label;
	float value;
} data_view_value_item;

typedef struct {
	Elm_Object_Item *obj_item;
	float value;
	sensor_extra_value *value_info;
} data_view_extra_item;

typedef struct {
	Evas_Object *navi;
	Elm_Object_Item *navi_item;

	Evas_Object *box;
	Evas_Object *chart;
	Evas_Object *genlist;

	data_view_value_item *value_items;
	data_view_extra_item *extra_items;

	const sensor_info *sensor_info;
	sensor_listener_h sensor_listener;
} data_view;

static void _data_view_destroy_cb(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void _data_view_sensor_start(data_view *view);

static Evas_Object *_data_view_chart_create(data_view *view);
static Evas_Object *_data_view_genlist_create(data_view *view);

static void _data_view_value_items_create(data_view *view);
static void _data_view_extra_items_create(data_view *view);

static Elm_Object_Item *_data_view_gl_item_append(Evas_Object *genlist,
		Elm_Genlist_Item_Class *itc, void *item_data);

static void _data_view_gl_items_append(data_view *view);
static void _data_view_gl_title_append(data_view *view);
static void _data_view_gl_value_items_append(data_view *view);
static void _data_view_gl_extra_items_append(data_view *view);

static char *_data_view_gl_title_get(void *data, Evas_Object *obj, const char *part);
static void _data_view_gl_title_del(void *data, Evas_Object *obj);
static char *_data_view_gl_text_get(void *data, Evas_Object *obj, const char *part);
static char *_data_view_gl_extra_text_get(void *data, Evas_Object *obj, const char *part);
static Evas_Object *_data_view_gl_content_get(void *data, Evas_Object *obj, const char *part);

static void _data_view_value_items_update(data_view *view, float *values);
static void _data_view_extra_items_update(data_view *view, float *values);

static void _data_view_transition_finished_cb(void *data, Evas_Object *obj, void *event_info);
static void _data_view_sensor_cb(sensor_h sensor, sensor_event_s *sensor_data, void *user_data);

static bool _data_view_is_correct_sensor_values(sensor_event_s *sensor_data, const sensor_info *view_sensor_info);

Evas_Object *sensor_data_view_create(Evas_Object *navi, const sensor_info *sensor_info)
{
	data_view *view = calloc(1, sizeof(data_view));
	RETVM_IF(!view, NULL, "calloc() failed");
	view->navi = navi;
	view->sensor_info = sensor_info;

	view->box = elm_box_add(view->navi);
	if (!view->box) {
		ERR("elm_box_add() failed");
		free(view);
		return NULL;
	}

	evas_object_event_callback_add(view->box, EVAS_CALLBACK_FREE, _data_view_destroy_cb, view);

	view->chart = _data_view_chart_create(view);
	view->genlist = _data_view_genlist_create(view);
	if (!view->genlist) {
		evas_object_del(view->box);
		return NULL;
	}

	if (view->sensor_info->extra_values) {
		_data_view_extra_items_create(view);
	}

	_data_view_value_items_create(view);
	_data_view_gl_items_append(view);

	view->navi_item = elm_naviframe_item_push(view->navi,
			view->sensor_info->name, NULL, NULL, view->box, NULL);
	evas_object_smart_callback_add(view->navi, "transition,finished",
			_data_view_transition_finished_cb, view);

	return view->box;
}

static void _data_view_destroy_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is NULL");
	data_view *view = data;

	sensor_listener_stop(view->sensor_listener);
	sensor_destroy_listener(view->sensor_listener);

	free(view->extra_items);
	free(view->value_items);
	free(view);
}

static void _data_view_sensor_start(data_view *view)
{
	sensor_error_e err = SENSOR_ERROR_NONE;
	err = sensor_create_listener(view->sensor_info->sensor, &view->sensor_listener);
	RETM_IF(err != SENSOR_ERROR_NONE, "sensor_create_listener() failed(%d)", err);
	sensor_listener_set_event_cb(view->sensor_listener, SENSOR_INTERVAL, _data_view_sensor_cb, view);
	sensor_listener_start(view->sensor_listener);
}

static Evas_Object *_data_view_chart_create(data_view *view)
{
	Evas_Object *chart = NULL;
	if (view->sensor_info->value_count > 1) {
		if (view->sensor_info->units == SENSOR_UNIT_DEGREE
		 || view->sensor_info->units == SENSOR_UNIT_DEGREE_PER_SECOND) {
			chart = sensor_angle_chart_create(view->box);
			RETVM_IF(!chart, NULL, "sensor_angle_chart_create() failed");
		} else {
			chart = sensor_vector_chart_create(view->box);
			RETVM_IF(!chart, NULL, "sensor_vector_chart_create() failed");
		}
	}

	if (chart) {
		evas_object_size_hint_weight_set(chart, EVAS_HINT_FILL, 0.5);
		evas_object_size_hint_align_set(chart, EVAS_HINT_FILL, EVAS_HINT_FILL);
		elm_box_pack_end(view->box, chart);
		evas_object_show(chart);
	}

	return chart;
}

static Evas_Object *_data_view_genlist_create(data_view *view)
{
	Evas_Object *genlist = elm_genlist_add(view->box);
	RETVM_IF(!genlist, NULL, "elm_genlist_add() failed");

	evas_object_data_set(genlist, "view_data", view);
	evas_object_size_hint_weight_set(genlist, EVAS_HINT_FILL, 0.5);
	evas_object_size_hint_align_set(genlist, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_box_pack_end(view->box, genlist);
	evas_object_show(genlist);

	return genlist;
}

static void _data_view_value_items_create(data_view *view)
{
	view->value_items = calloc(view->sensor_info->value_count,
			sizeof(data_view_value_item));
	RETM_IF(!view->value_items, "calloc() failed");

	const int *color = data_view_item_colors;
	const char **name = view->sensor_info->value_names;

	data_view_value_item *item = view->value_items;
	data_view_value_item *end = item + view->sensor_info->value_count;
	for (; item != end; ++item, ++name, ++color) {
		item->label = *name;
		item->color = *color;
	}
}

static void _data_view_extra_items_create(data_view *view)
{
	view->extra_items = calloc(view->sensor_info->extra_value_count,
			sizeof(data_view_extra_item));
	RETM_IF(!view->extra_items, "calloc() failed");

	sensor_extra_value *value = view->sensor_info->extra_values;
	data_view_extra_item *item = view->extra_items;
	data_view_extra_item *end = item + view->sensor_info->extra_value_count;
	for (; item != end; ++item, ++value) {
		item->value_info = value;
	}
}

static Elm_Object_Item *_data_view_gl_item_append(Evas_Object *genlist,
        Elm_Genlist_Item_Class *itc, void *item_data)
{
	Elm_Object_Item *obj_item = elm_genlist_item_append(genlist, itc,
			item_data, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);
	elm_genlist_item_select_mode_set(obj_item,
			ELM_OBJECT_SELECT_MODE_DISPLAY_ONLY);
	return obj_item;
}

static void _data_view_gl_items_append(data_view *view)
{
	_data_view_gl_title_append(view);
	_data_view_gl_extra_items_append(view);
	_data_view_gl_value_items_append(view);
}

static void _data_view_gl_title_append(data_view *view)
{
	Elm_Genlist_Item_Class *title_itc = elm_genlist_item_class_new();
	title_itc->item_style = STYLE_ITEM_TITLE;
	title_itc->func.text_get = _data_view_gl_title_get;
	title_itc->func.del = _data_view_gl_title_del;

	char title[GROUP_TITLE_LEN] = { '\0' };
	char format[GROUP_TITLE_LEN] = { '\0' };
	snprintf(format, sizeof(format), "Value Range: %s%s ~ %s%s",
			view->sensor_info->value_format, view->sensor_info->units_str,
			view->sensor_info->value_format, view->sensor_info->units_str);

	snprintf(title, sizeof(title), format, view->sensor_info->value_min,
			view->sensor_info->value_max);

	_data_view_gl_item_append(view->genlist, title_itc, (void *) strdup(title));

	elm_genlist_item_class_free(title_itc);
}

static void _data_view_gl_value_items_append(data_view *view)
{
	Elm_Genlist_Item_Class *value_itc = elm_genlist_item_class_new();
	value_itc->item_style = STYLE_ITEM_VALUE;
	value_itc->func.text_get = _data_view_gl_text_get;

	if (view->sensor_info->value_count > 1) {
		value_itc->func.content_get = _data_view_gl_content_get;
	}

	data_view_value_item *item = view->value_items;
	data_view_value_item *end = item + view->sensor_info->value_count;

	for (; item != end; ++item) {
		item->obj_item = _data_view_gl_item_append(view->genlist, value_itc, item);
	}

	elm_genlist_item_class_free(value_itc);
}

static void _data_view_gl_extra_items_append(data_view *view)
{
	Elm_Genlist_Item_Class *extra_itc = elm_genlist_item_class_new();
	extra_itc->item_style = STYLE_ITEM_VALUE;
	extra_itc->func.text_get = _data_view_gl_extra_text_get;

	data_view_extra_item *item = view->extra_items;
	data_view_extra_item *end = item + view->sensor_info->extra_value_count;

	for (; item != end; ++item) {
		item->obj_item = _data_view_gl_item_append(view->genlist, extra_itc, item);
	}

	elm_genlist_item_class_free(extra_itc);
}

static Evas_Object *_data_view_gl_content_get(void *data, Evas_Object *obj, const char *part)
{
	data_view_value_item *item = data;
	RETVM_IF(!item, NULL, "item is NULL");

	if (strcmp(part, PART_ICON) == 0) {
		Evas_Object *rect = evas_object_rectangle_add(evas_object_evas_get(obj));
		evas_object_color_set(rect, RGB(item->color), 255);

		Evas_Object *layout = elm_layout_add(obj);
		elm_layout_theme_set(layout, "layout", "list/A/left.icon", "default");
		elm_layout_content_set(layout, "elm.swallow.content", rect);

		return layout;
	}

	return NULL;
}

static char *_data_view_gl_title_get(void *data, Evas_Object *obj, const char *part)
{
	if (strcmp(part, PART_TITLE) == 0) {
		return strdup(data);
	} else {
		return NULL;
	}
}

static void _data_view_gl_title_del(void *data, Evas_Object *obj)
{
	free(data);
}

static char *_data_view_gl_text_get(void *data, Evas_Object *obj, const char *part)
{
	data_view_value_item *item = data;
	RETVM_IF(!item, NULL, "item is NULL");

	if (strcmp(part, PART_LABEL) == 0) {
		RETVM_IF(!item->label, NULL, "label is NULL");
		return strdup(item->label);
	} else if (strcmp(part, PART_VALUE) == 0) {
		data_view *view = evas_object_data_get(obj, "view_data");
		RETVM_IF(!view, NULL, "view is NULL");

		char buffer[ITEM_TEXT_LEN] = { '\0' };
		snprintf(buffer, sizeof(buffer), view->sensor_info->value_format, item->value);
		return strdup(buffer);
	} else {
		return NULL;
	}
}

static char *_data_view_gl_extra_text_get(void *data, Evas_Object *obj, const char *part)
{
	data_view_extra_item *item = data;
	RETVM_IF(!item, NULL, "item is NULL");

	if (strcmp(part, PART_LABEL) == 0) {
		RETVM_IF(!item->value_info->name, NULL, "name is NULL");
		return strdup(item->value_info->name);
	} else if (strcmp(part, PART_VALUE) == 0) {
		char buffer[ITEM_TEXT_LEN] = { '\0' };
		snprintf(buffer, sizeof(buffer), item->value_info->format, item->value);
		return strdup(buffer);
	} else {
		return NULL;
	}
}

static void _data_view_value_items_update(data_view *view, float *values)
{
	data_view_value_item *item = view->value_items;
	data_view_value_item *end = item + view->sensor_info->value_count;
	RETM_IF(!item, "item is NULL");

	float *value = values;
	bool update_chart = false;

	for (; item != end; ++item, ++value) {
		if (item->value != *value) {
			update_chart = true;
			item->value = *value;
			elm_genlist_item_fields_update(item->obj_item,
					PART_VALUE, ELM_GENLIST_ITEM_FIELD_TEXT);
		}
	}

	if (view->chart && update_chart) {
		sensor_data_chart_update(view->chart, view->sensor_info->value_range,
				view->sensor_info->axes, values, data_view_item_colors, view->sensor_info->value_count);
	}
}

static void _data_view_extra_items_update(data_view *view, float *values)
{
	data_view_extra_item *item = view->extra_items;
	data_view_extra_item *end = item + view->sensor_info->extra_value_count;

	for (; item != end; ++item) {
		if (item->value_info->value_get) {
			item->value = item->value_info->value_get(view->sensor_info, values);
			elm_genlist_item_fields_update(item->obj_item,
					PART_VALUE, ELM_GENLIST_ITEM_FIELD_TEXT);
		}
	}
}

static void _data_view_transition_finished_cb(void *data, Evas_Object *obj, void *event_info)
{
	data_view *view = data;
	RETM_IF(!view, "view is NULL");

	_data_view_sensor_start(view);
	evas_object_smart_callback_del(view->navi, "transition,finished",
			_data_view_transition_finished_cb);
}

static void _data_view_sensor_cb(sensor_h sensor, sensor_event_s *sensor_data, void *user_data)
{
	data_view *view = user_data;
	RETM_IF(!view, "view is NULL");
	int value_count = sensor_data->value_count;
	RETM_IF(!sensor_data, "sensor_data is NULL");

	RETM_IF(value_count != view->sensor_info->value_count,
			"value count is %d, but expected %d",
			value_count, view->sensor_info->value_count);
	RETM_IF(!_data_view_is_correct_sensor_values(sensor_data, view->sensor_info), "wrong sensor value");

	_data_view_value_items_update(view, sensor_data->values);
	_data_view_extra_items_update(view, sensor_data->values);
}

static bool _data_view_is_correct_sensor_values(sensor_event_s *sensor_data, const sensor_info *view_sensor_info)
{
	int i = 0;
	for (; i < sensor_data->value_count; ++i) {
		float value = sensor_data->values[i];

		if (isnan(value) || (value < view_sensor_info->value_min)
						|| (value > view_sensor_info->value_max)) {
			return false;
		}
	}

	return true;
}
