/*
 * Copyright (c) 2014-2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#include "view/sensor-list-view.h"

#include "model/sensor-list.h"
#include "utils/logger.h"
#include "view/sensor-data-view.h"

#include <Elementary.h>
#include <stdbool.h>

typedef struct {
	Evas_Object *navi;
	Elm_Object_Item *navi_item;
	Evas_Object *list;
} list_view;

static void _list_view_destroy_cb(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void _list_view_fill(list_view *view);
static void _list_view_sel_cb(void *data, Evas_Object *obj, void *event_info);

Evas_Object *sensor_list_view_create(Evas_Object *navi)
{
	list_view *view = calloc(1, sizeof(list_view));
	RETVM_IF(!view, NULL, "calloc() failed");

	view->navi = navi;
	view->list = elm_list_add(view->navi);
	if (!view->list) {
		ERR("elm_list_add() failed");
		free(view);
		return NULL;
	}

	evas_object_data_set(view->list, "view_data", view);
	evas_object_event_callback_add(view->list, EVAS_CALLBACK_FREE,
			_list_view_destroy_cb, view);

	sensor_list_init();
	_list_view_fill(view);

	view->navi_item = elm_naviframe_item_push(view->navi, "Sensor App", NULL, NULL,
			view->list, NULL);

	return view->list;
}

static void _list_view_destroy_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	free(data);
}

static void _list_view_fill(list_view *view)
{
	unsigned count = 0;
	const sensor_info *item = sensor_list_get(&count);
	const sensor_info *end = item + count;

	RETM_IF(!item, "item is NULL");
	for (; item != end; ++item) {
		bool is_supported = false;
		sensor_is_supported(item->type, &is_supported);
		if (is_supported) {
			elm_list_item_append(view->list, item->name, NULL, NULL,
					_list_view_sel_cb, item);
		}
	}
}

static void _list_view_sel_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is NULL");
	RETM_IF(!event_info, "event_info is NULL");
	sensor_info *item = data;
	list_view *view = evas_object_data_get(obj, "view_data");

	elm_list_item_selected_set(event_info, EINA_FALSE);
	sensor_data_view_create(view->navi, item);
}
