/*
 * Copyright (c) 2014-2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#include "view/sensor-vector-chart.h"
#include "sensor-data-chart-private.h"

#include <app.h>
#include <math.h>
#include <stdbool.h>
#include <sys/param.h>

#include "utils/color-utils.h"
#include "utils/logger.h"

static const int circle_min_radius = 25;
static const int circle_line_width = 5;
static const int circle_sign_text_size = 40;

static const int arrow_line_width = 10;
static const int arrow_tip_width = 40;
static const int arrow_tip_height = 15;

static void _vector_chart_draw_cb(data_chart *chart, float value_range,
		const int *axes, const float *values, const int *colors, unsigned value_count);
static void _vector_chart_draw_arrow(data_chart *chart, bool horizontal,
		float value, int color);
static void _vector_chart_draw_circle(data_chart *chart,
		float value, int color);

Evas_Object *sensor_vector_chart_create(Evas_Object *parent)
{
	Evas_Object *image = sensor_data_chart_create(parent);
	data_chart *chart = evas_object_data_get(image, chart_data_key);
	if (!chart) {
		ERR("chart is NULL");
		evas_object_del(image);
		return NULL;
	}

	chart->draw_cb = _vector_chart_draw_cb;
	return image;
}

static void _vector_chart_draw_cb(data_chart *chart, float value_range,
		const int *axes, const float *values, const int *colors, unsigned value_count)
{
	RETM_IF(value_count < 3, "value_count < 3, expected 3-d vector");

	float scale = 1.0;
	if (value_range) {
		scale = MIN(chart->width, chart->height) / value_range;
	}

	/*
	 *  Values should be adjusted as follows depending on device orientation:
	 *   0:   X  Y
	 *  90:   X -Y
	 *  180: -X -Y
	 *  270: -X  Y
	 *
	 *  In landscape orientations X should be vertical and Y horizontal.
	 *  Currently only portrait orientation supported.
	 */

	/* draw X arrow */
	_vector_chart_draw_arrow(chart, true, axes[0] * values[0] * scale, colors[0]);
	/* draw Y arrow */
	_vector_chart_draw_arrow(chart, false, axes[1] * values[1] * scale, colors[1]);
	/* draw Z circle */
	_vector_chart_draw_circle(chart, axes[2] * values[2] * scale, colors[2]);
}

static void _vector_chart_draw_arrow(data_chart *chart, bool horizontal,
		float value, int color)
{
	double x0 = chart->width / 2;
	double y0 = chart->height / 2;

	/* [0] and [1] - arrow start and end points coordinates,
	 * [2], [3], [4] - arrow tip triangle points coordinates */
	double x[5] = { x0, x0, x0, x0, x0 };
	double y[5] = { y0, y0, y0, y0, y0 };

	/* major axis (x for horizontal arrow, y for vertical) */
	double *a = x;

	/* minor axis */
	double *b = y;

	if (!horizontal) {
		a = y;
		b = x;
	}

	int sign = value < 0 ? -1 : 1;
	/* offset arrow start point */
	a[0] += sign * circle_min_radius;
	/* calculate arrow end point */
	a[1] = a[0] + value;
	/* calculate arrow tip point */
	a[2] = a[1] + sign * arrow_tip_height;
	/* calculate other 2 triangle points */
	a[3] = a[4] = a[1];
	b[3] += arrow_tip_width / 2;
	b[4] -= arrow_tip_width / 2;

	/* draw arrow line */
	cairo_set_line_width(chart->cairo, arrow_line_width);
	cairo_set_source_rgb(chart->cairo, FRGB(color));
	cairo_move_to(chart->cairo, x[0], y[0]);
	cairo_line_to(chart->cairo, x[1], y[1]);
	cairo_stroke(chart->cairo);

	/* draw arrow tip triangle */
	cairo_set_line_width(chart->cairo, 1.0);
	cairo_move_to(chart->cairo, x[2], y[2]);
	cairo_line_to(chart->cairo, x[3], y[3]);
	cairo_line_to(chart->cairo, x[4], y[4]);
	cairo_line_to(chart->cairo, x[2], y[2]);
	cairo_fill(chart->cairo);
}

static void _vector_chart_draw_circle(data_chart *chart,
		float value, int color)
{
	/* circle center coordinates */
	double x = chart->width / 2;
	double y = chart->height / 2;

	/* draw value sign */
	cairo_move_to(chart->cairo, x - circle_min_radius / 2,
								y + circle_min_radius / 2);
	cairo_set_source_rgb(chart->cairo, 0.0, 0.0, 0.0);
	cairo_set_font_size(chart->cairo, circle_sign_text_size);
	cairo_show_text(chart->cairo, value < 0 ? "-" : "+");
	cairo_stroke(chart->cairo);

	/* draw circle */
	double radius = fabs(value) + circle_min_radius;
	cairo_set_line_width(chart->cairo, circle_line_width);
	cairo_set_source_rgb(chart->cairo, FRGB(color));
	cairo_arc(chart->cairo, x, y, radius, 0.0, 2 * M_PI);
	cairo_stroke(chart->cairo);
}
