/*
 * Samsung API
 * Copyright (c) 2009-2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */



#include <tizen.h>
#include <time.h>

#include "log.h"
#include "main.h"
#include "thread.h"
#include "stopwatch.h"
#include "view.h"

#define PRIVATE_DATA_KEY_STOPWATCH_BOX1 "b_1"
#define PRIVATE_DATA_KEY_STOPWATCH_BOX2 "b_2"



static void win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	appdata_s *ad = NULL;

	ret_if(!data);

	ad = data;

	if (ad->vd) free(ad->vd);

	ui_app_exit();
}



static void win_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = NULL;

	ret_if(!data);

	_D("Let stopwatch go to hide");

	ad = data;

	/* Let window go to hide state. */
	elm_win_lower(ad->win);
}



static Evas_Object *_create_win(appdata_s *ad)
{
	Evas_Object *win = NULL;

	retv_if(!ad, NULL);

	win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	retv_if(!win, NULL);

	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_autodel_set(win, EINA_TRUE);

	elm_app_base_scale_set(1.7);

	evas_object_smart_callback_add(win, "delete,request", win_delete_request_cb, ad);
	eext_object_event_callback_add(win, EEXT_CALLBACK_BACK, win_back_cb, ad);

	return win;
}



static void _destroy_window(Evas_Object *win)
{
	ret_if(!win);

	evas_object_del(win);
}



static Evas_Object *_create_conform(appdata_s *ad)
{
	Evas_Object *conform = NULL;

	retv_if(!ad, NULL);

	conform = elm_conformant_add(ad->win);
	retv_if(!conform, NULL);

	elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(ad->win, conform);
	evas_object_show(conform);

	return conform;
}



static void _destroy_conform(Evas_Object *conform)
{
	ret_if(!conform);

	evas_object_del(conform);
}



static void _set_indicator_bg(appdata_s *ad)
{
	Evas_Object *bg = NULL;

	ret_if(!ad);

	bg = elm_bg_add(ad->conform);
	ret_if(!bg);

	elm_object_style_set(bg, "indicator/headerbg");
	elm_object_part_content_set(ad->conform, "elm.swallow.indicator_bg", bg);
	evas_object_show(bg);
}



static Evas_Object *_create_navi(appdata_s *ad)
{
	Evas_Object *nf = NULL;

	retv_if(!ad, NULL);

	nf = elm_naviframe_add(ad->conform);
	retv_if(!nf, NULL);

	elm_object_content_set(ad->conform, nf);
	evas_object_show(nf);

	return nf;
}



static void _destroy_navi(Evas_Object *nf)
{
	ret_if(!nf);

	evas_object_del(nf);
}



static Evas_Object *_create_layout(appdata_s *ad)
{
	Evas_Object *layout = NULL;

	retv_if(!ad, NULL);

	layout = elm_layout_add(ad->nf);
	retv_if(!layout, NULL);

	elm_layout_file_set(layout, LAYOUTEDJ, "main");
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	eext_object_event_callback_add(layout, EEXT_CALLBACK_BACK, win_back_cb, ad);

	return layout;
}



static void _destroy_layout(Evas_Object *layout)
{
	ret_if(!layout);

	evas_object_del(layout);
}



static void _create_stopwatch(appdata_s *ad)
{
	Elm_Object_Item *nf_it = NULL;
	Evas_Object *box1 = NULL;
	Evas_Object *box2 = NULL;

	ret_if(!ad);

	_D("Create stopwatch");

	/* Window */
	ad->win = _create_win(ad);
	ret_if(!ad->win);

	/* Conformant */
	ad->conform = _create_conform(ad);
	goto_if(!ad->conform, ERROR);

	/* Indicator BG */
	_set_indicator_bg(ad);

	/* Naviframe */
	ad->nf = _create_navi(ad);
	goto_if(!ad->nf, ERROR);

	/* Layout */
	ad->layout = _create_layout(ad);
	goto_if(!ad->layout, ERROR);

	/* three part in layout */
	box1 = view_create_stopwatch_display(ad);
	goto_if(!box1, ERROR);
	elm_object_part_content_set(ad->layout, "part_one", box1);
	evas_object_data_set(ad->layout, PRIVATE_DATA_KEY_STOPWATCH_BOX1, box1);

	box2 = view_create_stopwatch_button(ad);
	goto_if(!box2, ERROR);
	elm_object_part_content_set(ad->layout, "part_two", box2);
	evas_object_data_set(ad->layout, PRIVATE_DATA_KEY_STOPWATCH_BOX2, box2);

	ad->list = view_create_stopwatch_list(ad);
	goto_if(!ad->list, ERROR);
	elm_object_part_content_set(ad->layout, "part_three", ad->list);

	evas_object_show(ad->layout);

	/* insert layout to naviframe */
	nf_it = elm_naviframe_item_push(ad->nf, "STOPWATCH", NULL, NULL, ad->layout, NULL);
	goto_if(!nf_it, ERROR);

	/* Show window after base gui is set up */
	evas_object_show(ad->win);

	return;

ERROR:
	if (ad->list) {
		ad->list = elm_object_part_content_unset(ad->layout, "part_three");
		evas_object_del(ad->list);
	}
	if (box2) {
		box2 = elm_object_part_content_unset(ad->layout, "part_two");
		evas_object_del(box2);
	}
	if (box1) {
		box1 = elm_object_part_content_unset(ad->layout, "part_one");
		evas_object_del(box1);
	}
	if (ad->layout) _destroy_layout(ad->layout);
	if (ad->nf) _destroy_navi(ad->nf);
	if (ad->conform) _destroy_conform(ad->conform);
	if (ad->win) _destroy_window(ad->win);

	return;
}



static void _destroy_btn(Evas_Object *btn)
{
	ret_if(!btn);

	evas_object_del(btn);
}



static void _destroy_stopwatch(appdata_s *ad)
{
	Evas_Object *box1 = NULL;
	Evas_Object *box2 = NULL;

	ret_if(!ad);

	_D("Destroy stopwatch");

	if (ad->vd) {
		if (ad->vd->time) evas_object_del(ad->vd->time);
		if (ad->vd->msec) evas_object_del(ad->vd->msec);
		if (ad->vd->thread) ecore_thread_cancel(ad->vd->thread);

		free(ad->vd);
	}

	if (ad->btn_lap) _destroy_btn(ad->btn_lap);
	if (ad->btn_start) _destroy_btn(ad->btn_start);

	if (ad->nf) {
		ad->layout = elm_naviframe_item_pop(ad->nf);
		if (ad->layout) {
			if (ad->list) {
				ad->list = elm_object_part_content_unset(ad->layout, "part_three");
				evas_object_del(ad->list);
			}

			box2 = evas_object_data_del(ad->layout, PRIVATE_DATA_KEY_STOPWATCH_BOX2);
			if (box2) {
				box2 = elm_object_part_content_unset(ad->layout, "part_two");
				evas_object_del(box2);
			}

			box1 = evas_object_data_del(ad->layout, PRIVATE_DATA_KEY_STOPWATCH_BOX1);
			if (box1) {
				box1 = elm_object_part_content_unset(ad->layout, "part_one");
				evas_object_del(box1);
			}

			if (ad->layout) _destroy_layout(ad->layout);
		}
	}

	if (ad->nf) _destroy_navi(ad->nf);
	if (ad->conform) _destroy_conform(ad->conform);
	if (ad->win) _destroy_window(ad->win);
}



static bool app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
	   Initialize UI resources and application's data
	   If this function returns true, the main loop of application starts
	   If this function returns false, the application is terminated */
	appdata_s *ad = data;

	_create_stopwatch(ad);

	return true;
}



static void app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}



static void app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}



static void app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}



static void app_terminate(void *data)
{
	/* Release all resources. */
	appdata_s *ad = NULL;

	ret_if(!data);

	ad = data;

	_destroy_stopwatch(ad);
}



static void ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;

	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	ret_if(!locale);

	elm_language_set(locale);
	free(locale);

	return;
}



static void ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_DEVICE_ORIENTATION_CHANGED*/
	return;
}



static void ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_REGION_FORMAT_CHANGED*/
}



static void ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_BATTERY*/
}



static void ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}



int main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_orient_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_changed, &ad);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		_E("ui_app_main() is failed. err = %d", ret);
	}

	return ret;
}
/* End of file */
