#include <package_manager.h>
#include <app_common.h>
#include "$(appName).h"
#include "application_mgr.h"
#include "item.h"
#include "defines.h"

static s_appmgr_data appmgr_data = {
	.app_info_filter = NULL,
	.apps_list = NULL,
	.taskmgr_id = NULL
};

static int __compare_strings(char *app_id1, char *app_id2);
static int __compare_items_cb(const void *data1, const void *data2);
static bool __app_info_cb(app_info_h app_info, void *user_data);
static void __remove_app_from_list(char *app_id);
static void __clear_application_list(void);

bool
application_mgr_init(void)
{
	if (app_info_filter_create(&appmgr_data.app_info_filter) != APP_MANAGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create app info filter");
		return false;
	}

	if (app_info_filter_add_bool(appmgr_data.app_info_filter, PACKAGE_INFO_PROP_APP_NODISPLAY, false) != APP_MANAGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to set nodisplay property for filter");
		return false;
	}

	if (app_get_id(&appmgr_data.taskmgr_id) != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get taskmgr id");
		return false;
	}

	return true;
}

bool
application_mgr_destroy(void)
{
	if (app_info_filter_destroy(appmgr_data.app_info_filter) != APP_MANAGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to destroy filter handle");
		return false;
	}

	appmgr_data.app_info_filter = NULL;

	__clear_application_list();

	return true;
}

Eina_List *
application_mgr_get_running_apps(void)
{
	if (appmgr_data.apps_list)
		__clear_application_list();

	if (!appmgr_data.app_info_filter) {
		dlog_print(DLOG_ERROR, LOG_TAG, "App info filter not initialized");
		return NULL;
	}

	if (app_info_filter_foreach_appinfo(appmgr_data.app_info_filter, __app_info_cb, NULL) != APP_MANAGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to execute callback for each filtered application");
		return NULL;
	}

	appmgr_data.apps_list = eina_list_sort(appmgr_data.apps_list, 0, __compare_items_cb);

	return appmgr_data.apps_list;
}

bool
application_mgr_terminate_app(char *app_id)
{
	app_context_h app_context = NULL;

	if (!app_id) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Invalid parameter");
		return false;
	}

	if (app_manager_get_app_context(app_id, &app_context) != APP_MANAGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get app context");
		return false;
	}

	if (app_manager_request_terminate_bg_app(app_context) != APP_MANAGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to terminate app");
		return false;
	}

	if (app_context_destroy(app_context) != APP_MANAGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to destroy context handle");
		return false;
	}

	__remove_app_from_list(app_id);

	return true;
}

bool
application_mgr_resume_app(char *app_id)
{
	app_context_h app_context = NULL;

	if (!app_id) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Invalid parameter");
		return false;
	}

	if (app_manager_get_app_context(app_id, &app_context) != APP_MANAGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get app context");
		return false;
	}

	if (app_manager_resume_app(app_context) != APP_MANAGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to resume app");
		return false;
	}

	if (app_context_destroy(app_context) != APP_MANAGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to destroy context handle");
		return false;
	}

	return true;
}

static int
__compare_strings(char *app_id1, char *app_id2)
{
	int ret = strncmp(app_id1, app_id2, ITEM_APP_ID_STR_MAX_LENGTH);

	if (strlen(app_id1) == strlen(app_id2) && !ret) {
		return 0;
	} else {
		return ret > 0 ? 1 : -1;
	}
}

static int
__compare_items_cb(const void *data1, const void *data2)
{
	app_item_t *item1 = NULL;
	app_item_t *item2 = NULL;

	if (!data1 || !data2) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Invalid parameter");
		return 0;
	}

	item1 = (app_item_t *)data1;
	item2 = (app_item_t *)data2;

	return __compare_strings(item1->app_id, item2->app_id);
}

static bool
__app_info_cb(app_info_h app_info, void *user_data)
{
	char *app_id = NULL;
	char *app_icon = NULL;
	bool running = false;
	app_item_t *new_item = NULL;

	if (app_info_get_app_id(app_info, &app_id) != APP_MANAGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get app id");
		return false;
	}

	if (app_info_get_icon(app_info, &app_icon) != APP_MANAGER_ERROR_NONE) {
		free(app_id);
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get app icon");
		return false;
	}

	if (app_manager_is_running(app_id, &running) != APP_MANAGER_ERROR_NONE) {
		free(app_id);
		free(app_icon);
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get running state");
		return false;
	}

	if (running && __compare_strings(app_id, appmgr_data.taskmgr_id)) {
		new_item = item_create(app_id, app_icon);
		if (!new_item) {
			free(app_id);
			free(app_icon);
			dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create new item");
			return false;
		}

		appmgr_data.apps_list = eina_list_append(appmgr_data.apps_list, new_item);
	}

	free(app_id);
	free(app_icon);

	return true;
}

static void
__remove_app_from_list(char *app_id)
{
	Eina_List *it = NULL;
	app_item_t *item = NULL;

	EINA_LIST_FOREACH(appmgr_data.apps_list, it, item)
		if (!__compare_strings(item->app_id, app_id)) {
			appmgr_data.apps_list = eina_list_remove(appmgr_data.apps_list, item);
			free(item->app_id);
			free(item->icon_path);
			free(item);
			break;
		}
}

static void
__clear_application_list(void)
{
	Eina_List *it = NULL;
	app_item_t *item = NULL;

	EINA_LIST_FOREACH(appmgr_data.apps_list, it, item)
	{
		free(item->app_id);
		free(item->icon_path);
		free(item);
	}

	eina_list_free(appmgr_data.apps_list);
	appmgr_data.apps_list = NULL;
}
