#include "view.h"
#include "$(appName).h"
#include "application_mgr.h"
#include "item.h"
#include "defines.h"

static s_viewdata viewdata = {
	.win = NULL,
	.layout = NULL,
	.conform = NULL,
	.genlist = NULL,
	.item_class = NULL
 };

static void __delete_win_request_cb(void *data, Evas_Object *obj, void *event_info);
static void __layout_back_cb(void *data, Evas_Object *obj, void *event_info);
static void __get_app_resource(const char *edj_file_in, char *edj_path_out, int edj_path_max);
static Evas_Object *__create_icon(Evas_Object *parent, char *image_path);
static void __item_layout_mouse_up_cb(void *data, Evas *e, Evas_Object *obj, void *event_info);
static Evas_Object *__create_item_layout(Evas_Object *parent, char *app_id, char *icon_path);
static Evas_Object *__get_item_content(void *data, Evas_Object *obj, const char *part);
static Evas_Object *__create_genlist(void);
static void __delete_item(Elm_Object_Item *genlist_item);

void
view_display_no_applications(void)
{
	if (viewdata.layout)
		elm_layout_signal_emit(viewdata.layout, TASKMGR_PROGRAM_NO_APPS_SHOW_SIGNAL, TASKMGR_PROGRAM_NO_APPS_SHOW_SOURCE);

	if (viewdata.genlist)
		elm_genlist_clear(viewdata.genlist);
}

bool
view_display_application_list(Eina_List *app_list)
{
	Eina_List *it = NULL;
	app_item_t *item = NULL;

	if (!app_list) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Invalid argument");
		return false;
	}

	if (viewdata.layout)
		elm_layout_signal_emit(viewdata.layout, TASKMGR_PROGRAM_NO_APPS_HIDE_SIGNAL, TASKMGR_PROGRAM_NO_APPS_HIDE_SOURCE);

	if (elm_genlist_items_count(viewdata.genlist) != 0)
		elm_genlist_clear(viewdata.genlist);

	EINA_LIST_FOREACH(app_list, it, item)
		elm_genlist_item_append(viewdata.genlist, viewdata.item_class, (void *)item, NULL,
				ELM_GENLIST_ITEM_NONE, NULL, NULL);

	return true;
}

bool
view_create_base_gui(void)
{
	char edj_path[PATH_MAX] = {0, };

	/* Window */
	viewdata.win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	if (!viewdata.win) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create window");
		return false;
	}
	elm_win_conformant_set(viewdata.win, EINA_TRUE);
	elm_win_autodel_set(viewdata.win, EINA_TRUE);

	evas_object_smart_callback_add(viewdata.win, "delete,request", __delete_win_request_cb, NULL);

	/* Conformant */
	viewdata.conform = elm_conformant_add(viewdata.win);
	if (!viewdata.conform) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create conformant");
		evas_object_del(viewdata.win);
		return false;
	}

	elm_win_indicator_mode_set(viewdata.win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(viewdata.win, ELM_WIN_INDICATOR_OPAQUE);
	evas_object_size_hint_weight_set(viewdata.conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(viewdata.win, viewdata.conform);
	evas_object_show(viewdata.conform);

	/* Base Layout */
	__get_app_resource(MAIN_EDJ_FILE, edj_path, (int)PATH_MAX);
	viewdata.layout = elm_layout_add(viewdata.win);
	if (!viewdata.layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create layout");
		evas_object_del(viewdata.win);
		return false;
	}

	elm_layout_file_set(viewdata.layout, edj_path, TASKMGR_MAIN_GROUP);
	evas_object_size_hint_weight_set(viewdata.layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	eext_object_event_callback_add(viewdata.layout, EEXT_CALLBACK_BACK, __layout_back_cb, NULL);
	elm_object_content_set(viewdata.conform, viewdata.layout);

	/* Genlist */
	viewdata.genlist = __create_genlist();
	if (!viewdata.genlist) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create genlist");
		evas_object_del(viewdata.win);
		return false;
	}

	/* Show window after base gui is set up */
	evas_object_show(viewdata.win);

	return true;
}

static void
__delete_win_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

static void
__layout_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	/* Let window go to hide state. */
	elm_win_lower(viewdata.win);
}

static void
__get_app_resource(const char *edj_file_in, char *edj_path_out, int edj_path_max)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, edj_path_max, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

static Evas_Object *
__create_icon(Evas_Object *parent, char *image_path)
{
	Evas_Object *icon = elm_image_add(parent);
	if (!icon) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create an image");
		return NULL;
	}

	elm_image_file_set(icon, image_path, NULL);
	evas_object_show(icon);

	return icon;
}

static void
__item_layout_mouse_up_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	Evas_Object *edje = NULL;
	Elm_Object_Item *genlist_item = NULL;
	double dx = 0;
	Evas_Event_Mouse_Up *mouse_up_ev = (Evas_Event_Mouse_Up *)event_info;

	edje = elm_layout_edje_get(obj);
	edje_object_part_drag_value_get(edje, ITEM_PART_BASE, &dx, NULL);

	if ((int)dx == 0) {
		/* Item was not dragged left/right - item was clicked, resume application */
		genlist_item = elm_genlist_at_xy_item_get(viewdata.genlist, mouse_up_ev->canvas.x, mouse_up_ev->canvas.y, NULL);
		if (!genlist_item) {
				dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get item at %d, %d", mouse_up_ev->canvas.x, mouse_up_ev->canvas.y);
				return;
		}

		app_item_t *item = (app_item_t *)elm_object_item_data_get(genlist_item);
		if (!item) {
			dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get item data");
			return;
		}

		application_mgr_resume_app(item->app_id);

	} else if (dx < ITEM_DRAG_OUTSIDE_VAL && dx > -ITEM_DRAG_OUTSIDE_VAL) {
		/* Item was dragged, but not far enough to terminate application */
		elm_layout_signal_emit(obj, ITEM_PROGRAM_RESET_POSITION_SIGNAL, ITEM_PROGRAM_RESET_POSITION_SOURCE);
	} else {
		/* Item was dragged to left/right edge of the screen - terminate application */
		elm_layout_signal_emit(obj, ITEM_PROGRAM_HIDE_SIGNAL, ITEM_PROGRAM_HIDE_SOURCE);

		genlist_item = elm_genlist_at_xy_item_get(viewdata.genlist, mouse_up_ev->canvas.x, mouse_up_ev->canvas.y, NULL);
		if (!genlist_item) {
				dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get item at %d, %d", mouse_up_ev->canvas.x, mouse_up_ev->canvas.y);
				return;
		}

		__delete_item(genlist_item);

		if (elm_genlist_items_count(viewdata.genlist) == 0)
			view_display_no_applications();
	}
}

static Evas_Object *
__create_item_layout(Evas_Object *parent, char *app_id, char *icon_path)
{
	Evas_Object *icon = NULL;
	Evas_Object *layout = NULL;
	char edj_path[PATH_MAX] = {0, };

	__get_app_resource(ITEM_EDJ_FILE, edj_path, (int)PATH_MAX);

	layout = elm_layout_add(parent);
	if (!layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create layout");
		return NULL;
	}

	elm_layout_file_set(layout, edj_path, GENLIST_CUSTOM_ITEM_GROUP);
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(layout, EVAS_HINT_FILL, EVAS_HINT_FILL);

	evas_object_event_callback_add(layout, EVAS_CALLBACK_MOUSE_UP, __item_layout_mouse_up_cb, NULL);

	icon = __create_icon(parent, icon_path);
	elm_object_part_content_set(layout, ITEM_PART_CONTENT, icon);
	elm_object_part_text_set(layout, ITEM_PART_TEXT, app_id);

	evas_object_show(layout);

	return layout;
}

static Evas_Object *
__get_item_content(void *data, Evas_Object *obj, const char *part)
{
	if (!data)
		return NULL;

	app_item_t *item = (app_item_t *)data;

	return __create_item_layout(obj, item->app_id, item->icon_path);
}

static Evas_Object *
__create_genlist(void)
{
	Evas_Object *genlist = elm_genlist_add(viewdata.win);
	if (!genlist) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create genlist");
		return NULL;
	}

	viewdata.item_class = elm_genlist_item_class_new();
	if (!viewdata.item_class) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create item class");
		evas_object_del(genlist);
		return NULL;
	}

	viewdata.item_class->item_style = "full";
	viewdata.item_class->func.text_get = NULL;
	viewdata.item_class->func.content_get = __get_item_content;
	viewdata.item_class->func.state_get = NULL;
	viewdata.item_class->func.del = NULL;

	elm_genlist_mode_set(genlist, ELM_LIST_SCROLL);
	elm_genlist_select_mode_set(genlist, ELM_OBJECT_SELECT_MODE_NONE);

	elm_object_part_content_set(viewdata.layout, TASKMGR_PART_CONTENT, genlist);

	return genlist;
}

static void
__delete_item(Elm_Object_Item *genlist_item)
{
	app_item_t *item = (app_item_t *)elm_object_item_data_get(genlist_item);
	if (!item) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get item data");
		return;
	}

	application_mgr_terminate_app(item->app_id);

	elm_object_item_del(genlist_item);
}
