/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "main.h"
#include "user_callbacks.h"

static app_control_h app_control = NULL;

static bool get_sample_image_path(char *path, int size)
{
    char *shared_path = app_get_shared_resource_path();
    if (shared_path == NULL) {
        PRINT_MSG("Could not get shared resource path");
        return false;
    }

    const char *file_name = "sample.jpg";
    snprintf(path, size, "file://%s%s", shared_path, file_name);
    free(shared_path);

    return true;
}

static bool on_foreach_registered_alarm(int alarm_id, void *user_data)
{
    int flag;
    int ret = 0;
    struct tm date;
    time_t time_current;

    ret = alarm_get_scheduled_date(alarm_id, &date);
    if (ret != ALARM_ERROR_NONE) {
        dlog_print(DLOG_ERROR, LOG_TAG, "alarm_get_scheduled_date() error : %d", ret);
        PRINT_MSG("Get time Error : %d", ret);
    }

    // Logging scheduled alarm’s info
    time_current = mktime(&date);
    dlog_print(DLOG_DEBUG, LOG_TAG,
               "Registered alarm: %d on date: %s", alarm_id, ctime(&time_current));
    PRINT_MSG("Registered alarm: %d on date: %s", alarm_id, ctime(&time_current));

    ret = alarm_get_scheduled_recurrence_week_flag(alarm_id, &flag);
    if (ret == APP_CONTROL_ERROR_NONE) {
        if (flag & ALARM_WEEK_FLAG_SUNDAY) {
            dlog_print(DLOG_DEBUG, LOG_TAG, "Alarm recurrence on SUNDAY");
            PRINT_MSG("Alarm recurrence on SUNDAY");
        }

        if (flag & ALARM_WEEK_FLAG_MONDAY) {
            dlog_print(DLOG_DEBUG, LOG_TAG, "Alarm recurrence on MONDAY");
            PRINT_MSG("Alarm recurrence on MONDAY");
        }

        if (flag & ALARM_WEEK_FLAG_TUESDAY) {
            dlog_print(DLOG_DEBUG, LOG_TAG, "Alarm recurrence on TUESDAY");
            PRINT_MSG("Alarm recurrence on TUESDAY");
        }

        if (flag & ALARM_WEEK_FLAG_WEDNESDAY) {
            dlog_print(DLOG_DEBUG, LOG_TAG, "Alarm recurrence on WEDNESDAY");
            PRINT_MSG("Alarm recurrence on WEDNESDAY");
        }

        if (flag & ALARM_WEEK_FLAG_THURSDAY) {
            dlog_print(DLOG_DEBUG, LOG_TAG, "Alarm recurrence on THURSDAY");
            PRINT_MSG("Alarm recurrence on THURSDAY");
        }

        if (flag & ALARM_WEEK_FLAG_FRIDAY) {
            dlog_print(DLOG_DEBUG, LOG_TAG, "Alarm recurrence on FRIDAY");
            PRINT_MSG("Alarm recurrence on FRIDAY");
        }

        if (flag & ALARM_WEEK_FLAG_SATURDAY) {
            dlog_print(DLOG_DEBUG, LOG_TAG, "Alarm recurrence on SATURDAY");
            PRINT_MSG("Alarm recurrence on SATURDAY");
        }
    }

    return true;
}

static bool on_foreach_cancel_alarm(int alarm_id, void *user_data)
{
    // Cancel scheduled alarms

    int ret = alarm_cancel(alarm_id);
    if (ret != ALARM_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "alarm_cancel() failed, err = %d", ret);

    return true;
}

void _alarm_cleanup()
{
    int ret = alarm_foreach_registered_alarm(on_foreach_cancel_alarm, NULL);
    if (ret != ALARM_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "alarm_foreach_registered_alarm() failed, err = %d", ret);

    if (app_control) {
        ret = app_control_destroy(app_control);
        if (ret != ALARM_ERROR_NONE)
            dlog_print(DLOG_ERROR, LOG_TAG, "app_control_destroy() failed, err = %d", ret);

        app_control = NULL;
    }
}

void _alarm_single(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    // Setting an Alarm after Specific Time

    static int DELAY = 2;
    static int REMIND = 5;
    int alarm_id;
    int ret = alarm_schedule_after_delay(app_control, DELAY, REMIND, &alarm_id);
    if (ret != ALARM_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "schedule Error : %d", ret);

    PRINT_MSG("alarm_schedule_after_delay %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    // Setting an Alarm on Specific Date

    struct tm date;
    ret = alarm_get_current_time(&date);
    if (ret != ALARM_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "Get time Error : %d", ret);

    PRINT_MSG("alarm_get_current_time %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    date.tm_sec += 4;
    ret = alarm_schedule_once_at_date(app_control, &date, &alarm_id);
    if (ret != ALARM_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "schedule Error : %d", ret);

    PRINT_MSG("alarm_schedule_at_date %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");
}

void _alarm_reccuring(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    int tid;
    struct tm date;

    // Setting a Recurring Alarm at a Specific Time of the Day

    int ret = alarm_get_current_time(&date);
    if (ret != ALARM_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "alarm_get_current_time() falied : %d", ret);

    PRINT_MSG("alarm_get_current_time %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    date.tm_sec += 10;          // Add 10 seconds timeout to alarm
    ret =
        alarm_schedule_with_recurrence_week_flag(app_control, &date,
                ALARM_WEEK_FLAG_TUESDAY | ALARM_WEEK_FLAG_FRIDAY,
                &tid);
    PRINT_MSG("alarm_schedule_with_recurrence_week_flag %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");
}

void _alarm_list_all(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    // Listing All Scheduled Alarms

    int ret = alarm_foreach_registered_alarm(on_foreach_registered_alarm, NULL);
    if (ret != ALARM_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "Listing Error : %d", ret);

    PRINT_MSG("alarm_foreach_registered_alarm %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");
}

void _alarm_remove(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    // Remove All Scheduled Alarms
    alarm_foreach_registered_alarm(on_foreach_cancel_alarm, NULL);
    PRINT_MSG("Scheduled alarms removed");
}

void _app_control_uri_mime_type(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    app_control_h app_control;
    int ret = app_control_create(&app_control);
    PRINT_MSG("app_control_create %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = app_control_set_operation(app_control, APP_CONTROL_OPERATION_VIEW);
    PRINT_MSG("app_control_set_operation %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    char image_path[200];
    if (!get_sample_image_path(image_path, sizeof(image_path))) {
        PRINT_MSG("Could not get sample image path");
        return;
    }

    ret = app_control_set_uri(app_control, image_path);
    PRINT_MSG("app_control_set_uri %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = app_control_set_mime(app_control, "image/*");
    PRINT_MSG("app_control_set_mime %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = app_control_send_launch_request(app_control, NULL, NULL);
    PRINT_MSG("app_control_send_launch_request %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = app_control_destroy(app_control);
    if (ret == APP_CONTROL_ERROR_NONE)
        app_control = NULL;

    PRINT_MSG("app_control_destroy %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");
}

bool _app_control_extra_data_cb(app_control_h app_control, const char *key, void *user_data)
{
    char *value = NULL;

    int ret = app_control_get_extra_data(app_control, key, &value);
    dlog_print(DLOG_DEBUG, LOG_TAG, "KEY: %s, VALUE: %s", key, value);
    PRINT_MSG("app_control_get_extra_data %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");
    PRINT_MSG("KEY: %s, VALUE: %s", key, value);

    free(value);
    return true;
}

void _app_control_extra_data(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    app_control_h app_control;
    int ret = app_control_create(&app_control);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_create() failed, err = %d", ret);

    PRINT_MSG("app_control_create %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    char *text = "Sample text contents";

    ret = app_control_set_operation(app_control, APP_CONTROL_OPERATION_VIEW);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_set_operation() failed, err = %d", ret);

    PRINT_MSG("app_control_set_operation %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = app_control_set_mime(app_control, "text/plain");
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_set_mime() failed, err = %d", ret);

    PRINT_MSG("app_control_set_mime %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = app_control_add_extra_data(app_control, APP_CONTROL_DATA_TEXT, text);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_add_extra_data() failed, err = %d", ret);

    PRINT_MSG("app_control_add_extra_data 2/2 %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    char *my_app_id = NULL;
    app_get_id(&my_app_id);

    // Optional - set the app ID
    // If the app ID is not set, the system will decide which app should handle the request
    ret = app_control_set_app_id(app_control, my_app_id);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_set_app_id() failed, err = %d", ret);

    PRINT_MSG("app_control_set_app_id %s %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed", my_app_id);
    free(my_app_id);

    ret = app_control_send_launch_request(app_control, NULL, NULL);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_send_launch_request() failed, err = %d", ret);

    PRINT_MSG("app_control_send_launch_request %s, ret = %d",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed", ret);

    ret = app_control_foreach_extra_data(app_control, _app_control_extra_data_cb, 0);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_foreach_extra_data() failed, err = %d", ret);

    PRINT_MSG("app_control_foreach_extra_data %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = app_control_destroy(app_control);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_destroy() failed, err = %d", ret);
    else
        app_control = NULL;

    PRINT_MSG("app_control_destroy %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");
}

void _app_control_launch_mode(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    app_control_h app_control;
    int ret = app_control_create(&app_control);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_create() failed, err = %d", ret);
    PRINT_MSG("app_control_create %s", (ret == 0) ? "successful" : "failed");

    ret = app_control_set_operation(app_control, APP_CONTROL_OPERATION_VIEW);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_set_operation() failed, err = %d", ret);
    PRINT_MSG("app_control_set_operation %s", (ret == 0) ? "successful" : "failed");

    char image_path[200];
    if (!get_sample_image_path(image_path, sizeof(image_path))) {
        PRINT_MSG("Could not get sample image path");
        return;
    }

    ret = app_control_set_uri(app_control, image_path);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_set_uri() failed, err = %d", ret);
    PRINT_MSG("app_control_set_uri %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = app_control_set_mime(app_control, "image/*");
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_set_mime() failed, err = %d", ret);
    PRINT_MSG("app_control_set_mime %s", (ret == 0) ? "successful" : "failed");

    ret = app_control_set_launch_mode(app_control, APP_CONTROL_LAUNCH_MODE_GROUP);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_set_launch_mode() failed, err = %d", ret);
    PRINT_MSG("app_control_set_launch_mode %s", (ret == 0) ? "successful" : "failed");

    ret = app_control_send_launch_request(app_control, NULL, NULL);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_send_launch_request() failed, err = %d", ret);
    PRINT_MSG("app_control_send_launch_request %s", (ret == 0) ? "successful" : "failed");

    ret = app_control_destroy(app_control);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_destroy() failed, err = %d", ret);
    PRINT_MSG("app_control_destroy %s", (ret == 0) ? "successful" : "failed");
}

static void preference_changed_cb_impl(const char *key, void *user_data)
{
    dlog_print(DLOG_DEBUG, LOG_TAG, "[preference_changed_cb_impl]");

    bool existing;

    int ret = preference_is_existing(key, &existing);
    PRINT_MSG("preference_is_existing %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    if (existing) {
        int integer_output = 0;

        ret = preference_get_int(key, &integer_output);
        PRINT_MSG("preference_get_int %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

        if (NULL != user_data) {
            dlog_print(DLOG_DEBUG, LOG_TAG, "Key: %s has changed its value from %d to %d",
                       key, *((int *)user_data), integer_output);

            PRINT_MSG("Key: %s has changed its value from %d to %d", key, *((int *)user_data),
                      integer_output);
            *((int *)user_data) = integer_output;
        }
    }
}

bool preference_foreach_item_cb(const char *key, void *user_data)
{
    dlog_print(DLOG_DEBUG, LOG_TAG, "[preference_foreach_item_cb]");
    dlog_print(DLOG_DEBUG, LOG_TAG, "Key found: %s", key);
    PRINT_MSG("Key found: %s", key);
    return true;
}

bool _preference_pop_cb()
{
    // Deleting Records
    const char *integer_key = "integer_key";

    int ret = preference_unset_changed_cb(integer_key);
    PRINT_MSG("preference_unset_changed_cb %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = preference_remove(integer_key);
    PRINT_MSG("preference_remove %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = preference_remove_all();
    PRINT_MSG("preference_remove_all %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    return true;
}

static void _preference_scenario(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    // Storing and Retrieving Simple Type Variables

    const char *integer_key = "integer_key";
    int integer_value = 1;

    int ret = preference_set_int(integer_key, integer_value);
    PRINT_MSG("preference_set_int %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    bool existing;

    ret = preference_is_existing(integer_key, &existing);
    PRINT_MSG("preference_is_existing %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    int current_value = 0;

    ret = preference_get_int(integer_key, &current_value);
    PRINT_MSG("preference_get_int %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    // Storing and Retrieving String Variables

    const char *string_key = "string_key";
    const char *string_value = "Sample content";

    ret = preference_set_string(string_key, string_value);
    PRINT_MSG("preference_set_string %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = preference_is_existing(string_key, &existing);
    PRINT_MSG("preference_is_existing %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    char *string_output;

    ret = preference_get_string(string_key, &string_output);
    PRINT_MSG("preference_get_string %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    free(string_output);

    // Tracking Variables

    ret = preference_set_changed_cb(integer_key, preference_changed_cb_impl, &current_value);
    PRINT_MSG("preference_set_changed_cb %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = preference_set_int(integer_key, current_value + 1);
    PRINT_MSG("preference_set_int %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    // Listing All Records

    ret = preference_foreach_item(preference_foreach_item_cb, NULL);
    PRINT_MSG("preference_foreach_item %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");
}

void _alarm_cb(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = (appdata_s *)data;
    Evas_Object *display = _create_new_cd_display(ad, "Alarm", _alarm_cleanup);

    int ret = app_control_create(&app_control);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_create() failed, err = %d", ret);

    PRINT_MSG("app_control_create %s", (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = app_control_set_operation(app_control, APP_CONTROL_OPERATION_VIEW);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_set_operation() failed, err = %d", ret);

    ret = app_control_set_mime(app_control, "text/plain");
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_set_mime() failed, err = %d", ret);

    PRINT_MSG("app_control_set_operation %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed");

    ret = app_control_add_extra_data(app_control, APP_CONTROL_DATA_TEXT, "Alarm called");
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_add_extra_data() failed, err = %d", ret);

    char *my_app_id = NULL;
    app_get_id(&my_app_id);

    // Set the app ID
    // UI application ID is needed to run all functions successfully
    // Here is used this application ID (my_app_id)
    // It can be replaced by other application ID which you wish to run using alarm
    ret = app_control_set_app_id(app_control, my_app_id);
    if (ret != APP_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "app_control_set_app_id() failed, err = %d", ret);

    PRINT_MSG("app_control_set_app_id %s %s",
              (ret == APP_CONTROL_ERROR_NONE) ? "successful" : "failed", my_app_id);
    free(my_app_id);

    _new_button(ad, display, "Scheduling a Single Occurrence", _alarm_single);
    _new_button(ad, display, "Scheduling a Recurring Alarm", _alarm_reccuring);
    _new_button(ad, display, "List All Scheduled Alarms", _alarm_list_all);
    _new_button(ad, display, "Remove Scheduled Alarms", _alarm_remove);
}

void _app_control_cb(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = (appdata_s *)data;
    Evas_Object *display = _create_new_cd_display(ad, "App Control", NULL);

    _new_button(ad, display, "Launching with a URI and MIME Type", _app_control_uri_mime_type);
    _new_button(ad, display, "Launching with Extra Data", _app_control_extra_data);
    _new_button(ad, display, "Controlling the Launch Mode", _app_control_launch_mode);
}

static void _preference_cb(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = (appdata_s *)data;
    Evas_Object *display = _create_new_cd_display(ad, "Preference", _preference_pop_cb);

    _new_button(ad, display, "Start", _preference_scenario);
}

void create_buttons_in_main_window(appdata_s *ad)
{
    elm_list_item_append(ad->parent_list, "Alarm", NULL, NULL, _alarm_cb, ad);
    elm_list_item_append(ad->parent_list, "App Control", NULL, NULL, _app_control_cb, ad);
    elm_list_item_append(ad->parent_list, "Preference", NULL, NULL, _preference_cb, ad);
}
