/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "user_callbacks.h"
#include "main.h"

#include <app_manager.h>

void _app_manager_and_context(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    app_context_h app_context = NULL;

    // Get the context of the currently running application
    char *my_app_id = NULL;
    int ret = app_get_id(&my_app_id);
    if (ret != APP_ERROR_NONE) {
        PRINT_MSG("app_get_id failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_get_id failed, error: %d", ret);
    }

    PRINT_MSG("App ID: %s", my_app_id);

    bool running = false;
    ret = app_manager_is_running(my_app_id, &running);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_manager_is_running failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_manager_is_running failed, error: %d", ret);
    }

    PRINT_MSG("The app is running");

    char *path = "";
    ret = app_manager_get_shared_resource_path(my_app_id, &path);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_manager_get_shared_resource_path failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_manager_get_shared_resource_path failed, error: %d", ret);
    }

    PRINT_MSG("Shared resources path: %s", path);
    free(path);

    ret = app_manager_get_app_context(my_app_id, &app_context);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_manager_get_app_context failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_manager_get_app_context failed, error: %d", ret);
    }

    free(my_app_id);

    // Get the application ID and application process ID
    char *app_id = NULL;
    int pid = 0;

    ret = app_context_get_app_id(app_context, &app_id);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_context_get_app_id failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_context_get_app_id failed, error: %d", ret);
    }

    PRINT_MSG("App ID by context: %s", app_id);

    ret = app_context_get_pid(app_context, &pid);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_context_get_pid failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_context_get_pid failed, error: %d", ret);
    }

    PRINT_MSG("The app PID: %d", pid);

    // Check whether the application with the given application context is terminated
    bool terminated = false;

    ret = app_context_is_terminated(app_context, &terminated);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_context_is_terminated failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_context_is_terminated failed, error: %d", ret);
    }

    PRINT_MSG("The app is %s", (terminated) ? "terminated" : "running");

    free(app_id);

    // Clone the given context handle
    app_context_h app_context_cloned = NULL;

    ret = app_context_clone(&app_context_cloned, app_context);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_context_is_terminated failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_context_is_terminated failed, error: %d", ret);
    }

    PRINT_MSG("Context cloned successfully");

    // Check whether two contexts are equal
    bool equal = false;

    ret = app_context_is_equal(app_context, app_context_cloned, &equal);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_context_is_equal failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_context_is_equal failed, error: %d", ret);
    }

    if (equal) {
        PRINT_MSG("Contexts are equal");
        dlog_print(DLOG_INFO, LOG_TAG, "Contexts are equal");
    } else {
        PRINT_MSG("Contexts are not equal");
        dlog_print(DLOG_ERROR, LOG_TAG, "Contexts are equal");
    }

    // Release all resources
    ret = app_context_destroy(app_context);
    if (APP_MANAGER_ERROR_NONE != ret) {
        PRINT_MSG("app_context_destroy failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_context_destroy failed, error: %d", ret);
    }

    ret = app_context_destroy(app_context_cloned);
    if (APP_MANAGER_ERROR_NONE != ret) {
        PRINT_MSG("app_context_destroy failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_context_destroy failed, error: %d", ret);
    }
}

void _app_info(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    char *my_app_id = NULL;
    int ret = app_get_id(&my_app_id);
    if (ret != APP_ERROR_NONE) {
        PRINT_MSG("app_get_id failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_get_id failed, error: %d", ret);
    }
    PRINT_MSG("App ID: %s", my_app_id);

    app_info_h app_info = NULL;
    ret = app_info_create(my_app_id, &app_info);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_info_create failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_info_create failed, error: %d", ret);
    }
    free(my_app_id);

    char *label = NULL;
    ret = app_info_get_label(app_info, &label);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_info_get_label failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_info_get_label failed, error: %d", ret);
    }

    PRINT_MSG("Label: %s", label);
    free(label);

    char *exec = NULL;
    ret = app_info_get_exec(app_info, &exec);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_info_get_exec failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_info_get_exec failed, error: %d", ret);
    }

    PRINT_MSG("Exec path: %s", exec);
    free(exec);

    char *type = NULL;
    ret = app_info_get_type(app_info, &type);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_info_get_type failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_info_get_type failed, error: %d", ret);
    }

    PRINT_MSG("Type: %s", type);
    free(type);

    bool is_onboot = false;
    ret = app_info_is_onboot(app_info, &is_onboot);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_info_is_onboot failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_info_is_onboot failed, error: %d", ret);
    }

    PRINT_MSG("On boot: %s", (is_onboot) ? "yes" : "no");

    bool is_preloaded = false;
    ret = app_info_is_preload(app_info, &is_preloaded);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_info_is_preload failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_info_is_preload failed, error: %d", ret);
    }

    PRINT_MSG("Is preloaded: %s", (is_preloaded) ? "yes" : "no");
}

bool app_info_filtered_cb(app_info_h app_info, void *user_data)
{
    int ret;

    char *app_id = NULL;

    ret = app_info_get_app_id(app_info, &app_id);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_info_get_app_id failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_info_get_app_id failed, error: %d", ret);
        return true;
    }

    PRINT_MSG("App ID: %s", app_id);
    dlog_print(DLOG_DEBUG, LOG_TAG, "App ID: %s", app_id);

    free(app_id);

    return true;
}

void _app_info_filtering(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    int ret;

    app_info_filter_h app_info_filter = NULL;

    ret = app_info_filter_create(&app_info_filter);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_info_filter_create failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_info_filter_create failed, error: %d", ret);
        return;
    }

    ret = app_info_filter_add_string(app_info_filter, PACKAGE_INFO_PROP_APP_TYPE, "capp");
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_info_filter_add_string failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_info_filter_add_string failed, error: %d", ret);
        return;
    }

    ret = app_info_filter_foreach_appinfo(app_info_filter, app_info_filtered_cb, NULL);
    if (ret != APP_MANAGER_ERROR_NONE) {
        PRINT_MSG("app_info_filter_foreach_appinfo failed, error: %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "app_info_filter_foreach_appinfo failed, error: %d", ret);
        return;
    }

    if (app_info_filter) {
        ret = app_info_filter_destroy(app_info_filter);
        if (ret != APP_MANAGER_ERROR_NONE) {
            PRINT_MSG("app_info_filter_destroy failed, error: %d", ret);
            dlog_print(DLOG_ERROR, LOG_TAG, "app_info_filter_destroy failed, error: %d", ret);
        } else {
            app_info_filter = NULL;
        }
    }
}

void create_buttons_in_main_window(appdata_s *ad)
{
    Evas_Object *display = _create_new_cd_display(ad, "Application Manager", _pop_cb);
    _new_button(ad, display, "Application Manager and Context", _app_manager_and_context);
    _new_button(ad, display, "Application Information", _app_info);
    _new_button(ad, display, "Application Information Filtering", _app_info_filtering);
}
