/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "user_callbacks.h"
#include "main.h"

#include <bundle.h>

char *bundle_value_type_to_str(const int type)
{
    switch (type) {
    case BUNDLE_TYPE_NONE:
        return "BUNDLE_TYPE_NONE";

    case BUNDLE_TYPE_ANY:
        return "BUNDLE_TYPE_ANY";

    case BUNDLE_TYPE_STR:
        return "BUNDLE_TYPE_STR";

    case BUNDLE_TYPE_STR_ARRAY:
        return "BUNDLE_TYPE_STR_ARRAY";

    case BUNDLE_TYPE_BYTE:
        return "BUNDLE_TYPE_BYTE";

    case BUNDLE_TYPE_BYTE_ARRAY:
        return "BUNDLE_TYPE_BYTE_ARRAY";

    default:
        return "Unknown";
    }
}

void bundle_foreach_cb(const char *key, const int type, const bundle_keyval_t *kv, void *user_data)
{
    dlog_print(DLOG_DEBUG, LOG_TAG, "key : %s, type : %d ", key, type);
    PRINT_MSG("KEY: %s, TYPE: %s", key, bundle_value_type_to_str(type));

    void *ptr = NULL;
    char *buff = NULL;
    unsigned int size = 0;

    if (type == BUNDLE_TYPE_STR) {
        bundle_keyval_get_basic_val((bundle_keyval_t *)kv, &ptr, &size);
        buff = malloc(sizeof(char) * size + 1);
        snprintf(buff, size + 1, "%s", ((char *)ptr));
        PRINT_MSG("Found STR: KEY: %s, VAL: %s, SIZE: %d", key, buff, size);
        dlog_print(DLOG_DEBUG, LOG_TAG, "Found STR: KEY: %s, VAL: %s, SIZE: %d", key, buff, size);
        free(buff);
    } else if (type == BUNDLE_TYPE_BYTE) {
        bundle_keyval_get_basic_val((bundle_keyval_t *)kv, &ptr, &size);
        buff = malloc(sizeof(char) * size + 1);
        snprintf(buff, size + 1, "%s", ((char *)ptr));
        PRINT_MSG("Found BYTE: KEY: %s, VAL: %s, SIZE: %d", key, buff, size);
        dlog_print(DLOG_DEBUG, LOG_TAG, "Found BYTE: KEY: %s, VAL: %s, SIZE: %d", key, buff, size);
        free(buff);
    } else if (type == BUNDLE_TYPE_STR_ARRAY) {
        void **array;
        unsigned int len = 0;
        size_t *element_size = NULL;
        dlog_print(DLOG_DEBUG, LOG_TAG, "Found STR_ARRAY: KEY: %s", key);
        PRINT_MSG("Found STR_ARRAY: KEY: %s", key);
        bundle_keyval_get_array_val((bundle_keyval_t *)kv, &array, &len, &element_size);
        dlog_print(DLOG_DEBUG, LOG_TAG, "Array length: %d", len);
        PRINT_MSG("Array length: %d", len);

        char **c_array = (char **)array;

        for (int i = 0; i < len; i++)
            PRINT_MSG("[%d]: %s", i, (char *)c_array[i]);
    }
}

void _bundle(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    int ret = 0;
    bundle *bund = NULL;

    // Create a bundle, add key/value pairs
    bund = bundle_create();
    PRINT_MSG("bundle_create %s", (bund != NULL) ? "successful" : "failed");

    const char *array[3] = { "Var1", "Var2", "Var3" };
    int array_len = 3;

    ret = bundle_add_str(bund, "Str", "String content");
    PRINT_MSG("bundle_add_str %s", (ret == 0) ? "successful" : "failed");

    ret = bundle_add_str_array(bund, "Array", array, array_len);
    PRINT_MSG("bundle_add_str_array %s", (ret == 0) ? "successful" : "failed");

    ret = bundle_add_byte(bund, "Byte", "Byte content", 12);
    PRINT_MSG("bundle_add_byte %s", (ret == 0) ? "successful" : "failed");

    ret = bundle_free(bund);
    PRINT_MSG("bundle_free %s", (ret == 0) ? "successful" : "failed");

    // Add, delete and get data
    bundle *b = NULL;
    int count = 0;
    char *value;

    b = bundle_create();

    bundle_add_str(b, "key1", "val1");
    bundle_add_str(b, "key2", "val2");
    bundle_get_str(b, "key2", &value);
    dlog_print(DLOG_DEBUG, LOG_TAG, "Value of key2 : %s", value);

    count = bundle_get_count(b);
    dlog_print(DLOG_DEBUG, LOG_TAG, "Number of bundle items : %d", count);

    bundle_del(b, "key2");

    count = bundle_get_count(b);
    dlog_print(DLOG_DEBUG, LOG_TAG, "Number of bundle items : %d", count);

    ret = bundle_free(b);
    PRINT_MSG("bundle_free %s", (ret == 0) ? "successful" : "failed");
    b = NULL;

    // For each key/value pair ...
    const char *s_arr[] = { "abc", "bcd", "cde" };
    b = bundle_create();

    bundle_add_str(b, "k1", "v1");
    bundle_add_byte(b, "k2", "v2", 3);
    bundle_add_str_array(b, "k3", s_arr, 3);

    bundle_foreach(b, bundle_foreach_cb, NULL);

    ret = bundle_free(b);
    PRINT_MSG("bundle_free %s", (ret == 0) ? "successful" : "failed");

    // Encode and decode
    bundle *b1, *b2;
    bundle_raw *r;
    int size_r;

    b1 = bundle_create();
    bundle_add_str(b1, "k1", "v1");
    bundle_add_str(b1, "k2", "v2");

    bundle_encode(b1, &r, &size_r);

    b2 = bundle_decode(r, size_r);

    bundle_get_str(b2, "k1", &value);
    PRINT_MSG("Value of k1 after decoding: %s", value);
    dlog_print(DLOG_DEBUG, LOG_TAG, "Value of k1 after decoding: %s", value);

    ret = bundle_free(b1);
    PRINT_MSG("bundle_free %s", (ret == 0) ? "successful" : "failed");

    ret = bundle_free(b2);
    PRINT_MSG("bundle_free %s", (ret == 0) ? "successful" : "failed");
}

void create_buttons_in_main_window(appdata_s *ad)
{
    Evas_Object *display = _create_new_cd_display(ad, "Bundle", _pop_cb);
    _new_button(ad, display, "Start", _bundle);
}
