/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "main.h"

typedef struct map_data {
    char **str_arr;
    int arr_size;
} map_data_s;

static GHashTable *map_repository_test;

void get_value_request_cb(int request_id, data_control_h provider, const char *key, void *user_data)
{
    map_data_s *map_data = (map_data_s *)g_hash_table_lookup(map_repository_test, key);

    int ret_value_count = 0;
    char **val_arr = NULL;

    if (map_data != NULL) {
        val_arr = map_data->str_arr;
        ret_value_count = map_data->arr_size;
    }

    int ret = data_control_provider_send_map_get_value_result(request_id, val_arr, ret_value_count);
    if (ret != DATA_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "send_map_get_result failed with error: %d", ret);
    else
        dlog_print(DLOG_INFO, LOG_TAG, "Get value success request_id : %d", request_id);
}

void set_value_request_cb(int request_id, data_control_h provider, const char *key,
                          const char *old_value, const char *new_value, void *user_data)
{
    map_data_s *map_data = (map_data_s *)g_hash_table_lookup(map_repository_test, key);

    if (map_data != NULL) {
        for (int i = 0; i < map_data->arr_size; i++) {
            if (strcmp(map_data->str_arr[i], old_value) == 0)
                map_data->str_arr[i] = g_strdup(new_value);
        }
    }

    int ret = data_control_provider_send_map_result(request_id);
    if (ret != DATA_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "send_map_result failed with error: %d", ret);
    else
        dlog_print(DLOG_INFO, LOG_TAG, "Set value success request_id : %d", request_id);
}

void add_value_request_cb(int request_id, data_control_h provider, const char *key,
                          const char *value, void *user_data)
{
    map_data_s *map_data = (map_data_s *)g_hash_table_lookup(map_repository_test, key);

    if (map_data == NULL) {
        map_data = (map_data_s *)(g_malloc(sizeof(*map_data)));
        map_data->arr_size = 0;
        map_data->str_arr = (char **)calloc(1, sizeof(char *));
        map_data->str_arr[0] = g_strdup(value);
        g_hash_table_insert(map_repository_test, g_strdup(key), map_data);
    } else {
        char **new_arr = (char **)calloc(map_data->arr_size + 2, sizeof(char *));

        for (int i = 0; i < map_data->arr_size; i++) {
            new_arr[i] = g_strdup(map_data->str_arr[i]);
            free(map_data->str_arr[i]);
        }

        free(map_data->str_arr);
        new_arr[map_data->arr_size] = g_strdup(value);
        map_data->str_arr = new_arr;
    }

    map_data->arr_size += 1;

    int ret = data_control_provider_send_map_result(request_id);
    if (ret != DATA_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "send_map_result failed with error: %d", ret);
    else {
        dlog_print(DLOG_INFO, LOG_TAG, "Add value success request_id : %d %d %s", request_id,
                   map_data->arr_size, map_data->str_arr[0]);
    }
}

void remove_value_request_cb(int request_id, data_control_h provider, const char *key,
                             const char *value, void *user_data)
{
    map_data_s *map_data = (map_data_s *)g_hash_table_lookup(map_repository_test, key);

    if (map_data != NULL) {
        int size = map_data->arr_size;

        for (int i = 0; i < size; i++) {
            if (strcmp(map_data->str_arr[i], value) == 0) {
                free(map_data->str_arr[i]);
                map_data->arr_size--;
            }
        }

        if (map_data->arr_size == 0) {
            if (!g_hash_table_remove(map_repository_test, key)) {
                dlog_print(DLOG_ERROR, LOG_TAG, "remove value fail -%s", key);
                return;
            }
        }
    }

    int ret = data_control_provider_send_map_result(request_id);
    if (ret != DATA_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "send_map_result failed with error: %d", ret);
    else
        dlog_print(DLOG_INFO, LOG_TAG, "Remove value Success");
}

static sqlite3 *db = NULL;

void insert_request_cb(int request_id, data_control_h provider, bundle *insert_data,
                       void *user_data)
{
    char *str = NULL;
    bundle_get_str(insert_data, "WORD", &str);
    dlog_print(DLOG_INFO, LOG_TAG, "[insert_request_cb] send result rq_id %d bundle_string %s",
               request_id, str);
    char *command = data_control_provider_create_insert_statement(provider, insert_data);

    if (command == NULL) {
        dlog_print(DLOG_ERROR, LOG_TAG, "command is NULL in insert_request_cb()");
        return;
    }

    dlog_print(DLOG_INFO, LOG_TAG, "SQL insert: %s", command);
    dlog_print(DLOG_INFO, LOG_TAG, "Command %s", command);

    int ret = sqlite3_exec(db, command, NULL, NULL, NULL);
    if (ret != SQLITE_OK) {
        ret = data_control_provider_send_error(request_id, sqlite3_errmsg(db));
        free(command);
        return;
    }

    long long inserted_row_id = sqlite3_last_insert_rowid(db);
    dlog_print(DLOG_INFO, LOG_TAG, "inserted_row_id %d", inserted_row_id);
    ret = data_control_provider_send_insert_result(request_id, inserted_row_id);
    if (ret != DATA_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "insert_send_result failed with error: %d", ret);
    else
        dlog_print(DLOG_INFO, LOG_TAG, "[insert_request_cb] send result success");

    free(command);
}

void delete_request_cb(int request_id, data_control_h provider, const char *where, void *user_data)
{
    dlog_print(DLOG_INFO, LOG_TAG, "[delete_request_cb] request_id(%d)", request_id);
    char *command = data_control_provider_create_delete_statement(provider, where);
    int ret = sqlite3_exec(db, command, NULL, NULL, NULL);
    if (ret != SQLITE_OK) {
        ret = data_control_provider_send_error(request_id, sqlite3_errmsg(db));
        free(command);
        return;
    }

    ret = data_control_provider_send_delete_result(request_id);
    if (ret != DATA_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "delete_send_result failed with error: %d", ret);
    else
        dlog_print(DLOG_INFO, LOG_TAG, "[delete_request_cb] delete success");

    free(command);
}

void select_request_cb(int request_id, data_control_h provider, const char **column_list,
                       int column_count, const char *where, const char *order, void *user_data)
{
    sqlite3_stmt *sql_stmt = NULL;

    char *command =
        data_control_provider_create_select_statement(provider, column_list, column_count, where,
                order);
    dlog_print(DLOG_INFO, LOG_TAG, "SQL select: %s", command);

    if (command == NULL) {
        dlog_print(DLOG_ERROR, LOG_TAG, "command is NULL in select_request_cb()");
        return;
    }

    int ret = sqlite3_prepare_v2(db, command, strlen(command), &sql_stmt, NULL);
    if (ret != SQLITE_OK) {
        ret = data_control_provider_send_error(request_id, sqlite3_errmsg(db));
        free(command);
        return;
    }

    ret = data_control_provider_send_select_result(request_id, (void *)sql_stmt);
    if (ret != DATA_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "select_send_result failed with error: %d", ret);
    else
        dlog_print(DLOG_INFO, LOG_TAG, "[select_request_cb] send result success");

    sqlite3_finalize(sql_stmt);
    free(command);
}

void update_request_cb(int request_id, data_control_h provider, bundle *update_data,
                       const char *where, void *user_data)
{
    char *command = data_control_provider_create_update_statement(provider, update_data, where);
    int ret = sqlite3_exec(db, command, NULL, NULL, NULL);
    if (ret != SQLITE_OK) {
        ret = data_control_provider_send_error(request_id, sqlite3_errmsg(db));
        free(command);
        return;
    }

    ret = data_control_provider_send_update_result(request_id);
    if (ret != DATA_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "update_send_result failed with error: %d", ret);
    else
        dlog_print(DLOG_INFO, LOG_TAG, "[update_request_cb] send result success");

    free(command);
}

data_control_provider_map_cb map_provider_callback;
data_control_provider_sql_cb sql_provider_callback;
data_control_h provider = NULL;

void __free_key(gpointer data)
{
    if (data) {
        g_free(data);
        data = NULL;
        dlog_print(DLOG_INFO, LOG_TAG, "Remove key");
    }
}

void __free_data(gpointer data)
{
    if (data) {
        g_free(data);
        data = NULL;
        dlog_print(DLOG_INFO, LOG_TAG, "Remove value");
    }
}

int create_database()
{
    char db_path[200];
    snprintf(db_path, 200, "%s%s", app_get_data_path(), "test.db");
    dlog_print(DLOG_INFO, LOG_TAG, "%s", db_path);

    int open_flags = (SQLITE_OPEN_READWRITE | SQLITE_OPEN_CREATE);

    int ret = sqlite3_open_v2(db_path, &db, open_flags, NULL);
    if (ret != SQLITE_OK) {
        dlog_print(DLOG_ERROR, LOG_TAG, "database creation failed with error: %d", ret);
        return ret;
    }

    char *sql_command =
        "CREATE TABLE IF NOT EXISTS Dictionary ( WORD VARCHAR(30) , WORD_DESC TEXT, WORD_NUM INT, Point INT)";
    ret = sqlite3_exec(db, sql_command, NULL, NULL, NULL);
    if (ret != SQLITE_OK)
        dlog_print(DLOG_ERROR, LOG_TAG, "database table creation failed with error: %d", ret);

    dlog_print(DLOG_INFO, LOG_TAG, "DB init Success.");
    return ret;
}

void initialize_datacontrol_provider()
{
    // Snippets initialize_datacontrol_provider() part
    // MAP provider
    int ret = 0;

    map_repository_test = g_hash_table_new_full(g_str_hash, g_str_equal, __free_key, __free_data);

    map_provider_callback.get_cb = get_value_request_cb;
    map_provider_callback.add_cb = add_value_request_cb;
    map_provider_callback.remove_cb = remove_value_request_cb;
    map_provider_callback.set_cb = set_value_request_cb;

    ret = data_control_provider_map_register_cb(&map_provider_callback, NULL);
    if (ret != DATA_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "data_control_provider_map_cb failed with error: %d", ret);
    else
        dlog_print(DLOG_INFO, LOG_TAG, "Provider map register success");

    // SQL provider
    ret = create_database();
    if (ret != SQLITE_OK) {
        dlog_print(DLOG_ERROR, LOG_TAG, "Create database failed");
        return;
    }

    sql_provider_callback.select_cb = select_request_cb;
    sql_provider_callback.insert_cb = insert_request_cb;
    sql_provider_callback.delete_cb = delete_request_cb;
    sql_provider_callback.update_cb = update_request_cb;

    ret = data_control_provider_sql_register_cb(&sql_provider_callback, NULL);
    if (ret != DATA_CONTROL_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "data_control_provider_sql_cb failed with error: %d", ret);
}

void deinitialize_datacontrol_provider()
{
    int ret;

    ret = data_control_provider_map_unregister_cb();
    if (ret != DATA_CONTROL_ERROR_NONE) {
        dlog_print(DLOG_ERROR, LOG_TAG,
                   "data_control_provider_map_unregister_cb failed with error: %d", ret);
    }

    ret = data_control_provider_sql_unregister_cb();
    if (ret != DATA_CONTROL_ERROR_NONE) {
        dlog_print(DLOG_ERROR, LOG_TAG,
                   "data_control_provider_sql_unregister_cb failed with error: %d", ret);
    }
}

static bool service_app_create(void *data)
{
    dlog_print(DLOG_DEBUG, LOG_TAG, "%s", __func__);

    /* Hook to take necessary actions before main event loop starts
       Initialize UI resources and application's data
       If this function returns true, the main loop of application starts
       If this function returns false, the application is terminated */
    initialize_datacontrol_provider();

    return true;
}

void service_app_terminate(void *data)
{
    dlog_print(DLOG_DEBUG, LOG_TAG, "%s", __func__);

    deinitialize_datacontrol_provider();

    return;
}

void service_app_control(app_control_h app_control, void *data)
{
    dlog_print(DLOG_DEBUG, LOG_TAG, "%s", __func__);

    return;
}

int main(int argc, char *argv[])
{
    service_app_lifecycle_callback_s event_callback = { 0, };

    event_callback.create = service_app_create;
    event_callback.terminate = service_app_terminate;
    event_callback.app_control = service_app_control;

    int ret = service_app_main(argc, argv, &event_callback, NULL);
    if (ret != APP_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "service_app_main() failed with error: %d", ret);

    return ret;
}
