/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "user_callbacks.h"
#include "main.h"
#include <notification.h>

#define BUFLEN 200

static notification_h notification = NULL;
static Ecore_Timer *notification_timer;


void remove_notification()
{
    // Delete existing notification

    int ret;

    if (notification != NULL) {
        ret = notification_delete(notification);
        PRINT_MSG("notification_delete %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");
        dlog_print(DLOG_INFO, LOG_TAG, "notification_delete %s",
                   (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

        ret = notification_free(notification);
        PRINT_MSG("notification_free %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");
        dlog_print(DLOG_INFO, LOG_TAG, "notification_free %s",
                   (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");
        notification = NULL;
    }
}

void remove_ongoing_notification()
{
    // Delete existing ongoing notification

    int ret;

    ecore_timer_del(notification_timer);

    notification_h notification_prog = notification_load_by_tag("noti_tag");
    if (notification_prog != NULL) {
        ret = notification_delete(notification_prog);
        dlog_print(DLOG_INFO, LOG_TAG, "notification_delete %s",
                   (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

        ret = notification_free(notification_prog);
        dlog_print(DLOG_INFO, LOG_TAG, "notification_free %s",
                   (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");
    }
}

static bool get_image_path(char *path, int size)
{
    char *shared_path = app_get_shared_resource_path();
    if (shared_path == NULL) {
        PRINT_MSG("Could not get shared resource path");
        return false;
    }

    snprintf(path, size, "%s$(appName).png", shared_path);
    free(shared_path);
    return true;
}

void notification_cleanup()
{
    remove_notification();
    remove_ongoing_notification();
}

void _post_notification(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    int ret;

    char image_path[BUFLEN];
    if (!get_image_path(image_path, BUFLEN)) {
        PRINT_MSG("Could not get image path");
        return;
    }

    // Delete existing notification
    remove_notification();

    // Create a new notification
    notification = notification_create(NOTIFICATION_TYPE_NOTI);
    PRINT_MSG("notification_create %s", (notification != NULL) ? "successful" : "failed");

    // Set notification attributes

    ret =
        notification_set_text(notification, NOTIFICATION_TEXT_TYPE_TITLE, "Notification text", NULL,
                              NOTIFICATION_VARIABLE_TYPE_NONE);
    PRINT_MSG("notification_set_text %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    ret = notification_set_time_to_text(notification, NOTIFICATION_TEXT_TYPE_CONTENT, time(NULL));
    PRINT_MSG("notification_set_time_to_text %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    ret = notification_set_image(notification, NOTIFICATION_IMAGE_TYPE_ICON, image_path);
    PRINT_MSG("notification_set_image %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    ret =
        notification_set_display_applist(notification,
                                         NOTIFICATION_DISPLAY_APP_NOTIFICATION_TRAY |
                                         NOTIFICATION_DISPLAY_APP_TICKER);
    PRINT_MSG("notification_set_display_applist %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    ret = notification_set_led(notification, NOTIFICATION_LED_OP_ON, 100);
    PRINT_MSG("notification_set_led %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    ret = notification_set_led_time_period(notification, 100, 100);
    PRINT_MSG("notification_set_led_time_period %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    // Post the notification

    ret = notification_post(notification);
    PRINT_MSG("notification_post %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    // Update notification content

    ret = notification_set_property(notification, NOTIFICATION_PROP_DISABLE_APP_LAUNCH);
    PRINT_MSG("notification_set_property %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    ret = notification_update(notification);
    PRINT_MSG("notification_update %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");
}

static int noti_prog_counter = 0;
static Eina_Bool timeout_func(void *data)
{
    double progress = 0;
    noti_prog_counter++;

    notification_h notification_prog = notification_load_by_tag("noti_tag");
    if (notification_prog == NULL) {
        PRINT_MSG("notification_load_by_tag() returned NULL");
        return ECORE_CALLBACK_DONE;
    }

    // Update notification progress
    notification_set_progress(notification_prog, (double)noti_prog_counter / 10.0);

    int ret =
        notification_set_text(notification_prog, NOTIFICATION_TEXT_TYPE_TITLE, "Progress %d%%",
                              NULL, NOTIFICATION_VARIABLE_TYPE_INT, noti_prog_counter * 10,
                              NOTIFICATION_VARIABLE_TYPE_NONE);
    PRINT_MSG("notification_set_text %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    notification_update(notification_prog);
    PRINT_MSG("notification_update %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    notification_get_progress(notification_prog, &progress);
    PRINT_MSG("notification_get_progress %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    PRINT_MSG("Progress: %f\n", progress);
    dlog_print(DLOG_INFO, LOG_TAG, "Progress: %f\n", progress);

    if (10 == noti_prog_counter) {
        // Remove the notification
        ret = notification_delete(notification_prog);
        PRINT_MSG("notification_delete %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");
        ret = notification_free(notification_prog);
        PRINT_MSG("notification_free %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");
        notification_prog = NULL;
        dlog_print(DLOG_INFO, LOG_TAG, "Process completed\n");
        noti_prog_counter = 0;
        return ECORE_CALLBACK_DONE;
    }

    return ECORE_CALLBACK_PASS_ON;
}

void _post_noti_progress_bar(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    notification_h notification_prog = notification_load_by_tag("noti_tag");
    if (notification_prog != NULL) {
        PRINT_MSG("Progress notification is ongoing");
        return;
    }

    char image_path[BUFLEN];
    if (!get_image_path(image_path, BUFLEN)) {
        PRINT_MSG("Could not get image path");
        return;
    }

    // Create the notification
    notification_prog = notification_create(NOTIFICATION_TYPE_ONGOING);
    PRINT_MSG("notification_create %s", (notification_prog != NULL) ? "successful" : "failed");

    // Set parameters
    int ret =
        notification_set_text(notification_prog, NOTIFICATION_TEXT_TYPE_TITLE, "Progress %d%%",
                              NULL, NOTIFICATION_VARIABLE_TYPE_INT, 0,
                              NOTIFICATION_VARIABLE_TYPE_NONE);
    PRINT_MSG("notification_set_text %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    ret = notification_set_image(notification, NOTIFICATION_IMAGE_TYPE_ICON, image_path);
    PRINT_MSG("notification_set_image %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    ret = notification_set_progress(notification_prog, 0);
    PRINT_MSG("notification_set_progress %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    ret = notification_set_tag(notification_prog, "noti_tag");
    PRINT_MSG("notification_set_tag %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    // Display the notification
    ret = notification_post(notification_prog);
    PRINT_MSG("notification_post %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    // Change status of the notification
    noti_prog_counter = 0;
    notification_timer = ecore_timer_add(3, timeout_func, NULL);

    ret = notification_set_layout(notification_prog, NOTIFICATION_LY_ONGOING_PROGRESS);
    PRINT_MSG("notification_set_layout %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    ret = notification_set_display_applist(notification_prog, NOTIFICATION_DISPLAY_APP_NOTIFICATION_TRAY);
    PRINT_MSG("notification_set_display_applist %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");

    PRINT_MSG("Check the ongoing notification appeared in the quickpanel");

}

void _show_status_message(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    int ret = notification_status_message_post("Sample status message");
    PRINT_MSG("notification_status_message_post %s", (ret == NOTIFICATION_ERROR_NONE) ? "successful" : "failed");
}

void _post_notification_button(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    int ret;

    char image_path[BUFLEN];
    if (!get_image_path(image_path, BUFLEN)) {
        PRINT_MSG("Could not get image path");
        return;
    }

    // Delete existing notification
    remove_notification();

    // Create a new notification
    notification = notification_create(NOTIFICATION_TYPE_NOTI);
    PRINT_MSG("notification_create %s", (notification != NULL) ? "successful" : "failed");

    // Set notification attributes
    ret = notification_set_text(notification, NOTIFICATION_TEXT_TYPE_TITLE,
        "Notification with a button", NULL, NOTIFICATION_VARIABLE_TYPE_NONE);
    PRINT_MSG("notification_set_text %s", (ret == 0) ? "successful" : "failed");

    ret = notification_set_time_to_text(notification, NOTIFICATION_TEXT_TYPE_CONTENT, time(NULL));
    PRINT_MSG("notification_set_time_to_text %s", (ret == 0) ? "successful" : "failed");

    ret = notification_set_image(notification, NOTIFICATION_IMAGE_TYPE_ICON, image_path);
    PRINT_MSG("notification_set_image %s", (ret == 0) ? "successful" : "failed");

    ret = notification_add_button(notification, NOTIFICATION_BUTTON_1);
    PRINT_MSG("notification_add_button %s", (ret == 0) ? "successful" : "failed");

    app_control_h app_control = NULL;

    ret = app_control_create(&app_control);
    PRINT_MSG("app_control_create %s", (ret == 0) ? "successful" : "failed");

    app_control_set_app_id(app_control, "org.tizen.gallery");
    PRINT_MSG("app_control_set_app_id %s", (ret == 0) ? "successful" : "failed");

    ret = notification_set_event_handler(notification, NOTIFICATION_EVENT_TYPE_CLICK_ON_BUTTON_1, app_control);
    PRINT_MSG("notification_set_event_handler %s", (ret == 0) ? "successful" : "failed");

    ret = app_control_destroy(app_control);
    PRINT_MSG("app_control_destroy %s", (ret == 0) ? "successful" : "failed");

    ret = notification_post(notification);
    PRINT_MSG("notification_post %s", (ret == 0) ? "successful" : "failed");
}

void create_buttons_in_main_window(appdata_s *ad)
{
    Evas_Object *display = _create_new_cd_display(ad, "Notification", _pop_cb);

    _new_button(ad, display, "Post a Notification", _post_notification);
    _new_button(ad, display, "Post a Notification with a Progress Bar", _post_noti_progress_bar);
    _new_button(ad, display, "Post a Notification with a Button", _post_notification_button);
    _new_button(ad, display, "Show Status Message", _show_status_message);
}
