/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "user_callbacks.h"

#define BUFLEN 400

static const int key_len = 256; // AES key length

static const unsigned char password[] = { "DummyPassword" };    // Password to generate key

static sqlite3 *db;             // Database handle

static unsigned char salt[9];   // Encryption salt
static unsigned char iv[17];    // Encryption initial vector

static const char *hellomsg = "Hello Tizen! SQLite OpenSSL";
static char text[BUFLEN];

static const char encoding_table[] = {
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
    'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
    'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
    'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
    'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
    'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
    'w', 'x', 'y', 'z', '0', '1', '2', '3',
    '4', '5', '6', '7', '8', '9', '+', '/'
};

static char *decoding_table = NULL;
static int mod_table[] = { 0, 2, 1 };

/*
 * Encrypted Database
 */
static void build_decoding_table()
{
    if (decoding_table != NULL)
        free(decoding_table);

    decoding_table = malloc(256);
    int i;

    for (i = 0; i < 64; i++)
        decoding_table[(unsigned char)encoding_table[i]] = i;
}

static char *base64_encode(const unsigned char *data, size_t input_length, size_t *output_length)
{
    *output_length = 4 * ((input_length + 2) / 3);

    char *encoded_data = malloc(*output_length);

    if (encoded_data == NULL)
        return NULL;

    int i, j;

    for (i = 0, j = 0; i < input_length;) {
        uint32_t octet_a = i < input_length ? (unsigned char)data[i++] : 0;
        uint32_t octet_b = i < input_length ? (unsigned char)data[i++] : 0;
        uint32_t octet_c = i < input_length ? (unsigned char)data[i++] : 0;
        uint32_t triple = (octet_a << 0x10) + (octet_b << 0x08) + octet_c;

        encoded_data[j++] = encoding_table[(triple >> 3 * 6) & 0x3F];
        encoded_data[j++] = encoding_table[(triple >> 2 * 6) & 0x3F];
        encoded_data[j++] = encoding_table[(triple >> 1 * 6) & 0x3F];
        encoded_data[j++] = encoding_table[(triple >> 0 * 6) & 0x3F];
    }

    int size = mod_table[input_length % 3];

    for (i = 0; i < size; i++)
        encoded_data[*output_length - 1 - i] = '=';

    return encoded_data;
}

static unsigned char *base64_decode(const char *data, size_t input_length, size_t *output_length)
{
    if (decoding_table == NULL)
        build_decoding_table();

    if (input_length % 4 != 0)
        return NULL;

    *output_length = input_length / 4 * 3;

    if (data[input_length - 1] == '=')
        (*output_length)--;

    if (data[input_length - 2] == '=')
        (*output_length)--;

    unsigned char *decoded_data = malloc(*output_length);

    if (decoded_data == NULL)
        return NULL;

    int i, j;

    for (i = 0, j = 0; i < input_length;) {

        uint32_t sextet_a = data[i] == '=' ? 0 & i++ : decoding_table[(int)data[i++]];
        uint32_t sextet_b = data[i] == '=' ? 0 & i++ : decoding_table[(int)data[i++]];
        uint32_t sextet_c = data[i] == '=' ? 0 & i++ : decoding_table[(int)data[i++]];
        uint32_t sextet_d = data[i] == '=' ? 0 & i++ : decoding_table[(int)data[i++]];

        uint32_t triple =
            (sextet_a << 3 * 6) + (sextet_b << 2 * 6) + (sextet_c << 1 * 6) + (sextet_d << 0 * 6);

        if (j < *output_length)
            decoded_data[j++] = (triple >> 2 * 8) & 0xFF;

        if (j < *output_length)
            decoded_data[j++] = (triple >> 1 * 8) & 0xFF;

        if (j < *output_length)
            decoded_data[j++] = (triple >> 0 * 8) & 0xFF;
    }

    return decoded_data;
}

static int callback(void *counter, int argc, char **argv, char **azColName)
{
    int *localcounter = (int *)counter;

    PRINT_MSG("-%d: ", *localcounter);
    dlog_print(DLOG_DEBUG, LOG_TAG, "-%d: ", *localcounter);

    int i;

    for (i = 0; i < argc; i++) {
        PRINT_MSG("%s = %s | ", azColName[i], argv[i] ? argv[i] : "NULL");
        dlog_print(DLOG_DEBUG, LOG_TAG, "%s = %s | ", azColName[i], argv[i] ? argv[i] : "NULL");
    }

    (*localcounter)++;
    PRINT_MSG("\n");
    dlog_print(DLOG_DEBUG, LOG_TAG, "\n");

    return 0;
}

static void PrepareToSQL(unsigned char *msg)
{
    int i = 0;

    while (msg[i] != 0x00) {
        if (msg[i] == '\'')
            msg[i] = 'a';

        if (msg[i] == '\"')
            msg[i] = 'b';

        if (msg[i] == '-')
            msg[i] = 'c';

        if (msg[i] == '=')
            msg[i] = 'd';

        if (msg[i] == ' ')
            msg[i] = 'd';

        if (msg[i] == '\n')
            msg[i] = 'm';

        ++i;
    }
}

static int CreateTable()
{
    char *ErrMsg;

    const char *sql = "CREATE TABLE IF NOT EXISTS EncryptedData("
                      "DATA  TEXT  NOT NULL, "
                      "ENCRYPTED  INT  NOT NULL, "
                      "SALT  TEXT  NOT NULL, "
                      "IV  TEXT  NOT NULL, " "PART  INTEGER, " "KEY  INTEGER  PRIMARY KEY);";

    int ret = sqlite3_exec(db, sql, callback, 0, &ErrMsg);
    if (ret) {
        PRINT_MSG("Error: %s %d\n", ErrMsg, ret);
        sqlite3_free(ErrMsg);
    }

    return 0;
}

static int DecryptMsg(unsigned char *in, unsigned char *out, const unsigned char *password,
                      unsigned char *localsalt, unsigned char *vector)
{
    AES_KEY decryption_key;
    int iter = (int)vector[0];
    unsigned char key[key_len + 1];

    memset(out, 0x00, BUFLEN);

    PKCS5_PBKDF2_HMAC_SHA1((char *)password, sizeof(password) / sizeof(unsigned char), localsalt,
                           sizeof(localsalt) / sizeof(unsigned char), iter, key_len, key);

    AES_set_decrypt_key(key, key_len, &decryption_key);
    AES_decrypt((unsigned char *)in, out, &decryption_key);

    int x = 0;

    for (; out[x] != 0x00; x++) ;

    return x;
}

static int callbackdecrypt(void *counter, int argc, char **argv, char **azColName)
{
    unsigned char decrypted_out[BUFLEN];
    int *localcounter = (int *)counter;
    unsigned int olen;
    dlog_print(DLOG_DEBUG, LOG_TAG, "-%d: ", *localcounter);

    PRINT_MSG("-%d: ", *localcounter);

    int i;

    for (i = 0; i < argc; i++) {
        if (argv[i] && i == 0) {
            unsigned char *basebuffer = base64_decode(argv[i], strlen(argv[i]), &olen);
            unsigned char *decryptbuffer = malloc(sizeof(char) * olen + 1);
            memset(decryptbuffer, 0x00, olen + 1);
            memcpy(decryptbuffer, basebuffer, olen);
            decryptbuffer[olen] = 0x00;
            free(basebuffer);

            DecryptMsg((unsigned char *)decryptbuffer, decrypted_out, password,
                       (unsigned char *)argv[2], (unsigned char *)argv[3]);

            free(decryptbuffer);

            dlog_print(DLOG_DEBUG, LOG_TAG, "%s = %s  |  ", azColName[i], (char *)decrypted_out);

            PRINT_MSG("%s = %s  |  ", azColName[i], (char *)decrypted_out);

        } else {
            dlog_print(DLOG_DEBUG, LOG_TAG, "%s = %s | ", azColName[i], argv[i] ? argv[i] : "NULL");
            PRINT_MSG("%s = %s | ", azColName[i], argv[i] ? argv[i] : "NULL");
        }
    }

    (*localcounter)++;
    PRINT_MSG("\n");
    dlog_print(DLOG_DEBUG, LOG_TAG, "\n");
    return 0;
}

static void DecryptRecords()
{
    const char *sql = "select * from EncryptedData where ENCRYPTED='1'";
    int counter = 0;
    int ret = 0;
    char *ErrMsg = NULL;

    ret = sqlite3_exec(db, sql, callbackdecrypt, &counter, &ErrMsg);
    if (ret) {
        PRINT_MSG("Error: %s %d\n", ErrMsg, ret);
        dlog_print(DLOG_DEBUG, LOG_TAG, "Error: %s\n", ErrMsg);
        sqlite3_free(ErrMsg);
    }
}

static int EncryptMsg(char *in, unsigned char *out, const unsigned char *password,
                      unsigned char *localsalt, unsigned char *vector)
{
    AES_KEY encryption_key;
    int iter = (int)vector[0];
    unsigned char key[key_len + 1];
    char *msgbuff;
    unsigned char buf[BUFLEN];
    unsigned int retlen;
    PRINT_MSG("Encrypt: %s", in);
    memset(key, 0x00, key_len + 1);

    PKCS5_PBKDF2_HMAC_SHA1((char *)password, sizeof(password) / sizeof(unsigned char), localsalt,
                           sizeof(localsalt) / sizeof(unsigned char), iter, key_len, key);

    memset(buf, 0x00, BUFLEN);
    AES_set_encrypt_key(key, key_len, &encryption_key);
    AES_encrypt((unsigned char *)in, (unsigned char *)buf, &encryption_key);

    unsigned char decrypted_out1[BUFLEN];
    DecryptMsg((unsigned char *)buf, decrypted_out1, password, localsalt, vector);

    msgbuff = base64_encode(buf, 16, &retlen);
    memset(buf, '\0', BUFLEN);
    memcpy(buf, msgbuff, retlen);
    buf[retlen + 1] = 0x00;
    free(msgbuff);

    memcpy(out, buf, retlen + 1);

    int x;

    for (x = 0; buf[x] != 0x00; x++) ;

    return x;
}

static int InsertRecord(unsigned char *msg, int encrypted, int part, int len)
{
    char sqlbuff[BUFLEN];
    snprintf(sqlbuff, BUFLEN,
             "INSERT INTO EncryptedData VALUES(\'%s\', %d, \'%s\', \'%s\', %d, NULL);", msg,
             encrypted, salt, iv, part);

    char *ErrMsg;
    int ret = sqlite3_exec(db, sqlbuff, callback, 0, &ErrMsg);
    if (ret) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "Error: %s\n", ErrMsg);
        sqlite3_free(ErrMsg);
        return 1;
    }

    return 0;
}

static int InsertMessage(unsigned char *text)
{
    unsigned char encrypted_out[BUFLEN];
    int x;
    int len, retlen = 0, parts = 0, pos;
    char membuf[17];

    for (len = 0; text[len] != 0x00; len++) ;

    for (pos = 0; (len - pos) > 16; pos += 16) {
        memcpy(membuf, &text[pos], 16);
        membuf[16] = 0x00;

        EncryptMsg((char *)membuf, encrypted_out, password, salt, iv);

        for (x = 0; encrypted_out[x] != 0x00; x++) ;

        InsertRecord(encrypted_out, 1, parts, x);
        parts++;
    }

    if (len - pos > 0) {
        retlen = EncryptMsg((char *)&text[pos], encrypted_out, password, salt, iv);
        InsertRecord(encrypted_out, 1, parts, retlen);
    }

    return 0;
}

static void ShowRecords()
{
    const char *sql = "select * from EncryptedData";
    int counter = 0;
    char *ErrMsg;

    int ret = sqlite3_exec(db, sql, callback, &counter, &ErrMsg);
    if (ret) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "Error: %s\n", ErrMsg);
        sqlite3_free(ErrMsg);
    }

    return;
}

void _sqlite_start(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    int ret;
    unsigned char decrypted_out[BUFLEN];

    sqlite3_shutdown();

    sqlite3_config(SQLITE_CONFIG_URI, 1);

    sqlite3_initialize();

    char *resource = app_get_data_path();
    int siz = strlen(resource) + 10;

    char *path = malloc(sizeof(char) * siz);
    memset(path, 0, siz);

    strncat(path, resource, siz);
    strncat(path, "test.db", siz);

    dlog_print(DLOG_DEBUG, LOG_TAG, "DB Path: %s", path);
    PRINT_MSG("DB Path: %s", path);

    sqlite3_open(path, &db);

    free(resource);
    free(path);

    CreateTable();

    RAND_bytes(salt, 8);
    RAND_bytes(iv, 16);
    salt[8] = 0x00;
    iv[16] = 0x00;

    PrepareToSQL(salt);
    PrepareToSQL(iv);

    // Storing Encrypted Data
    char *ShortMsg = "Short Msg.";
    EncryptMsg(ShortMsg, decrypted_out, password, salt, iv);

    ret = InsertRecord(decrypted_out, 1, 0, strlen(ShortMsg));
    if (ret) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "Insert Record MessageError\n");
        return;
    }

    time_t rawtime;
    struct tm *timeinfo;
    time(&rawtime);
    timeinfo = localtime(&rawtime);
    sprintf(text, "%s %s", hellomsg, asctime(timeinfo));

    PRINT_MSG("Original text %s", text);
    ret = InsertMessage((unsigned char *)text);
    if (ret) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "Insert ENCRYPTED MessageError\n");
        return;
    }

    ShowRecords();

    PRINT_MSG("Decoded:");
    DecryptRecords();

    sqlite3_close(db);

    free(decoding_table);
    decoding_table = NULL;
}
