/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "extract_gzip.h"
#include "main.h"

/* Extract one file from a gzip file */
int do_extract_currentfile_gz(gzFile uf, const int popt_extract_without_path, int popt_overwrite,
                              const char *gzfilename)
{
    char *ex = strrchr(gzfilename, '.');

    if (ex)
        ex[0] = '\0';

    const char *filename_ingzip = gzfilename;
    char *filename_withoutpath = NULL;
    int err = Z_OK;
    FILE *fout = NULL;
    void *buf = NULL;
    uInt size_buf;

    size_buf = WRITEBUFFERSIZE;
    buf = (void *)malloc(size_buf);

    if (buf == NULL) {
        dlog_print(DLOG_ERROR, LOG_TAG, "Error allocating memory\n");
        return UNZ_INTERNALERROR;
    }

    char *p = filename_withoutpath = (char *)filename_ingzip;
    dlog_print(DLOG_DEBUG, LOG_TAG, "filename_withoutpath = %s\n", filename_withoutpath);

    while (*p != '\0') {
        if (*p == '/')
            filename_withoutpath = p + 1;

        p++;
    }

    dlog_print(DLOG_ERROR, LOG_TAG, "filename_withoutpath = %s\n", filename_withoutpath);

    if ((*filename_withoutpath) == '\0') {
        if ((popt_extract_without_path) == 0)
            mkdir(filename_ingzip, 0775);
    } else {
        const char *write_filename;
        int skip = 0;

        if ((popt_extract_without_path) == 0)
            write_filename = filename_ingzip;
        else
            write_filename = filename_withoutpath;

        if ((popt_overwrite == 0) && (err == Z_OK) && check_exist_file(write_filename))
            skip = 1;

        if ((skip == 0) && (err == Z_OK)) {
            if (popt_overwrite == 2)
                fout = fopen(write_filename, "ab");
            else
                fout = fopen(write_filename, "wb");

            /* some gzipfiles don't contain directory alone before file */
            if ((fout == NULL) && ((popt_extract_without_path) == 0) &&
                    (filename_withoutpath != (char *)filename_ingzip)) {
                char c = *(filename_withoutpath - 1);
                *(filename_withoutpath - 1) = '\0';
                makedir(write_filename);
                *(filename_withoutpath - 1) = c;

                if (popt_overwrite == 2)
                    fout = fopen(write_filename, "ab");
                else
                    fout = fopen(write_filename, "wb");
            }

            if (fout == NULL)
                dlog_print(DLOG_ERROR, LOG_TAG, "Error opening %s\n", write_filename);
        }

        if (fout != NULL) {
            do {
                err = gzread(uf, buf, size_buf);

                if (err < 0) {
                    dlog_print(DLOG_ERROR, LOG_TAG, "error %d with gzipfile in gzread\n", err);
                    break;
                }

                if (err > 0)
                    if (fwrite(buf, err, 1, fout) != 1) {
                        dlog_print(DLOG_ERROR, LOG_TAG, "error in writing extracted file\n");
                        err = UNZ_ERRNO;
                        break;
                    }
            } while (err > 0);

            if (fout)
                fclose(fout);
        }
    }

    free(buf);
    return err;
}

/* Extract gzip */
int do_extract_gz(gzFile uf, int opt_extract_without_path, int opt_overwrite,
                  const char *gzfilename)
{
    int err;

    err = do_extract_currentfile_gz(uf, opt_extract_without_path, opt_overwrite, gzfilename);

    if (err != Z_OK)
        dlog_print(DLOG_ERROR, LOG_TAG, "Failed to extract current file: %d \n", err);

    return 0;
}
