/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdio.h>
#include "extract_zip.h"
#include "main.h"

/* Extract one file
 * unzFile uf;
 * const int *popt_extract_without_path;
 * int *popt_overwrite;
 * const char *format;
 */
int do_extract_currentfile(unzFile uf, const int popt_extract_without_path, int popt_overwrite)
{
    char filename_inzip[256];
    char *filename_withoutpath;
    int err = UNZ_OK;
    FILE *fout = NULL;
    void *buf;
    unsigned size_buf;
    unz_file_info64 file_info;

    err =
        unzGetCurrentFileInfo64(uf, &file_info, filename_inzip, sizeof(filename_inzip), NULL, 0,
                                NULL, 0);

    if (err != UNZ_OK) {
        dlog_print(DLOG_ERROR, LOG_TAG, "Get Info about the current file failed\n");
        return err;
    }

    size_buf = WRITEBUFFERSIZE;
    buf = (void *)malloc(size_buf);

    if (buf == NULL) {
        dlog_print(DLOG_ERROR, LOG_TAG, "Error allocating memory\n");
        return UNZ_INTERNALERROR;
    }

    char *p = filename_withoutpath = filename_inzip;

    while (*p != '\0') {
        if (*p == '/')
            filename_withoutpath = p + 1;

        p++;
    }

    dlog_print(DLOG_DEBUG, LOG_TAG, "filename_withoutpath = %s\n", filename_withoutpath);

    if ((*filename_withoutpath) == '\0') {
        if ((popt_extract_without_path) == 0)
            mkdir(filename_inzip, 0775);
    } else {
        const char *write_filename;
        int skip = 0;

        if ((popt_extract_without_path) == 0)
            write_filename = filename_inzip;
        else
            write_filename = filename_withoutpath;

        if ((popt_overwrite == 0) && (err == UNZ_OK) && check_exist_file(write_filename))
            skip = 1;

        if ((skip == 0) && (err == UNZ_OK)) {
            if (popt_overwrite == 2)
                fout = fopen(write_filename, "ab");
            else
                fout = fopen(write_filename, "wb");

            /* some zipfiles don't contain directory alone before file */
            if ((fout == NULL) && ((popt_extract_without_path) == 0) &&
                    (filename_withoutpath != (char *)filename_inzip)) {
                char c = *(filename_withoutpath - 1);
                *(filename_withoutpath - 1) = '\0';
                makedir(write_filename);
                *(filename_withoutpath - 1) = c;

                if (popt_overwrite == 2)
                    fout = fopen(write_filename, "ab");
                else
                    fout = fopen(write_filename, "wb");
            }

            if (fout == NULL)
                dlog_print(DLOG_ERROR, LOG_TAG, "Error while opening %s\n", write_filename);
        }

        if (fout != NULL) {
            err = unzOpenCurrentFile(uf);

            if (err != UNZ_OK) {
                printf("could not open file\n");
                unzClose(uf);
            } else
                do {
                    err = unzReadCurrentFile(uf, buf, size_buf);

                    if (err < 0) {
                        dlog_print(DLOG_ERROR, LOG_TAG, "Error while reading zip file: %d\n", err);
                        break;
                    }

                    if (err > 0)
                        if (fwrite(buf, err, 1, fout) != 1) {
                            dlog_print(DLOG_ERROR, LOG_TAG, "Error while writing extracted file\n");
                            err = UNZ_ERRNO;
                            break;
                        }
                } while (err > 0);

            if (fout)
                fclose(fout);

            if (err == 0)
                change_file_date(write_filename, file_info.dosDate, file_info.tmu_date);
        }

        if (err == UNZ_OK) {
            err = unzCloseCurrentFile(uf);

            if (err != Z_OK && err != UNZ_OK)
                dlog_print(DLOG_ERROR, LOG_TAG, "Error while closing extracted file.\n", err);
        } else {
            unzCloseCurrentFile(uf);    /* don't lose the error */
        }
    }

    free(buf);
    return err;
}

/* Extract
 * unzFile uf;
 * int opt_extract_without_path;
 * int opt_overwrite;
 * const char *format;
 */
int do_extract(unzFile uf, int opt_extract_without_path, int opt_overwrite)
{
    uLong i;
    unz_global_info64 gi;
    int err;

    err = unzGetGlobalInfo64(uf, &gi);

    if (err != UNZ_OK)
        printf("error %d with zipfile in unzGetGlobalInfo \n", err);

    for (i = 0; i < gi.number_entry; i++) {
        if (do_extract_currentfile(uf, opt_extract_without_path, opt_overwrite) != UNZ_OK)
            break;

        if ((i + 1) < gi.number_entry) {
            err = unzGoToNextFile(uf);

            if (err != UNZ_OK) {
                dlog_print(DLOG_ERROR, LOG_TAG, "Failed to extract current file: %d \n", err);
                break;
            }
        }
    }

    return 0;
}
