/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __USE_FILE_OFFSET64
#define __USE_FILE_OFFSET64
#endif
#ifndef __USE_LARGEFILE64
#define __USE_LARGEFILE64
#endif
#ifndef _LARGEFILE64_SOURCE
#define _LARGEFILE64_SOURCE
#endif
#ifndef _FILE_OFFSET_BIT
#define _FILE_OFFSET_BIT 64
#endif
#define STORAGE_ID 0

#include "user_callbacks.h"
#include "my_file_selector.h"
#include "main.h"
#include "extract_gzip.h"
#include "extract_zip.h"

// ZIP functions

/*
 * Get the time information associated with a file
 * char *f; // name of file to get info on
 * tm_zip *tmzip; // return value: access, modific. and creation times
 * uLong *dt; // dostime
 */
uLong filetime(char *f, tm_zip *tmzip, uLong *dt)
{
    int ret = 0;
    struct stat s;              // results of stat()
    struct tm *filedate;
    time_t tm_t = 0;

    if (strcmp(f, "-") != 0) {
        char name[MAXFILENAME + 1];
        int len = strlen(f);

        if (len > MAXFILENAME)
            len = MAXFILENAME;

        strncpy(name, f, MAXFILENAME - 1);
        /* strncpy doesnt append the trailing NULL, of the string is too long. */
        name[MAXFILENAME] = '\0';

        if (name[len - 1] == '/')
            name[len - 1] = '\0';

        /* not all systems allow stat'ing a file with / appended */
        if (stat(name, &s) == 0) {
            tm_t = s.st_mtime;
            ret = 1;
        }
    }

    filedate = localtime(&tm_t);

    tmzip->tm_sec = filedate->tm_sec;
    tmzip->tm_min = filedate->tm_min;
    tmzip->tm_hour = filedate->tm_hour;
    tmzip->tm_mday = filedate->tm_mday;
    tmzip->tm_mon = filedate->tm_mon;
    tmzip->tm_year = filedate->tm_year;

    return ret;
}

/* Check if a zip64 extended information block should be added to the local file header */
int isLargeFile(const char *filename)
{
    struct stat data;

    if (stat(filename, &data))
        return data.st_size;

    return 0;
}

/* Check if file exists */
int check_exist_file(const char *filename)
{
    FILE *ftestexist;
    int ret = 1;
    ftestexist = fopen(filename, "rb");

    if (ftestexist == NULL) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "File %s doesn't exist.\n", filename);
        ret = 0;
    } else {
        fseek(ftestexist, 0, SEEK_END);
        ftell(ftestexist);
        rewind(ftestexist);
        fclose(ftestexist);
    }

    return ret;
}

// UNZIP functions

/*
 * Set the time information associated with a file
 * const char *filename; // the filename of the file where date/time must be modified
 * uLong dosdate; // the new date at the MSDos format (4 bytes)
 * tm_unz tmu_date; // the SAME new date at the tm_unz format
 */
void change_file_date(const char *filename, uLong dosdate, tm_unz tmu_date)
{
    struct utimbuf ut;
    struct tm newdate;
    newdate.tm_sec = tmu_date.tm_sec;
    newdate.tm_min = tmu_date.tm_min;
    newdate.tm_hour = tmu_date.tm_hour;
    newdate.tm_mday = tmu_date.tm_mday;
    newdate.tm_mon = tmu_date.tm_mon;

    if (tmu_date.tm_year > 1900)
        newdate.tm_year = tmu_date.tm_year - 1900;
    else
        newdate.tm_year = tmu_date.tm_year;

    newdate.tm_isdst = -1;

    ut.actime = ut.modtime = mktime(&newdate);
    utime(filename, &ut);
}

/* Make a direcory */
int makedir(const char *newdir)
{
    if (newdir == NULL) {
        dlog_print(DLOG_ERROR, LOG_TAG, "newdir is empty");
        return 0;
    }

    char *buffer;
    char *p;
    int len = (int)strlen(newdir);

    if (len <= 0)
        return 0;

    buffer = (char *)malloc(len);

    if (buffer == NULL) {
        dlog_print(DLOG_ERROR, LOG_TAG, "Error allocating memory\n");
        return UNZ_INTERNALERROR;
    }

    strncpy(buffer, newdir, len);

    if (buffer[len - 1] == '/')
        buffer[len - 1] = '\0';

    if (mkdir(buffer, 0777) == 0) {
        free(buffer);
        return 1;
    }

    p = buffer + 1;

    while (1) {
        char hold;

        while (*p && *p != '\\' && *p != '/')
            p++;

        hold = *p;
        *p = 0;

        if ((mkdir(buffer, 0777) == -1) && (errno == ENOENT)) {
            dlog_print(DLOG_ERROR, LOG_TAG, "Couldn't create directory %s\n", buffer);
            free(buffer);
            return 0;
        }

        if (hold == 0)
            break;

        *p++ = hold;
    }

    free(buffer);
    return 1;
}

// APP functions
static void win_delete_request_cb(void *data, Evas_Object *obj, void *event_info)
{
    elm_exit();
}

static Eina_Bool _pop_cb()
{
    elm_exit();
    return EINA_FALSE;
}

static void _dest_button(void *data, Evas_Object *btn, void *ev)
{
    show_file_popup(_fs_selected_dest, true, data);
}

static void _file_button(void *data, Evas_Object *btn, void *ev)
{
    show_file_popup(_fs_selected_file, false, data);
}

static void win_back_cb(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = data;

    if (file_selector_shown())
        hide_file_popup();
    else
        elm_win_lower(ad->win);
}

/* Check if file is not a directory */
char *check_if_file(void *data, const char *file)
{
    struct stat s;
    char *message = NULL;

    if (stat(file, &s) == 0) {
        if (s.st_mode & S_IFDIR) {
            dlog_print(DLOG_ERROR, LOG_TAG, "%s is a directory", file);
            message = "<align=center>Please choose a file, not a directory</align>";
        } else if (s.st_mode & S_IFREG)
            dlog_print(DLOG_DEBUG, LOG_TAG, "%s is a file", file);
        else {
            dlog_print(DLOG_ERROR, LOG_TAG, "%s is not a file nor a directory", file);
            message = "<align=center>Unknown file type</align>";
        }
    } else {
        dlog_print(DLOG_ERROR, LOG_TAG, "Problem to check if %s is file or a directory", file);
        message = "<align=center>Please choose a file</align>";
    }

    return message;
}

void _compress(char *file, char *dest, int opt_overwrite, appdata_s *ad)
{
    char *message;
    dlog_print(DLOG_DEBUG, LOG_TAG, "Compression");
    message = "<align=center>File successfully compressed</align>";
    int err = 0;
    int zipok = 1, gzok = 1;
    char zipfilename[MAXFILENAME];
    char gzfilename[MAXFILENAME];

    bool _format_zip = elm_check_state_get(ad->bt1);    // true - .zip (can be both formats at once)
    bool _format_gz = elm_check_state_get(ad->bt2); // true - .gz

    void *buf = (void *)calloc(1, WRITEBUFFERSIZE);
    char archive_name[MAXFILENAME];
    char *filenameinzip = basename(file);
    snprintf(archive_name, MAXFILENAME, "%s", filenameinzip);
    char *ex = strrchr(archive_name, '.');

    if (ex)
        ex[0] = '\0';

    const char *full_dest = NULL;

    int len = strlen(dest);

    if (dest[len - 1] != '/')
        full_dest = strncat(dest, "/", 1);
    else
        full_dest = dest;

    size_t size_read = 0;

    // This loop runs only once to allow break in case of error
    do {
        if (_format_zip) {      /* compress to .zip file */
            dlog_print(DLOG_DEBUG, LOG_TAG, "archive_name %s.\n", archive_name);
            strncat(archive_name, ".zip", 4);
            snprintf(zipfilename, MAXFILENAME, "%s%s", full_dest, archive_name);
            dlog_print(DLOG_DEBUG, LOG_TAG, "Compress to .zip file %s %s.\n", zipfilename);

            if (opt_overwrite == 2) {
                /* if the file doesn't exist, we won't append file */
                if (check_exist_file(zipfilename) == 0)
                    opt_overwrite = 1;
            } else if (opt_overwrite == 0)
                if (check_exist_file(zipfilename) != 0) {
                    if (!elm_check_state_get(ad->bt3))
                        zipok = 0;
                    else
                        opt_overwrite = 1;
                }

            if (zipok == 1) {
                int errclose;

                zipFile zf = zipOpen64(zipfilename, (opt_overwrite == 2) ? 2 : 0);

                if (zf == NULL) {
                    dlog_print(DLOG_ERROR, LOG_TAG, "Error while opening %s\n", zipfilename);
                    return;
                }

                dlog_print(DLOG_DEBUG, LOG_TAG, "Creating %s\n", zipfilename);

                FILE *fin = NULL;
                zip_fileinfo zi;

                zi.tmz_date.tm_sec = zi.tmz_date.tm_min = zi.tmz_date.tm_hour =
                                         zi.tmz_date.tm_mday = zi.tmz_date.tm_mon = zi.tmz_date.tm_year = 0;
                zi.dosDate = 0;
                zi.internal_fa = 0;
                zi.external_fa = 0;

                filetime(filenameinzip, &zi.tmz_date, &zi.dosDate);

                int zip64 = isLargeFile(filenameinzip);
                const char *savefilenameinzip = filenameinzip;

                err = zipOpenNewFileInZip3_64(zf, savefilenameinzip, &zi,
                                              NULL, 0, NULL, 0, NULL,
                                              Z_DEFLATED, Z_DEFAULT_COMPRESSION, 0,
                                              -MAX_WBITS, DEF_MEM_LEVEL, Z_DEFAULT_STRATEGY,
                                              NULL, 0, zip64);

                if (err != ZIP_OK) {
                    dlog_print(DLOG_ERROR, LOG_TAG, "Error while opening %s in .zip file.n",
                               savefilenameinzip);
                    message = "<align=center>Error while opening file in .zip file</align>";
                } else {
                    fin = fopen(file, "rb");

                    if (fin == NULL) {
                        err = ZIP_ERRNO;
                        dlog_print(DLOG_ERROR, LOG_TAG, "Error while opening %s for reading", file);
                        message = "<align=center>Error while opening file in .zip file for reading</align>";
                    } else
                        do {
                            size_read = fread(buf, 1, WRITEBUFFERSIZE, fin);
                            err = zipWriteInFileInZip(zf, buf, size_read);

                            if (err < 0) {
                                dlog_print(DLOG_ERROR, LOG_TAG,
                                           "Error while writing %s to the .zip file.",
                                           savefilenameinzip);
                                message = "<align=center>Error while writing file to .zip file</align>";
                            }
                        } while ((err == ZIP_OK) && (size_read > 0) && !feof(fin));

                    if (fin)
                        fclose(fin);

                    if (err > 0) {
                        err = zipCloseFileInZip(zf);

                        if (err != ZIP_OK)
                            dlog_print(DLOG_ERROR, LOG_TAG,
                                       "Error while closing %s in the .zip file\n", filenameinzip);
                    }

                    errclose = zipClose(zf, NULL);

                    if (errclose != ZIP_OK)
                        dlog_print(DLOG_ERROR, LOG_TAG, "Error while closing %s.\n", filenameinzip);
                }
            }
        }

        if (_format_gz) {       /* compress to .gz file */
            char *ex = strrchr(archive_name, '.');

            if (ex)
                ex[0] = '\0';

            dlog_print(DLOG_DEBUG, LOG_TAG, "archive_name %s.\n", archive_name);
            strncat(archive_name, ".gz", 3);
            snprintf(gzfilename, MAXFILENAME, "%s%s", full_dest, archive_name);
            dlog_print(DLOG_DEBUG, LOG_TAG, "Compress to .gz file %s %s.\n", gzfilename);

            if (opt_overwrite == 2) {
                /* if the file don't exist, we not append file */
                if (check_exist_file(gzfilename) == 0)
                    opt_overwrite = 1;
            } else if (opt_overwrite == 0) {
                if (check_exist_file(gzfilename) != 0) {
                    if (!elm_check_state_get(ad->bt3))
                        gzok = 0;
                    else
                        opt_overwrite = 1;
                }
            }

            if (gzok == 1) {
                char *opt = NULL;

                if (opt_overwrite == 2)
                    opt = "ab"; // append
                else
                    opt = "wb"; // write

                gzFile gf = gzopen(gzfilename, opt);

                if (gf == NULL) {
                    dlog_print(DLOG_ERROR, LOG_TAG, "Error while opening %s\n", gzfilename);
                    break;
                } else
                    dlog_print(DLOG_DEBUG, LOG_TAG, "Creating %s.\n", gzfilename);

                err = gzbuffer(gf, WRITEBUFFERSIZE);

                if (err != 0) {
                    dlog_print(DLOG_ERROR, LOG_TAG, "Error allocating memory.\n");
                    break;
                }

                err = gzsetparams(gf, Z_DEFAULT_COMPRESSION, Z_DEFAULT_STRATEGY);

                if (err != 0) {
                    dlog_print(DLOG_ERROR, LOG_TAG,
                               "Cannot update the compression level and strategy");
                }

                FILE *fin;
                zip_fileinfo zi;

                zi.tmz_date.tm_sec = zi.tmz_date.tm_min = zi.tmz_date.tm_hour =
                                         zi.tmz_date.tm_mday = zi.tmz_date.tm_mon = zi.tmz_date.tm_year = 0;
                zi.dosDate = 0;
                zi.internal_fa = 0;
                zi.external_fa = 0;

                check_exist_file(file);

                fin = fopen64(file, "rb");

                if (fin == NULL)
                    fin = fopen(file, "rb");

                if (fin == NULL) {
                    dlog_print(DLOG_ERROR, LOG_TAG, "Error while opening %s for reading.\n", file);
                } else
                    do {
                        size_read = fread(buf, 1, WRITEBUFFERSIZE, fin);
                        err = gzwrite(gf, buf, size_read);

                        if (err <= 0) {
                            dlog_print(DLOG_ERROR, LOG_TAG,
                                       "Error while writing %s to the .gz file.\n", file);
                            message = "<align=center>Error while writing file to .gz file</align>";
                        }
                    } while ((err != 0) && (size_read > 0) && !feof(fin));

                if (fin)
                    fclose(fin);

                err = gzclose(gf);

                if (err != Z_OK)
                    dlog_print(DLOG_ERROR, LOG_TAG, "Error while closing %s.\n", gzfilename);
            }
        }
    } while (false);

    free(buf);
    show_popup(ad, "Result", message);
}

void _extract(const char *file, const char *dest, int opt_overwrite, appdata_s *ad,
              compression_type_e compression_type)
{
    int err = 0;
    dlog_print(DLOG_DEBUG, LOG_TAG, "Extraction");
    char filename_try[MAXFILENAME + 16] = "";
    char *message = "<align=center>File successfully extracted</align>";
    int opt_extractdir = 1;
    const char *dirname = dest;

    do {
        if (compression_type == ZIP) {
            unzFile uf = NULL;

            snprintf(filename_try, MAXFILENAME, "%s", file);
            /* strncpy doesnt append the trailing NULL, of the string is too long. */

            uf = unzOpen64(file);

            if (uf == NULL) {
                strncat(filename_try, ".zip", 4);
                uf = unzOpen(filename_try);
            }

            if (uf == NULL) {
                dlog_print(DLOG_ERROR, LOG_TAG, "Cannot open %s and %s.\n", file, filename_try);
                break;
            }

            dlog_print(DLOG_DEBUG, LOG_TAG, "%s opened.\n", filename_try);

            if (opt_extractdir && chdir(dirname)) {
                dlog_print(DLOG_ERROR, LOG_TAG, "Error changing into %s, aborting\n", dirname);
                return;
            }

            do_extract(uf, 0, opt_overwrite);
            err = unzClose(uf);

            if (err != UNZ_OK) {
                dlog_print(DLOG_ERROR, LOG_TAG, "Error while closing %s.\n", file);
                break;
            }
        } else {
            gzFile gf = NULL;

            strncpy(filename_try, file, MAXFILENAME - 1);
            /* strncpy doesnt append the trailing NULL, of the string is too long. */
            filename_try[MAXFILENAME] = '\0';

            gf = gzopen(file, "rb");

            if (gf == NULL) {
                strncat(filename_try, ".gz", 3);
                gf = gzopen(filename_try, "rb");
            }

            if (gf == NULL) {
                dlog_print(DLOG_ERROR, LOG_TAG, "Cannot open %s and %s.\n", file, filename_try);
                break;
            }

            dlog_print(DLOG_DEBUG, LOG_TAG, "%s opened\n", filename_try);

            if (opt_extractdir && chdir(dirname)) {
                dlog_print(DLOG_ERROR, LOG_TAG, "Error changing into %s, aborting\n", dirname);
                return;
            }

            do_extract_gz(gf, 0, opt_overwrite, file);
            err = gzclose(gf);

            if (err != Z_OK) {
                dlog_print(DLOG_ERROR, LOG_TAG, "Error while closing %s.", file);
                break;
            }
        }

        show_popup(ad, "Result", message);
    } while (false);
}

/* Create fileselector display to choose file and destination */
void _convert_file(appdata_s *ad, char *ext)
{
    char *file = (char *)elm_object_text_get(ad->lab_file);
    char *dest = (char *)elm_object_text_get(ad->lab_destination);
    dlog_print(DLOG_DEBUG, LOG_TAG, "Convert %s to %s", file, dest);
    dlog_print(DLOG_DEBUG, LOG_TAG, "file = %s", file);
    dlog_print(DLOG_DEBUG, LOG_TAG, "dest = %s", dest);
    int opt_overwrite = 0;
    int err = 0;

    if (elm_check_state_get(ad->bt3))
        opt_overwrite = 1;

    if (elm_check_state_get(ad->bt4))
        opt_overwrite = 2;

    err = makedir(dest);
    dlog_print(DLOG_DEBUG, LOG_TAG, "Destination created = %d", err);

    // Get extension of a file
    dlog_print(DLOG_DEBUG, LOG_TAG, "file %s", file);

    if (ext != NULL && !strcmp(ext + 1, "zip")) // extract
        _extract(file, dest, opt_overwrite, ad, ZIP);
    else if (ext != NULL && !strcmp(ext + 1, "gz")) // extract
        _extract(file, dest, opt_overwrite, ad, GZ);
    else                        // compress
        _compress(file, dest, opt_overwrite, ad);
}

static void _path_to_destination_init(void *data)
{
    appdata_s *ad = data;
    int ret = storage_get_directory(STORAGE_ID, STORAGE_DIRECTORY_DOCUMENTS, &ad->destination_folder_path);
    if (STORAGE_ERROR_NONE != ret) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "storage_get_directory() failed: %s ", ret);
    }
    dlog_print(DLOG_DEBUG, LOG_TAG, "destination url: %s ", ad->destination_folder_path);
}

static void _path_to_media_init(void *data)
{
    appdata_s *ad = data;

    int destination_folder_path_len = strlen(ad->destination_folder_path);
    ad->media_folder_path = (char*)malloc(destination_folder_path_len+1);
    strncpy(ad->media_folder_path, ad->destination_folder_path, destination_folder_path_len);
    char *documents_part = strrchr(ad->destination_folder_path, '/');
    if (documents_part != NULL) {
        int documents_part_len = strlen(documents_part);
        ad->media_folder_path[destination_folder_path_len - documents_part_len] = '\0';
    }
    dlog_print(DLOG_DEBUG, LOG_TAG, "media url: %s ", ad->media_folder_path);
}

static void create_base_gui(appdata_s *ad)
{
    /* Window */
    ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
    elm_win_conformant_set(ad->win, EINA_TRUE);
    elm_win_autodel_set(ad->win, EINA_TRUE);
    elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
    elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
    evas_object_smart_callback_add(ad->win, "delete,request", win_delete_request_cb, NULL);

    /* Create conformant */
    Evas_Object *conform = elm_conformant_add(ad->win);
    evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    elm_win_resize_object_add(ad->win, conform);
    evas_object_show(conform);

    // Create naviframe
    ad->navi = elm_naviframe_add(conform);
    elm_object_part_content_set(conform, "elm.swallow.content", ad->navi);

    evas_object_size_hint_align_set(ad->navi, EVAS_HINT_FILL, EVAS_HINT_FILL);
    evas_object_size_hint_weight_set(ad->navi, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    elm_object_content_set(conform, ad->navi);
    evas_object_show(ad->navi);

    ad->parent_list = elm_list_add(ad->win);
    elm_scroller_policy_set(ad->parent_list, ELM_SCROLLER_POLICY_OFF, ELM_SCROLLER_POLICY_OFF);
    evas_object_show(ad->parent_list);

    Evas_Object *bt_file = NULL, *bt_destination = NULL, *bt_convert = NULL;
    Elm_Object_Item *item = NULL;

    Evas_Object *scroller = elm_scroller_add(ad->win);
    evas_object_size_hint_weight_set(scroller, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

    item = elm_naviframe_item_push(ad->navi, "ZIP", NULL, NULL, scroller, NULL);
    elm_naviframe_item_pop_cb_set(item, _pop_cb, ad);

    /*
     * Get the path to media
     */
    _path_to_destination_init(ad);
    _path_to_media_init(ad);

    // Create box
    Evas_Object *box = elm_box_add(ad->win);
    elm_object_content_set(scroller, box);
    evas_object_size_hint_align_set(box, EVAS_HINT_FILL, 0);
    evas_object_size_hint_weight_set(box, EVAS_HINT_EXPAND, 0);
    evas_object_show(box);

    bt_file = elm_button_add(ad->win);
    elm_object_text_set(bt_file, "Choose file");
    evas_object_size_hint_align_set(bt_file, EVAS_HINT_FILL, EVAS_HINT_FILL);
    evas_object_size_hint_weight_set(bt_file, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_smart_callback_add(bt_file, "clicked", _file_button, ad);
    evas_object_show(bt_file);

    ad->lab_file = elm_entry_add(ad->win);
    elm_entry_single_line_set(ad->lab_file, EINA_FALSE);
    elm_entry_editable_set(ad->lab_file, EINA_FALSE);
    evas_object_size_hint_align_set(ad->lab_file, EVAS_HINT_FILL, EVAS_HINT_FILL);
    evas_object_size_hint_weight_set(ad->lab_file, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_show(ad->lab_file);

    bt_destination = elm_button_add(ad->win);
    elm_object_text_set(bt_destination, "Choose destination");
    evas_object_size_hint_align_set(bt_destination, EVAS_HINT_FILL, EVAS_HINT_FILL);
    evas_object_size_hint_weight_set(bt_destination, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_smart_callback_add(bt_destination, "clicked", _dest_button, ad);
    evas_object_show(bt_destination);

    ad->lab_destination = elm_entry_add(ad->win);
    elm_object_text_set(ad->lab_destination, ad->destination_folder_path);
    elm_entry_single_line_set(ad->lab_destination, EINA_FALSE);
    evas_object_size_hint_align_set(ad->lab_destination, EVAS_HINT_FILL, EVAS_HINT_FILL);
    evas_object_size_hint_weight_set(ad->lab_destination, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    elm_entry_editable_set(ad->lab_destination, EINA_FALSE);
    evas_object_show(ad->lab_destination);

    // compression format
    ad->bt1 = elm_check_add(ad->win);
    elm_object_text_set(ad->bt1, ".zip");
    evas_object_size_hint_align_set(ad->bt1, EVAS_HINT_FILL, 0);
    evas_object_size_hint_weight_set(ad->bt1, EVAS_HINT_EXPAND, 0);
    evas_object_show(ad->bt1);
    evas_object_smart_callback_add(ad->bt1, "changed", _check_format_cb, ad);
    elm_check_state_set(ad->bt1, EINA_TRUE);

    ad->bt2 = elm_check_add(ad->win);
    elm_object_text_set(ad->bt2, ".gz");
    evas_object_size_hint_align_set(ad->bt2, EVAS_HINT_FILL, 0);
    evas_object_size_hint_weight_set(ad->bt2, EVAS_HINT_EXPAND, 0);
    evas_object_show(ad->bt2);
    evas_object_smart_callback_add(ad->bt2, "changed", _check_format_cb, ad);

    // overwrite
    ad->bt3 = elm_check_add(ad->win);
    elm_object_text_set(ad->bt3, "overwrite");
    evas_object_size_hint_align_set(ad->bt3, EVAS_HINT_FILL, 0);
    evas_object_size_hint_weight_set(ad->bt3, EVAS_HINT_EXPAND, 0);
    evas_object_show(ad->bt3);
    evas_object_smart_callback_add(ad->bt3, "changed", _check_overwrite_cb, ad);

    // append
    ad->bt4 = elm_check_add(ad->win);
    elm_object_text_set(ad->bt4, "append");
    evas_object_size_hint_align_set(ad->bt4, EVAS_HINT_FILL, 0);
    evas_object_size_hint_weight_set(ad->bt4, EVAS_HINT_EXPAND, 0);
    evas_object_show(ad->bt4);
    evas_object_smart_callback_add(ad->bt4, "changed", _check_append_cb, ad);

    bt_convert = elm_button_add(ad->win);
    elm_object_text_set(bt_convert, "Compress / Extract");
    evas_object_size_hint_align_set(bt_convert, EVAS_HINT_FILL, EVAS_HINT_FILL);
    evas_object_size_hint_weight_set(bt_convert, 0, EVAS_HINT_EXPAND);
    evas_object_smart_callback_add(bt_convert, "clicked", _convert_cb, ad);
    evas_object_show(bt_convert);

    // Create file selector
    create_file_popup(ad->win);

    elm_box_pack_end(box, bt_file);
    elm_box_pack_end(box, ad->lab_file);
    elm_box_pack_end(box, bt_destination);
    elm_box_pack_end(box, ad->lab_destination);
    elm_box_pack_end(box, ad->lab_zip);
    elm_box_pack_end(box, ad->bt1);
    elm_box_pack_end(box, ad->lab_gz);
    elm_box_pack_end(box, ad->bt2);
    elm_box_pack_end(box, ad->bt3);
    elm_box_pack_end(box, ad->bt4);
    elm_box_pack_end(box, bt_convert);

    eext_object_event_callback_add(ad->win, EEXT_CALLBACK_BACK, win_back_cb, (void *)ad);
    evas_object_show(ad->win);
}

static bool app_create(void *data)
{
    /* Hook to take necessary actions before main event loop starts
       Initialize UI resources and application's data
       If this function returns true, the main loop of application starts
       If this function returns false, the application is terminated */
    appdata_s *ad = data;

    ad->destination_folder_path = NULL;
    ad->media_folder_path = NULL;

    create_base_gui(ad);

    return true;
}

static void app_terminate(void *data)
{
    appdata_s *ad = data;
    free(ad->destination_folder_path);
    free(ad->media_folder_path);
}

int main(int argc, char *argv[])
{
    appdata_s ad;
    memset(&ad, 0x00, sizeof(appdata_s));

    ui_app_lifecycle_callback_s event_callback;
    memset(&event_callback, 0x00, sizeof(ui_app_lifecycle_callback_s));

    event_callback.create = app_create;
    event_callback.terminate = app_terminate;

    int ret = ui_app_main(argc, argv, &event_callback, &ad);
    if (ret != APP_ERROR_NONE)
        dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() failed with error %d", ret);

    return ret;
}
