/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <storage.h>

#include "my_file_selector.h"

#define BUFLEN 256

// Callback function invoked after selecting ".." to get to the parent directory.
void _fs_back_file_cb(void *data, Evas_Object *obj, void *event_info)
{
    Elm_Object_Item *list_item = (Elm_Object_Item *)event_info;
    elm_list_item_selected_set(list_item, EINA_FALSE);

    // Get the last reachable directory for selecting image file.
    char *directory = NULL;
    storage_get_root_directory(0, &directory);

    // If the last symbol of the directory path is "/" then remove it.
    int len = strlen(file_selector_data.dir_path);

    if (file_selector_data.dir_path[len - 1] == '/')
        memset(file_selector_data.dir_path + len - 1, '\0', BUFLEN - len - 1);

    // Set maximal achievable parent directory to media storage.
    int is_max_parent =
        strncmp(file_selector_data.dir_path, directory, strlen(file_selector_data.dir_path));
    free(directory);

    if (!is_max_parent)
        return;

    // Find the last occurrence of the "/" symbol in the directory path.
    int last_slash = strrchr(file_selector_data.dir_path, '/') - file_selector_data.dir_path;

    if (last_slash <= 0)
        return;

    // Clear the list with files and directories.
    elm_list_clear(obj);

    // Change the directory path to the parent directory.
    file_selector_data.dir_path[last_slash] = '\0';

    // Leave only relative path
    char *relative_path = strstr(file_selector_data.dir_path, "media") + strlen("media");

    if (relative_path[0] == '/')
        relative_path++;

    elm_object_text_set(file_selector_data.folder_path, relative_path);

    // Fill the list with files and directories found in the parent directory.
    fill_file_list(obj);
}

void _fs_selected_dest_cb(void *data, Evas_Object *obj, void *event_info)
{
    // Get the path to the file selected in the popup.
    char full_path[BUFLEN];

    // data contains the full path to the selected file
    const char *file_name = data;
    snprintf(full_path, BUFLEN, "%s/%s", file_selector_data.dir_path, file_name);

    // Hide the popup
    evas_object_hide(file_selector_data.file_popup);

    // Invoke function used when the file was selected.
    (*file_selector_data.selected_callback) (full_path, file_selector_data.extra_data);
}

// Callback function invoked after selecting a file in the file selecting popup.
void _fs_selected_file_cb(void *data, Evas_Object *obj, void *event_info)
{
    Elm_Object_Item *list_item = (Elm_Object_Item *)event_info;
    elm_list_item_selected_set(list_item, EINA_FALSE);

    // Get the path to the file selected in the popup.
    char full_path[BUFLEN];

    /*
     * event_info contains the full path to the selected file or NULL
     * if none was selected (or cancel was pressed).
     */
    const char *file_name = elm_object_item_text_get(event_info);
    snprintf(full_path, BUFLEN, "%s/%s", file_selector_data.dir_path, file_name);

    // Hide popup
    dlog_print(DLOG_INFO, LOG_TAG, "Selected file: %s.", file_name);
    evas_object_hide(file_selector_data.file_popup);

    // Invoke function used when the file was selected.
    (*file_selector_data.selected_callback) (full_path, file_selector_data.extra_data);
}

/*
 * Function used for comparing two files/directories names during list creation.
 * Directories has the priority, they are placed higher on the list. Files are placed
 * after all directories.
 */
int _compare_fun_cb(const void *data1, const void *data2)
{
    Elm_Object_Item *item1 = (Elm_Object_Item *)data1;
    Elm_Object_Item *item2 = (Elm_Object_Item *)data2;

    const char *string1 = elm_object_item_part_text_get(item1, NULL);
    const char *string2 = elm_object_item_part_text_get(item2, NULL);

    int len1 = strlen(string1);
    int len2 = strlen(string2);

    const char *last1 = string1 + len1 - 1;
    const char *last2 = string2 + len2 - 1;

    if (*last1 == '/') {
        if (*last2 == '/')
            return strcmp(string1, string2);
        else
            return -1;
    } else if (*last2 == '/')
        return 1;
    else
        return strcmp(string1, string2);
}

static bool dir_selected = false;
void job_selected(void *data)
{
    if (dir_selected) {
        dir_selected = false;
        return;
    }

    Evas_Object *obj = (Evas_Object *)data;
    Elm_Object_Item *item = elm_list_selected_item_get(obj);
    const char *file_name = elm_object_item_text_get(item);
    elm_list_item_selected_set(item, EINA_FALSE);

    // Create a full path to the selected directory.
    char full_path[BUFLEN];
    snprintf(full_path, BUFLEN, "%s/%s", file_selector_data.dir_path, file_name);

    snprintf(file_selector_data.dir_path, BUFLEN, "%s", full_path);
    dlog_print(DLOG_INFO, LOG_TAG, "%s", file_selector_data.dir_path);

    // Fill the list with the files and directories contained in the selected directory.
    elm_list_clear(obj);
    fill_file_list(obj);

    int len = strlen(file_selector_data.dir_path);

    if (file_selector_data.dir_path[len - 1] == '/')
        memset(file_selector_data.dir_path + len - 1, '\0', BUFLEN - len - 1);

    elm_object_text_set(file_selector_data.folder_path,
                        strstr(file_selector_data.dir_path, "media") + strlen("media/"));
}

// Callback function invoked after selecting a directory in the file selecting popup.
void _fs_to_selected_dir_cb(void *data, Evas_Object *obj, void *event_info)
{
   ecore_job_add(job_selected, obj);
}

// Callback function invoked after pressing "Select" button of a directory in the file selecting popup.
Evas_Event_Flags _fs_select_dir_cb(void *data, void *event_info)
{
    dir_selected = true;

    Elm_Object_Item *item = evas_object_data_get(data, "item");

    const char *file_name = elm_object_item_text_get(item);
    _fs_selected_dest_cb((void *)file_name, NULL, NULL);

    return EVAS_EVENT_FLAG_ON_HOLD;
}

// Fill the given list with the files and directories contained in given folder.
void fill_file_list(Evas_Object *file_list)
{
    // Open the directory stream to get its files and directories.
    DIR *const pDir = opendir(file_selector_data.dir_path);

    // Find all files and directories contained in the current directory.
    struct dirent ent_struct;
    struct dirent *ent = NULL;

    elm_list_select_mode_set(file_list, ELM_OBJECT_SELECT_MODE_ALWAYS);

    while ((readdir_r(pDir, &ent_struct, &ent) == 0) && ent) {
        char *ent_d_name_duplicate = strdup(ent->d_name);
        if (ent->d_type == DT_DIR && (strncmp(ent->d_name, ".", 1) != 0)) {
            char *ent_d_name_duplicate_with_slash = strncat(ent_d_name_duplicate, "/", 1);
            // Add directory to the list.
            if (file_selector_data.folder_selection) {
                Evas_Object *ok_button = elm_button_add(file_list);
                elm_object_text_set(ok_button, "Select");
                elm_object_style_set(ok_button, "anchor");

                Elm_Object_Item *item =
                    elm_list_item_sorted_insert(file_list, ent_d_name_duplicate_with_slash, NULL,
                                                ok_button, _fs_to_selected_dir_cb, NULL,
                                                _compare_fun_cb);

                Evas_Object *g = elm_gesture_layer_add(ok_button);
                evas_object_data_set(ok_button, "item", item);
                elm_gesture_layer_attach(g, ok_button);
                elm_gesture_layer_cb_set(g, ELM_GESTURE_N_TAPS,
                                         ELM_GESTURE_STATE_END, _fs_select_dir_cb, ok_button);
            } else
                elm_list_item_sorted_insert(file_list, ent_d_name_duplicate_with_slash, NULL, NULL,
                                            _fs_to_selected_dir_cb, NULL, _compare_fun_cb);

        } else if (ent->d_type == DT_REG && !file_selector_data.folder_selection) {
            // Add file to the list.
            elm_list_item_sorted_insert(file_list, ent_d_name_duplicate, NULL, NULL,
                                        _fs_selected_file_cb, NULL, _compare_fun_cb);
        }
        free(ent_d_name_duplicate);
    }

    // Add ".." button to enable navigating to the parent directories.
    elm_list_item_prepend(file_list, "..", NULL, NULL, _fs_back_file_cb, NULL);

    closedir(pDir);
}

// Create a popup for choosing a file.
void create_file_popup(Evas_Object *file_popup)
{
    // Create an element inside window for displaying list of files.
    file_selector_data.file_popup = elm_win_inwin_add(file_popup);
    elm_layout_theme_set(file_selector_data.file_popup, NULL, NULL, "minimal");

    // Create a box for the file list.
    Evas_Object *box = elm_box_add(file_selector_data.file_popup);
    evas_object_size_hint_weight_set(box, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(box, EVAS_HINT_FILL, EVAS_HINT_FILL);
    elm_win_inwin_content_set(file_selector_data.file_popup, box);
    elm_box_padding_set(box, 0.0, 0.0);
    evas_object_show(box);

    // Get the initial directory for selecting file.
    char *directory = NULL;
    storage_get_root_directory(0, &directory);
    snprintf(file_selector_data.dir_path, BUFLEN, "%s", directory);
    free(directory);

    file_selector_data.folder_path = elm_label_add(box);
    elm_object_text_set(file_selector_data.folder_path, "");
    elm_box_pack_start(box, file_selector_data.folder_path);
    evas_object_show(file_selector_data.folder_path);

    // Create a list used for displaying files and directories found inside the current directory.
    file_selector_data.file_list = elm_list_add(file_selector_data.file_popup);
    evas_object_size_hint_weight_set(file_selector_data.file_list, EVAS_HINT_EXPAND,
                                     EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(file_selector_data.file_list, EVAS_HINT_FILL, EVAS_HINT_FILL);
    elm_box_padding_set(box, 0.0, 0.0);
    elm_box_pack_end(box, file_selector_data.file_list);
    evas_object_show(file_selector_data.file_list);

    // Fill the list with the files and directories found inside the current directory.
    fill_file_list(file_selector_data.file_list);
}

void show_file_popup(file_selector_selected_cb selected_callback, bool folder_selection, void *data)
{
    if (folder_selection != file_selector_data.folder_selection) {
        file_selector_data.folder_selection = folder_selection;
        elm_list_clear(file_selector_data.file_list);
        fill_file_list(file_selector_data.file_list);
    }

    file_selector_data.selected_callback = selected_callback;
    file_selector_data.extra_data = data;
    evas_object_show(file_selector_data.file_popup);
}

void hide_file_popup()
{
    evas_object_hide(file_selector_data.file_popup);
}

Eina_Bool file_selector_shown()
{
    return evas_object_visible_get(file_selector_data.file_popup);
}
