/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "user_callbacks.h"
#include "my_file_selector.h"
#include <stdio.h>
#include <app_common.h>

void show_popup(void *data, const char *title, const char *message)
{
    appdata_s *ad = (appdata_s *)data;
    Evas_Object *popup, *content;
    content = elm_label_add(ad->win);
    elm_object_text_set(content, message);
    popup = elm_popup_add(ad->win);
    elm_popup_timeout_set(popup, 2.0);
    evas_object_smart_callback_add(popup, "timeout", _timeout, NULL);
    elm_object_content_set(popup, content);
    elm_object_part_text_set(popup, "title,text", title);
    evas_object_show(popup);
    evas_object_smart_callback_add(popup, "block,clicked", _block_clicked, NULL);
}

void _block_clicked(void *data, Evas_Object *obj, void *event_info)
{
    evas_object_hide(obj);
}

void _timeout(void *data, Evas_Object *obj, void *event_info)
{
    evas_object_hide(obj);
}

void _choose_destination_cb(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = (appdata_s *)data;
    const char *title = "Error";
    const char *message = NULL;
    /* show popup if file to convert has not been choosen */
    const char *file_name = elm_object_text_get(ad->lab_file);

    if (!file_name) {
        message = "<align=center>No file chosen</align>";
        show_popup(ad, title, message);
        return;
    }

    message = check_if_file(ad, file_name);

    if (message) {
        show_popup(ad, title, message);
        return;
    }
}

void _convert_cb(void *data, Evas_Object *obj, void *event_info)
{
    // DO compress / extract

    appdata_s *ad = (appdata_s *)data;
    const char *title = "Error";
    char *message;
    /* show popup if file to convert or destination has not been choosen */

    const char *file_name = elm_object_text_get(ad->lab_file);
    const char *dest_name = elm_object_text_get(ad->lab_destination);
    int dest_name_len = strlen(dest_name);
    free(ad->destination_folder_path);
    ad->destination_folder_path = (char*)calloc(1, dest_name_len);
    strncpy(ad->destination_folder_path, dest_name, dest_name_len);

    int path_to_media_len = strlen(ad->media_folder_path);
    char *destination_media_root_path = (char*)calloc(1, path_to_media_len+1);
    strncpy(destination_media_root_path, dest_name, path_to_media_len);
    int destination_media_root_path_len = strlen(destination_media_root_path);
    if (destination_media_root_path_len == 0) {
        free(ad->destination_folder_path);
        ad->destination_folder_path = (char*)calloc(1, path_to_media_len);
        strncpy(ad->destination_folder_path, ad->media_folder_path, path_to_media_len);
        elm_object_text_set(ad->lab_destination, ad->destination_folder_path);
        strncpy(destination_media_root_path, ad->destination_folder_path, path_to_media_len);
        show_popup(ad, title, message);
    }
    if (strcmp(destination_media_root_path, ad->media_folder_path)) {
        message = "<align=center>Wrong destination path<br>It should be inside media folder path</align>";
        show_popup(ad, title, message);
        free(destination_media_root_path);
        return;
    }
    free(destination_media_root_path);

    if (!file_name) {
        message = "<align=center>Set the file and the destination</align>";
        show_popup(ad, title, message);
        return;
    }

    message = check_if_file(ad, file_name);

    if (message) {
        show_popup(ad, title, message);
        return;
    }

    // Get extension of the choosen file
    char *ext = strrchr(file_name, '.');

    if (ext == NULL) {
        zipFile zf = NULL;
        gzFile gf = NULL;
        FILE *f = NULL;

        f = fopen(file_name, "rb");

        if (f == NULL) {
            zf = zipOpen64(file_name, 0);

            if (zf == NULL) {
                gf = gzopen(file_name, "rb");

                if (gf == NULL) {
                    const char *title = "Error";
                    const char *message = "<align=center>Invalid file format</align>";
                    show_popup(ad, title, message);
                    return;
                } else {
                    ext = ".gz";
                    gzclose(gf);
                }
            } else {
                ext = ".zip";
                zipClose(zf, NULL);
            }
        } else {
            ext = ".";
            fclose(f);
        }
    }

    _convert_file(ad, ext);
}

/* Callback invoked after select a file/directory in fileselector */
void _fs_selected_file(const char *file_path, void *data)
{
    appdata_s *ad = (appdata_s *)data;

    elm_object_text_set(ad->lab_file, file_path);
    /*
     * event_info contains the full path of the selected file or NULL
     * if none is selected (or cancel is pressed)
     */
    char *ext = NULL;

    if (file_path)
        ext = strrchr(file_path, '.');

    if (ext != NULL && (!strcmp(ext + 1, "zip") || !strcmp(ext + 1, "gz"))) {
        // hide checkbox for unzipping
        evas_object_hide(ad->lab_zip);
        evas_object_hide(ad->lab_gz);
        evas_object_hide(ad->bt1);
        evas_object_hide(ad->bt2);
    } else {
        // show checkbox for zipping
        evas_object_show(ad->lab_zip);
        evas_object_show(ad->lab_gz);
        evas_object_show(ad->bt1);
        evas_object_show(ad->bt2);
    }

    /* event_info contains the full path of the selected file */
    dlog_print(DLOG_INFO, LOG_TAG, "There's been a selection: %s", file_path);

    if (file_path)
        elm_entry_editable_set(ad->lab_destination, EINA_TRUE);
    else
        elm_entry_editable_set(ad->lab_destination, EINA_FALSE);

    hide_file_popup();
}

/* Callback invoked after select a file/directory in fileselector */
void _fs_selected_dest(const char *file_path, void *data)
{
    appdata_s *ad = (appdata_s *)data;
    char destination_name[MAXFILENAME];
    snprintf(destination_name, MAXFILENAME, "%s", elm_object_text_get(ad->lab_destination));
    /*
     * event_info contains the full path of the selected file or NULL
     * if none is selected (or cancel is pressed)
     */
    dlog_print(DLOG_INFO, LOG_TAG, "There's been a selection: %s", file_path);

    elm_object_text_set(ad->lab_destination, file_path);
}

/* Callback invoked if format state changed */
void _check_format_cb(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = (appdata_s *)data;
    bool _zip = elm_check_state_get(ad->bt1);
    bool _gzip = elm_check_state_get(ad->bt2);

    /* to avoid that none of formats is selected */
    if (_zip == false && _gzip == false)
        elm_check_state_set(ad->bt1, EINA_TRUE);
}

/* Callback invoked if _overwrite state changed */
void _check_overwrite_cb(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = (appdata_s *)data;
    bool _overwrite = elm_check_state_get(ad->bt3);
    bool _append = elm_check_state_get(ad->bt4);

    /* to avoid that both overwrite and append are true */
    if (_overwrite && _overwrite == _append)
        elm_check_state_set(ad->bt4, !_overwrite);
}

/* Callback invoked if _append state changed */
void _check_append_cb(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = (appdata_s *)data;
    bool _overwrite = elm_check_state_get(ad->bt3);
    bool _append = elm_check_state_get(ad->bt4);

    /* to avoid that both overwriet and append are true */
    if (_append && _overwrite == _append)
        elm_check_state_set(ad->bt3, !_append);
}
