/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "user_callbacks.h"
#include "main.h"

#include <mime_type.h>

void _mime_type_for_file_extension(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    dlog_print(DLOG_DEBUG, LOG_TAG, "in _mime_type_for_file_extension()");

    // Get the MIME type for a given file extension.
    char *mime_type;
    int error = mime_type_get_mime_type("jpg", &mime_type);

    if (MIME_TYPE_ERROR_NONE != error) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "mime_type_get_mime_type() failed, error: %d", error);
        PRINT_MSG("Getting MIME Type for 'jpg' file extension failed");
    } else
        PRINT_MSG("MIME Type for 'jpg' file extension: %s", mime_type);

    PRINT_MSG("");

    // Get the extensions of all files in the shared resources directory and get their MIME types.
    char *shared_path = app_get_shared_resource_path();
    struct dirent *pDirent = NULL;
    DIR *dir = opendir(shared_path);

    if (dir) {
        while ((pDirent = readdir(dir)) != NULL) {
            if (pDirent->d_type != DT_REG)  // Only regular files
                continue;

            // Get the extension from a file name
            char *extension = strrchr(pDirent->d_name, '.');

            if (!extension)
                continue;

            extension++;

            // Get the MIME type.
            error = mime_type_get_mime_type(extension, &mime_type);

            if (MIME_TYPE_ERROR_NONE != error) {
                dlog_print(DLOG_DEBUG, LOG_TAG, "mime_type_get_mime_type() failed, error: %d",
                           error);
                dlog_print(DLOG_DEBUG, LOG_TAG, "Current file: %s", extension);
            } else {
                PRINT_MSG("MIME Type for '%s' file: %s", extension, mime_type);
                PRINT_MSG("");
            }

            free(mime_type);
        }

        closedir(dir);
    } else {
        dlog_print(DLOG_DEBUG, LOG_TAG, "%s directory does not exist", shared_path);
        PRINT_MSG("%s directory does not exist", shared_path);
    }
    free(shared_path);

    PRINT_MSG("");
}

void _extensions_for_mime_type(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    dlog_print(DLOG_DEBUG, LOG_TAG, "in _extensions_for_mime_type()");

    char **extension;
    int length;
    int error = mime_type_get_file_extension("image/jpeg", &extension, &length);

    if (MIME_TYPE_ERROR_NONE != error) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "mime_type_get_file_extension() failed, error: %d", error);
        PRINT_MSG("Getting file extensions for the given 'image/jpeg' MIME type failed");
        return;
    }

    PRINT_MSG("File extensions for 'image/jpeg' MIME type:");
    int i;

    for (i = 0; i < length; i++) {
        PRINT_MSG("%s", extension[i]);
        free(extension[i]);
    }
}
