/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "user_callbacks.h"

#include <audio_io.h>
#include <storage.h>

#define BUFLEN 512
#define SAMPLE_RATE 48000
#define RECORDING_SEC 5         // Defines how many seconds will the recording last.
#define MIN_2BYTES_SIGNED (-32768)
#define MAX_2BYTES_SIGNED 32767

static audio_in_h input = NULL; // Audio input handle declaration
static audio_out_h output = NULL;   // Audio output handle declaration
static void *buffer = NULL;     // Buffer used for audio recording/playback
static int buffer_size;         // Size of the buffer used for audio recording/playback
static FILE *fp_w = NULL;       // File used for asynchronous audio recording
static FILE *fp_r = NULL;       // File used for asynchronous audio playback
static Evas_Object *audio_sync_rec_bt = NULL;
static Evas_Object *audio_sync_play_bt = NULL;
static Evas_Object *audio_sync_vol_bt = NULL;
static Evas_Object *audio_async_rec_bt = NULL;
static Evas_Object *audio_async_play_bt = NULL;
static char *sounds_directory = NULL;
static const char *audio_io_file_name = "pcm_w.raw";
static char audio_io_file_path[BUFLEN];

// Callback function invoked when the "List Error Codes" button is clicked.
static void __audioIO_ecodes(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    PRINT_MSG("AUDIO_IO_ERROR_OUT_OF_MEMORY %d", AUDIO_IO_ERROR_OUT_OF_MEMORY);
    PRINT_MSG("AUDIO_IO_ERROR_INVALID_PARAMETER %d", AUDIO_IO_ERROR_INVALID_PARAMETER);
    PRINT_MSG("AUDIO_IO_ERROR_INVALID_OPERATION %d", AUDIO_IO_ERROR_INVALID_OPERATION);
    PRINT_MSG("AUDIO_IO_ERROR_DEVICE_NOT_OPENED %d", AUDIO_IO_ERROR_DEVICE_NOT_OPENED);
    PRINT_MSG("AUDIO_IO_ERROR_DEVICE_NOT_CLOSED %d", AUDIO_IO_ERROR_DEVICE_NOT_CLOSED);
    PRINT_MSG("AUDIO_IO_ERROR_INVALID_BUFFER %d", AUDIO_IO_ERROR_INVALID_BUFFER);
    PRINT_MSG("AUDIO_IO_ERROR_SOUND_POLICY %d", AUDIO_IO_ERROR_SOUND_POLICY);
}

static bool _storage_cb(int storage_id, storage_type_e type, storage_state_e state,
                        const char *path, void *user_data)
{
    if (STORAGE_TYPE_INTERNAL == type) {
        int *internal_storage_id = (int *)user_data;

        if (NULL != internal_storage_id)
            *internal_storage_id = storage_id;

        // Internal storage found, stop the iteration.
        return false;
    } else {
        // Continue iterating over storages.
        return true;
    }
}

// Callback function invoked for every stored part of the audio.
static void _audio_io_stream_write_cb(audio_out_h handle, size_t nbytes, void *userdata)
{
    char *buff = NULL;

    if (nbytes > 0) {
        buff = malloc(nbytes);
        memset(buff, 0, nbytes);

        // Play the following part of the recording.
        fread(buff, sizeof(char), nbytes, fp_r);

        // Copy the recorded data from the buffer to the output buffer.
        int data_size = audio_out_write(handle, buff, nbytes);

        if (data_size < 0) {
            dlog_print(DLOG_ERROR, LOG_TAG, "audio_out_write() failed with error code: %d",
                       data_size);
        }

        free(buff);
    }
}

// Boolean variable that defines whether the asynchronous playback is ongoing or not.
static bool is_async_play_ongoing = false;

// Callback invoked when the "Asynchronous Playback" button is clicked.
static void __audioIO_cb_playasync(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    int error_code = AUDIO_IO_ERROR_NONE;

    if (!is_async_play_ongoing) {
        // Set a callback function that will be called asynchronously for every single part of the stored voice data.
        error_code = audio_out_set_stream_cb(output, _audio_io_stream_write_cb, NULL);
        if (AUDIO_IO_ERROR_NONE != error_code) {
            dlog_print(DLOG_ERROR, LOG_TAG, "audio_out_set_stream_cb() failed with error code: %d",
                       error_code);
            return;
        }

        // Open a file, that will be used for playback.
        fp_r = fopen(audio_io_file_path, "r");

        // Audio output prepare (starts the hardware playback process)
        error_code = audio_out_prepare(output);
        if (AUDIO_IO_ERROR_NONE != error_code) {
            dlog_print(DLOG_ERROR, LOG_TAG, "audio_out_prepare() failed with error code: %d",
                       error_code);
        } else {
            PRINT_MSG("Asynchronous playback started.");
            is_async_play_ongoing = true;

            // Disable "Asynchronous Recording" button until the playback finishes.
            elm_object_disabled_set(audio_async_rec_bt, EINA_TRUE);
        }

        elm_object_text_set(audio_async_play_bt, "Stop Asynchronous Playback");
    } else {
        // Stop the hardware playback process.
        error_code = audio_out_unprepare(output);
        if (AUDIO_IO_ERROR_NONE != error_code) {
            dlog_print(DLOG_ERROR, LOG_TAG, "audio_out_unprepare() failed with error code: %d",
                       error_code);
        } else {
            PRINT_MSG("Asynchronous playback stopped.");
            is_async_play_ongoing = false;

            // Enable "Asynchronous Recording" button, because the current playback is finished.
            elm_object_disabled_set(audio_async_rec_bt, EINA_FALSE);

            // Unset the callback function used for asynchronous playback process.
            error_code = audio_out_unset_stream_cb(output);
            if (AUDIO_IO_ERROR_NONE != error_code) {
                dlog_print(DLOG_ERROR, LOG_TAG,
                           "audio_out_unset_stream_cb() failed with error code: %d", error_code);
            }

            // Close the file used for playback.
            error_code = fclose(fp_r);
            if (error_code != 0)
                dlog_print(DLOG_ERROR, LOG_TAG, "fclose() failed with error code: %d", error_code);

            elm_object_text_set(audio_async_play_bt, "Start Asynchronous Playback");
        }
    }
}

// Callback invoked for every captured part of the recording.
static void _audio_io_stream_read_cb(audio_in_h handle, size_t nbytes, void *userdata)
{
    const void *buff = NULL;

    if (nbytes > 0) {
        // Retrieve buffer pointer from audio in buffer
        int error_code = audio_in_peek(handle, &buff, &nbytes);
        if (AUDIO_IO_ERROR_NONE != error_code) {
            dlog_print(DLOG_ERROR, LOG_TAG, "audio_in_peek() failed with error code: %d",
                       error_code);
            return;
        }

        // Store the recorded part in the file.
        fwrite(buff, sizeof(char), nbytes, fp_w);

        // Remove the obtained audio input data from the actual stream buff.
        error_code = audio_in_drop(handle);
        if (AUDIO_IO_ERROR_NONE != error_code) {
            dlog_print(DLOG_ERROR, LOG_TAG, "audio_in_drop() failed with error code: %d",
                       error_code);
        }
    }
}

// Boolean variable that defines whether the asynchronous recording is ongoing or not.
static bool is_async_rec_ongoing = false;

// Callback invoked when the "Asynchronous Recording" button is clicked.
static void __audioIO_cb_recordasync(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    int error_code = AUDIO_IO_ERROR_NONE;

    if (!is_async_rec_ongoing) {
        // Set a callback function that will be called asynchronously for every single part of the captured voice data.
        error_code = audio_in_set_stream_cb(input, _audio_io_stream_read_cb, NULL);
        if (AUDIO_IO_ERROR_NONE != error_code) {
            dlog_print(DLOG_ERROR, LOG_TAG, "audio_in_get_sample_type() failed with error code: %d",
                       error_code);
            return;
        }

        // Open a file, where the recorded data will be stored.
        fp_w = fopen(audio_io_file_path, "w");

        if (fp_w)
            PRINT_MSG("Recording stored in %s file.", audio_io_file_path);
        else {
            dlog_print(DLOG_ERROR, LOG_TAG, "fopen() function failed while opening %s file!",
                       audio_io_file_path);
            return;
        }

        // Audio input prepare (starts the hardware recording process).
        error_code = audio_in_prepare(input);
        if (AUDIO_IO_ERROR_NONE != error_code) {
            dlog_print(DLOG_ERROR, LOG_TAG, "audio_in_prepare() failed with error code: %d",
                       error_code);
        } else {
            PRINT_MSG("Asynchronous recording started.");
            is_async_rec_ongoing = true;

            // Disable buttons until the asynchronous recording finishes.
            elm_object_disabled_set(audio_async_play_bt, EINA_TRUE);
            elm_object_disabled_set(audio_sync_rec_bt, EINA_TRUE);
        }

        elm_object_text_set(audio_async_rec_bt, "Stop Asynchronous Recording");
    } else {
        // Stop the hardware recording process.
        error_code = audio_in_unprepare(input);
        if (AUDIO_IO_ERROR_NONE != error_code) {
            dlog_print(DLOG_ERROR, LOG_TAG, "audio_in_unprepare() failed with error code: %d",
                       error_code);
        } else {
            PRINT_MSG("Asynchronous recording stopped.");
            is_async_rec_ongoing = false;

            // Enable buttons, because the current asynchronous recording is finished.
            elm_object_disabled_set(audio_async_play_bt, EINA_FALSE);
            elm_object_disabled_set(audio_sync_rec_bt, EINA_FALSE);

            // Unset the callback function used for asynchronous recording process.
            error_code = audio_in_unset_stream_cb(input);
            if (AUDIO_IO_ERROR_NONE != error_code) {
                dlog_print(DLOG_ERROR, LOG_TAG,
                           "audio_in_unset_stream_cb() failed with error code: %d", error_code);
            }

            // Close the file used for recording.
            error_code = fclose(fp_w);
            if (error_code != 0)
                dlog_print(DLOG_ERROR, LOG_TAG, "fclose() failed with error code: %d", error_code);

            elm_object_text_set(audio_async_rec_bt, "Start Asynchronous Recording");
        }
    }
}

// Callback invoked when the "Sync Recording Volume Up" button is clicked.
static void __audioIO_cb_modify(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    // Get the sample type of the input.
    audio_sample_type_e sample_type;

    int error_code = audio_in_get_sample_type(input, &sample_type);
    if (AUDIO_IO_ERROR_NONE != error_code) {
        dlog_print(DLOG_ERROR, LOG_TAG, "audio_in_get_sample_type() failed with error code: %d",
                   error_code);
        return;
    }

    uint8_t *index = (uint8_t *)buffer;

    while (index < (((uint8_t *)buffer) + buffer_size)) {
        if (AUDIO_SAMPLE_TYPE_S16_LE == sample_type) {
            // We are using int16_t type, because it's 2 bytes long.
            int16_t *value = (int16_t *)index;

            // Make the sample louder.
            int32_t tmp = (*value) * 8;

            if (tmp > MAX_2BYTES_SIGNED)
                tmp = MAX_2BYTES_SIGNED;

            if (tmp < MIN_2BYTES_SIGNED)
                tmp = MIN_2BYTES_SIGNED;

            (*value) = tmp;
        } else {
            // We are using uint8_t type, because it's 1 byte long.
            uint8_t *value = (uint8_t *)index;

            // Make the sample louder.
            uint16_t tmp = (*value) * 8;

            if (tmp > 255)
                tmp = 255;

            (*value) = tmp;
        }

        // Go to next sample.
        index += sample_type == AUDIO_SAMPLE_TYPE_S16_LE ? 2 : 1;
    }

    PRINT_MSG("Volume of the synchronous recording increased.");
}

/*
 * Function that executes after the thread with synchronous playback ends.
 * It is safe to call EFL functions here because this function is called from the main thread.
 */
static void synchronous_playback_ended(void *data, Ecore_Thread *thread)
{
    // Enable buttons, because the current synchronous playback is finished.
    elm_object_disabled_set(audio_sync_play_bt, EINA_FALSE);
    elm_object_disabled_set(audio_sync_rec_bt, EINA_FALSE);

    PRINT_MSG("Synchronous playback ended.");
}

// Function that executes in a different thread, because it launches synchronous playback which is blocking.
static void synchronous_playback(void *data, Ecore_Thread *thread)
{
    // Audio output prepare (starts the hardware playback process).
    int error_code = audio_out_prepare(output);
    if (AUDIO_IO_ERROR_NONE != error_code) {
        dlog_print(DLOG_ERROR, LOG_TAG, "audio_out_prepare() failed with error code: %d",
                   error_code);
        return;
    }

    // Copy the recorded data from the local buffer to the output buffer.
    int bytes_number = audio_out_write(output, buffer, buffer_size);

    if (bytes_number < 0)
        dlog_print(DLOG_ERROR, LOG_TAG, "audio_out_write() failed with error code: %d", error_code);
    else {
        dlog_print(DLOG_INFO, LOG_TAG, "Synchronous playback ended.<br>%d bytes have been played.",
                   bytes_number);
    }

    // Stop the hardware playback process.
    error_code = audio_out_unprepare(output);
    if (AUDIO_IO_ERROR_NONE != error_code) {
        dlog_print(DLOG_ERROR, LOG_TAG, "audio_out_unprepare() failed with error code: %d",
                   error_code);
    }
}

// Callback invoked when the "Synchronous Playback" button is clicked.
static void __audioIO_cb_play(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    PRINT_MSG("Synchronous playback starts.");

    // Disable buttons until the synchronous playback finishes.
    elm_object_disabled_set(audio_sync_play_bt, EINA_TRUE);
    elm_object_disabled_set(audio_sync_rec_bt, EINA_TRUE);

    // Launch synchronous playback in a different thread to prevent blocking the application main loop.
    ecore_thread_run(synchronous_playback, synchronous_playback_ended, NULL, NULL);
}

/*
 * Function that executes after the thread with synchronous recording ends.
 * It is safe to call EFL functions here because this function is called from the main thread.
 */
static void synchronous_recording_ended(void *data, Ecore_Thread *thread)
{
    // Enable buttons, because the current synchronous recording is finished.
    elm_object_disabled_set(audio_sync_rec_bt, EINA_FALSE);
    elm_object_disabled_set(audio_sync_play_bt, EINA_FALSE);
    elm_object_disabled_set(audio_sync_vol_bt, EINA_FALSE);
    elm_object_disabled_set(audio_async_rec_bt, EINA_FALSE);

    PRINT_MSG("Synchronous recording ended.");
}

// Function that executes in a different thread, because it launches synchronous recording which is blocking.
static void synchronous_recording(void *data, Ecore_Thread *thread)
{
    // Audio input prepare (starts the hardware recording process)
    int error_code = audio_in_prepare(input);
    if (AUDIO_IO_ERROR_NONE != error_code) {
        dlog_print(DLOG_ERROR, LOG_TAG, "audio_in_prepare() failed with error code: %d",
                   error_code);
        return;
    }

    // Copy recorded data from the input buffer to the local buffer.
    int bytes_number = audio_in_read(input, buffer, buffer_size);

    if (bytes_number < 0)
        dlog_print(DLOG_ERROR, LOG_TAG, "audio_in_read() failed with error code: %d", bytes_number);
    else {
        dlog_print(DLOG_INFO, LOG_TAG,
                   "Synchronous recording ended.<br>%d bytes have been recorded.", bytes_number);
    }

    // Stop the hardware recording process.
    error_code = audio_in_unprepare(input);
    if (AUDIO_IO_ERROR_NONE != error_code) {
        dlog_print(DLOG_ERROR, LOG_TAG, "audio_in_unprepare() failed with error code: %d",
                   error_code);
    }
}

// Callback invoked when the "Synchronous Recording" button is clicked.
static void __audioIO_cb_record(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    PRINT_MSG("Synchronous recording starts.");

    // Disable buttons until the synchronous recording finishes.
    elm_object_disabled_set(audio_sync_rec_bt, EINA_TRUE);
    elm_object_disabled_set(audio_sync_play_bt, EINA_TRUE);
    elm_object_disabled_set(audio_sync_vol_bt, EINA_TRUE);
    elm_object_disabled_set(audio_async_rec_bt, EINA_TRUE);

    // Launch synchronous recording in a different thread to prevent blocking the application's main loop.
    ecore_thread_run(synchronous_recording, synchronous_recording_ended, NULL, NULL);
}

// Callback function invoked when the "Audio I/O" screen is being closed.
static Eina_Bool audio_pop_cb(void *data, Elm_Object_Item *item)
{
    // Release the memory allocated for the buffer used for recording/playing.
    free(buffer);

    // Stop the hardware recording process.
    audio_in_unprepare(input);

    // Unset the callback function used for asynchronous recording process.
    audio_in_unset_stream_cb(input);

    // Deinitialize audio input device.
    audio_in_destroy(input);

    // Stop the hardware playback process.
    audio_out_unprepare(output);

    // Unset the callback function used for asynchronous playback process.
    audio_out_unset_stream_cb(output);

    // Deinitialize audio output device.
    audio_out_destroy(output);

    // Free the Sounds directory path.
    free(sounds_directory);

    return _pop_cb(data, item);
}

// Audio I/O menu creation.
void create_buttons_in_main_window(appdata_s *ad)
{
    // Create the window for the Audio I/O.
    Evas_Object *display = _create_new_cd_display(ad, "Audio I/O", audio_pop_cb);

    // Create buttons for the Audio I/O.
    audio_sync_rec_bt = _new_button(ad, display, "Synchronous Recording", __audioIO_cb_record);
    audio_sync_play_bt = _new_button(ad, display, "Synchronous Playback", __audioIO_cb_play);
    audio_sync_vol_bt = _new_button(ad, display, "Sync Recording Volume Up", __audioIO_cb_modify);
    audio_async_rec_bt =
        _new_button(ad, display, "Start Asynchronous Recording", __audioIO_cb_recordasync);
    audio_async_play_bt =
        _new_button(ad, display, "Start Asynchronous Playback", __audioIO_cb_playasync);
    _new_button(ad, display, "List Error Codes", __audioIO_ecodes);

    // Disable playback buttons, because there is no recording yet.
    elm_object_disabled_set(audio_sync_play_bt, EINA_TRUE);
    elm_object_disabled_set(audio_sync_vol_bt, EINA_TRUE);
    elm_object_disabled_set(audio_async_play_bt, EINA_TRUE);

    // Audio input device initialization.
    int error_code =
        audio_in_create(SAMPLE_RATE, AUDIO_CHANNEL_STEREO, AUDIO_SAMPLE_TYPE_S16_LE, &input);
    if (AUDIO_IO_ERROR_NONE != error_code) {
        dlog_print(DLOG_ERROR, LOG_TAG, "audio_in_create() failed with error code: %d", error_code);
        return;
    }

    // Audio output device initialization.
    error_code =
        audio_out_create(SAMPLE_RATE, AUDIO_CHANNEL_STEREO, AUDIO_SAMPLE_TYPE_S16_LE,
                         SOUND_TYPE_MEDIA, &output);
    if (AUDIO_IO_ERROR_NONE != error_code) {
        dlog_print(DLOG_ERROR, LOG_TAG, "audio_out_create() failed with error code: %d",
                   error_code);
        return;
    }

    // Get the size required for the allocation of the buffer used for recording/playback. This function returns the size recommended by the sound server.
    error_code = audio_in_get_buffer_size(input, &buffer_size);
    if (AUDIO_IO_ERROR_NONE != error_code) {
        dlog_print(DLOG_ERROR, LOG_TAG, "audio_in_get_buffer_size() failed with error code: %d",
                   error_code);
        return;
    }

    // Multiply the buffer size by the number of seconds that the recording will take and by "10" (because normally audio_in_get_buffer_size() returns the size for around 100 milliseconds).
    buffer_size *= 10 * RECORDING_SEC;

    // Allocate the memory for the buffer used for recording/playback.
    buffer = malloc(buffer_size);

    // Get the path to the Sounds directory:

    // 1. Get internal storage id.
    int internal_storage_id = -1;

    error_code = storage_foreach_device_supported(_storage_cb, &internal_storage_id);
    if (STORAGE_ERROR_NONE != error_code) {
        dlog_print(DLOG_ERROR, LOG_TAG,
                   "storage_foreach_device_supported() failed! Error code = %d", error_code);
        return;
    }

    // 2. Get the path to the Sounds directory.
    error_code =
        storage_get_directory(internal_storage_id, STORAGE_DIRECTORY_SOUNDS, &sounds_directory);
    if (STORAGE_ERROR_NONE != error_code) {
        dlog_print(DLOG_ERROR, LOG_TAG, "storage_get_directory() failed with error code: %d",
                   error_code);
        return;
    }

    // Prepare a path to the file used for asynchronous playback.
    snprintf(audio_io_file_path, BUFLEN, "%s/%s", sounds_directory, audio_io_file_name);
}
