/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "user_callbacks.h"
#include <tone_player.h>

static Evas_Object *tone_start_bt = NULL;
static Evas_Object *tone_stop_bt = NULL;
static Evas_Object *tone_type_next_bt = NULL;
static Evas_Object *tone_sound_next_bt = NULL;
static int tone_player_id = -1;
static tone_type_e tone_type = TONE_TYPE_DEFAULT;
static sound_type_e sound_type = SOUND_TYPE_SYSTEM;

static const char *_sound_type_to_string(sound_type_e sound_type)
{
    switch (sound_type) {
    case SOUND_TYPE_SYSTEM:
        return "SOUND_TYPE_SYSTEM";

    case SOUND_TYPE_NOTIFICATION:
        return "SOUND_TYPE_NOTIFICATION";

    case SOUND_TYPE_ALARM:
        return "SOUND_TYPE_ALARM";

    case SOUND_TYPE_RINGTONE:
        return "SOUND_TYPE_RINGTONE";

    case SOUND_TYPE_MEDIA:
        return "SOUND_TYPE_MEDIA";

    case SOUND_TYPE_CALL:
        return "SOUND_TYPE_CALL";

    case SOUND_TYPE_VOIP:
        return "SOUND_TYPE_VOIP";

    case SOUND_TYPE_VOICE:
        return "SOUND_TYPE_VOICE";

    default:
        return "Unknown";
    }
}

static const char *_tone_type_to_string(tone_type_e tone_type)
{
    switch (tone_type) {
    case TONE_TYPE_DTMF_0:
        return "TONE_TYPE_DTMF_0";

    case TONE_TYPE_DTMF_1:
        return "TONE_TYPE_DTMF_1";

    case TONE_TYPE_DTMF_2:
        return "TONE_TYPE_DTMF_2";

    case TONE_TYPE_DTMF_3:
        return "TONE_TYPE_DTMF_3";

    case TONE_TYPE_DTMF_4:
        return "TONE_TYPE_DTMF_4";

    case TONE_TYPE_DTMF_5:
        return "TONE_TYPE_DTMF_5";

    case TONE_TYPE_DTMF_6:
        return "TONE_TYPE_DTMF_6";

    case TONE_TYPE_DTMF_7:
        return "TONE_TYPE_DTMF_7";

    case TONE_TYPE_DTMF_8:
        return "TONE_TYPE_DTMF_8";

    case TONE_TYPE_DTMF_9:
        return "TONE_TYPE_DTMF_9";

    case TONE_TYPE_DTMF_S:
        return "TONE_TYPE_DTMF_S";

    case TONE_TYPE_DTMF_P:
        return "TONE_TYPE_DTMF_P";

    case TONE_TYPE_DTMF_A:
        return "TONE_TYPE_DTMF_A";

    case TONE_TYPE_DTMF_B:
        return "TONE_TYPE_DTMF_B";

    case TONE_TYPE_DTMF_C:
        return "TONE_TYPE_DTMF_C";

    case TONE_TYPE_DTMF_D:
        return "TONE_TYPE_DTMF_D";

    case TONE_TYPE_SUP_DIAL:
        return "TONE_TYPE_SUP_DIAL";

    case TONE_TYPE_ANSI_DIAL:
        return "TONE_TYPE_ANSI_DIAL";

    case TONE_TYPE_JAPAN_DIAL:
        return "TONE_TYPE_JAPAN_DIAL";

    case TONE_TYPE_SUP_BUSY:
        return "TONE_TYPE_SUP_BUSY";

    case TONE_TYPE_ANSI_BUSY:
        return "TONE_TYPE_ANSI_BUSY";

    case TONE_TYPE_JAPAN_BUSY:
        return "TONE_TYPE_JAPAN_BUSY";

    case TONE_TYPE_SUP_CONGESTION:
        return "TONE_TYPE_SUP_CONGESTION";

    case TONE_TYPE_ANSI_CONGESTION:
        return "TONE_TYPE_ANSI_CONGESTION";

    case TONE_TYPE_SUP_RADIO_ACK:
        return "TONE_TYPE_SUP_RADIO_ACK";

    case TONE_TYPE_JAPAN_RADIO_ACK:
        return "TONE_TYPE_JAPAN_RADIO_ACK";

    case TONE_TYPE_SUP_RADIO_NOTAVAIL:
        return "TONE_TYPE_SUP_RADIO_NOTAVAIL";

    case TONE_TYPE_SUP_ERROR:
        return "TONE_TYPE_SUP_ERROR";

    case TONE_TYPE_SUP_CALL_WAITING:
        return "TONE_TYPE_SUP_CALL_WAITING";

    case TONE_TYPE_ANSI_CALL_WAITING:
        return "TONE_TYPE_ANSI_CALL_WAITING";

    case TONE_TYPE_SUP_RINGTONE:
        return "TONE_TYPE_SUP_RINGTONE";

    case TONE_TYPE_ANSI_RINGTONE:
        return "TONE_TYPE_ANSI_RINGTONE";

    case TONE_TYPE_PROP_BEEP:
        return "TONE_TYPE_PROP_BEEP";

    case TONE_TYPE_PROP_ACK:
        return "TONE_TYPE_PROP_ACK";

    case TONE_TYPE_PROP_NACK:
        return "TONE_TYPE_PROP_NACK";

    case TONE_TYPE_PROP_PROMPT:
        return "TONE_TYPE_PROP_PROMPT";

    case TONE_TYPE_PROP_BEEP2:
        return "TONE_TYPE_PROP_BEEP2";

    case TONE_TYPE_SUP_INTERCEPT:
        return "TONE_TYPE_SUP_INTERCEPT";

    case TONE_TYPE_SUP_INTERCEPT_ABBREV:
        return "TONE_TYPE_SUP_INTERCEPT_ABBREV";

    case TONE_TYPE_SUP_CONGESTION_ABBREV:
        return "TONE_TYPE_SUP_CONGESTION_ABBREV";

    case TONE_TYPE_SUP_CONFIRM:
        return "TONE_TYPE_SUP_CONFIRM";

    case TONE_TYPE_SUP_PIP:
        return "TONE_TYPE_SUP_PIP";

    case TONE_TYPE_CDMA_DIAL_TONE_LITE:
        return "TONE_TYPE_CDMA_DIAL_TONE_LITE";

    case TONE_TYPE_CDMA_NETWORK_USA_RINGBACK:
        return "TONE_TYPE_CDMA_NETWORK_USA_RINGBACK";

    case TONE_TYPE_CDMA_INTERCEPT:
        return "TONE_TYPE_CDMA_INTERCEPT";

    case TONE_TYPE_CDMA_ABBR_INTERCEPT:
        return "TONE_TYPE_CDMA_ABBR_INTERCEPT";

    case TONE_TYPE_CDMA_REORDER:
        return "TONE_TYPE_CDMA_REORDER";

    case TONE_TYPE_CDMA_ABBR_REORDER:
        return "TONE_TYPE_CDMA_ABBR_REORDER";

    case TONE_TYPE_CDMA_NETWORK_BUSY:
        return "TONE_TYPE_CDMA_NETWORK_BUSY";

    case TONE_TYPE_CDMA_CONFIRM:
        return "TONE_TYPE_CDMA_CONFIRM";

    case TONE_TYPE_CDMA_ANSWER:
        return "TONE_TYPE_CDMA_ANSWER";

    case TONE_TYPE_CDMA_NETWORK_CALLWAITING:
        return "TONE_TYPE_CDMA_NETWORK_CALLWAITING";

    case TONE_TYPE_CDMA_PIP:
        return "TONE_TYPE_CDMA_PIP";

    case TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_NORMAL:
        return "TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_NORMAL";

    case TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_INTERGROUP:
        return "TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_INTERGROUP";

    case TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_SP_PRI:
        return "TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_SP_PRI";

    case TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_PAT3:
        return "TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_PAT3";

    case TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_PING_RING:
        return "TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_PING_RING";

    case TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_PAT5:
        return "TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_PAT5";

    case TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_PAT6:
        return "TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_PAT6";

    case TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_PAT7:
        return "TONE_TYPE_CDMA_CALL_SIGNAL_ISDN_PAT7";

    case TONE_TYPE_CDMA_HIGH_L:
        return "TONE_TYPE_CDMA_HIGH_L";

    case TONE_TYPE_CDMA_MED_L:
        return "TONE_TYPE_CDMA_MED_L";

    case TONE_TYPE_CDMA_LOW_L:
        return "TONE_TYPE_CDMA_LOW_L";

    case TONE_TYPE_CDMA_HIGH_SS:
        return "TONE_TYPE_CDMA_HIGH_SS";

    case TONE_TYPE_CDMA_MED_SS:
        return "TONE_TYPE_CDMA_MED_SS";

    case TONE_TYPE_CDMA_LOW_SS:
        return "TONE_TYPE_CDMA_LOW_SS";

    case TONE_TYPE_CDMA_HIGH_SSL:
        return "TONE_TYPE_CDMA_HIGH_SSL";

    case TONE_TYPE_CDMA_MED_SSL:
        return "TONE_TYPE_CDMA_MED_SSL";

    case TONE_TYPE_CDMA_LOW_SSL:
        return "TONE_TYPE_CDMA_LOW_SSL";

    case TONE_TYPE_CDMA_HIGH_SS_2:
        return "TONE_TYPE_CDMA_HIGH_SS_2";

    case TONE_TYPE_CDMA_MED_SS_2:
        return "TONE_TYPE_CDMA_MED_SS_2";

    case TONE_TYPE_CDMA_LOW_SS_2:
        return "TONE_TYPE_CDMA_LOW_SS_2";

    case TONE_TYPE_CDMA_HIGH_SLS:
        return "TONE_TYPE_CDMA_HIGH_SLS";

    case TONE_TYPE_CDMA_MED_SLS:
        return "TONE_TYPE_CDMA_MED_SLS";

    case TONE_TYPE_CDMA_LOW_SLS:
        return "TONE_TYPE_CDMA_LOW_SLS";

    case TONE_TYPE_CDMA_HIGH_S_X4:
        return "TONE_TYPE_CDMA_HIGH_S_X4";

    case TONE_TYPE_CDMA_MED_S_X4:
        return "TONE_TYPE_CDMA_MED_S_X4";

    case TONE_TYPE_CDMA_LOW_S_X4:
        return "TONE_TYPE_CDMA_LOW_S_X4";

    case TONE_TYPE_CDMA_HIGH_PBX_L:
        return "TONE_TYPE_CDMA_HIGH_PBX_L";

    case TONE_TYPE_CDMA_MED_PBX_L:
        return "TONE_TYPE_CDMA_MED_PBX_L";

    case TONE_TYPE_CDMA_LOW_PBX_L:
        return "TONE_TYPE_CDMA_LOW_PBX_L";

    case TONE_TYPE_CDMA_HIGH_PBX_SS:
        return "TONE_TYPE_CDMA_HIGH_PBX_SS";

    case TONE_TYPE_CDMA_MED_PBX_SS:
        return "TONE_TYPE_CDMA_MED_PBX_SS";

    case TONE_TYPE_CDMA_LOW_PBX_SS:
        return "TONE_TYPE_CDMA_LOW_PBX_SS";

    case TONE_TYPE_CDMA_HIGH_PBX_SSL:
        return "TONE_TYPE_CDMA_HIGH_PBX_SSL";

    case TONE_TYPE_CDMA_MED_PBX_SSL:
        return "TONE_TYPE_CDMA_MED_PBX_SSL";

    case TONE_TYPE_CDMA_LOW_PBX_SSL:
        return "TONE_TYPE_CDMA_LOW_PBX_SSL";

    case TONE_TYPE_CDMA_HIGH_PBX_SLS:
        return "TONE_TYPE_CDMA_HIGH_PBX_SLS";

    case TONE_TYPE_CDMA_MED_PBX_SLS:
        return "TONE_TYPE_CDMA_MED_PBX_SLS";

    case TONE_TYPE_CDMA_LOW_PBX_SLS:
        return "TONE_TYPE_CDMA_LOW_PBX_SLS";

    case TONE_TYPE_CDMA_HIGH_PBX_S_X4:
        return "TONE_TYPE_CDMA_HIGH_PBX_S_X4";

    case TONE_TYPE_CDMA_MED_PBX_S_X4:
        return "TONE_TYPE_CDMA_MED_PBX_S_X4";

    case TONE_TYPE_CDMA_LOW_PBX_S_X4:
        return "TONE_TYPE_CDMA_LOW_PBX_S_X4";

    case TONE_TYPE_CDMA_ALERT_NETWORK_LITE:
        return "TONE_TYPE_CDMA_ALERT_NETWORK_LITE";

    case TONE_TYPE_CDMA_ALERT_AUTOREDIAL_LITE:
        return "TONE_TYPE_CDMA_ALERT_AUTOREDIAL_LITE";

    case TONE_TYPE_CDMA_ONE_MIN_BEEP:
        return "TONE_TYPE_CDMA_ONE_MIN_BEEP";

    case TONE_TYPE_CDMA_KEYPAD_VOLUME_KEY_LITE:
        return "TONE_TYPE_CDMA_KEYPAD_VOLUME_KEY_LITE";

    case TONE_TYPE_CDMA_PRESSHOLDKEY_LITE:
        return "TONE_TYPE_CDMA_PRESSHOLDKEY_LITE";

    case TONE_TYPE_CDMA_ALERT_INCALL_LITE:
        return "TONE_TYPE_CDMA_ALERT_INCALL_LITE";

    case TONE_TYPE_CDMA_EMERGENCY_RINGBACK:
        return "TONE_TYPE_CDMA_EMERGENCY_RINGBACK";

    case TONE_TYPE_CDMA_ALERT_CALL_GUARD:
        return "TONE_TYPE_CDMA_ALERT_CALL_GUARD";

    case TONE_TYPE_CDMA_SOFT_ERROR_LITE:
        return "TONE_TYPE_CDMA_SOFT_ERROR_LITE";

    case TONE_TYPE_CDMA_CALLDROP_LITE:
        return "TONE_TYPE_CDMA_CALLDROP_LITE";

    case TONE_TYPE_CDMA_NETWORK_BUSY_ONE_SHOT:
        return "TONE_TYPE_CDMA_NETWORK_BUSY_ONE_SHOT";

    case TONE_TYPE_CDMA_ABBR_ALERT:
        return "TONE_TYPE_CDMA_ABBR_ALERT";

    case TONE_TYPE_CDMA_SIGNAL_OFF:
        return "TONE_TYPE_CDMA_SIGNAL_OFF";

    case TONE_TYPE_USER_DEFINED_LOW_FRE:
        return "TONE_TYPE_USER_DEFINED_LOW_FRE";

    case TONE_TYPE_USER_DEFINED_MED_FRE:
        return "TONE_TYPE_USER_DEFINED_MED_FRE";

    case TONE_TYPE_USER_DEFINED_HIGH_FRE:
        return "TONE_TYPE_USER_DEFINED_HIGH_FRE";

    default:
        return "Unknown";
    }
}

// Callback function invoked when the "Stop tone" button is clicked.
static void __toneplayer_cb_stop(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    // Stop playing the tone.
    int error_code = tone_player_stop(tone_player_id);
    if (TONE_PLAYER_ERROR_NONE != error_code)
        dlog_print(DLOG_ERROR, LOG_TAG, "tone_player_stop() failed! Error code = %d", error_code);
    else {
        PRINT_MSG("Tone playback stopped.");

        // Disable the "Stop tone" button when the tone is stopped.
        elm_object_disabled_set(tone_stop_bt, EINA_TRUE);

        // Enable "Start tone" and "Next tone" buttons when the tone is stopped.
        elm_object_disabled_set(tone_start_bt, EINA_FALSE);
    }
}

// Callback function invoked when the "Start tone" button is clicked.
static void __toneplayer_cb_start(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    // Start playing the tone.
    int error_code = tone_player_start(tone_type, sound_type, -1, &tone_player_id);
    if (TONE_PLAYER_ERROR_NONE != error_code)
        dlog_print(DLOG_ERROR, LOG_TAG, "tone_player_start() failed! Error code = %d", error_code);
    else {
        PRINT_MSG("Tone playback started.");

        // Disable the "Start tone" button when the tone is currently being played.
        elm_object_disabled_set(tone_start_bt, EINA_TRUE);

        // Enable the "Stop tone" button when the tone is currently being played.
        elm_object_disabled_set(tone_stop_bt, EINA_FALSE);
    }
}

// Callback function invoked when the "Next tone type" button is clicked.
static void __toneplayer_cb_next_tone(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    // Set new value for the tone type.
    tone_type = tone_type < TONE_TYPE_USER_DEFINED_HIGH_FRE ? ++tone_type : TONE_TYPE_DEFAULT;

    // Stop the current playback process.
    __toneplayer_cb_stop(NULL, NULL, NULL);

    // Start the playback with the new tone type value.
    __toneplayer_cb_start(NULL, NULL, NULL);

    PRINT_MSG("Tone type value set to %s.", _tone_type_to_string(tone_type));
}

// Callback function invoked when the "Next sound type" button is clicked.
static void __toneplayer_cb_next_sound(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    // Set new value for the sound type.
    sound_type = sound_type < SOUND_TYPE_VOICE ? ++sound_type : SOUND_TYPE_SYSTEM;

    // Stop the current playback process.
    __toneplayer_cb_stop(NULL, NULL, NULL);

    // Start the playback with the new sound type value.
    __toneplayer_cb_start(NULL, NULL, NULL);

    PRINT_MSG("Sound type value set to %s.", _sound_type_to_string(sound_type));
}

// Callback invoked when the "Tone Player" screen is closed.
static Eina_Bool tone_player_pop_cb(void *data, Elm_Object_Item *item)
{
    // Stop the current playback process.
    __toneplayer_cb_stop(NULL, NULL, NULL);

    return _pop_cb(data, item);
}

// Tone Player menu creation.
void create_buttons_in_main_window(appdata_s *ad)
{
    // Create the window for the Tone Player.
    Evas_Object *display = _create_new_cd_display(ad, "Tone Player", tone_player_pop_cb);

    // Create buttons for the Tone Player.
    tone_start_bt = _new_button(ad, display, "Start tone", __toneplayer_cb_start);
    tone_stop_bt = _new_button(ad, display, "Stop tone", __toneplayer_cb_stop);
    tone_type_next_bt = _new_button(ad, display, "Next tone type", __toneplayer_cb_next_tone);
    tone_sound_next_bt = _new_button(ad, display, "Next sound type", __toneplayer_cb_next_sound);

    // Disable the "Stop tone" button when no tone is being played.
    elm_object_disabled_set(tone_stop_bt, EINA_TRUE);

    PRINT_MSG
    ("Tone type initial value set to TONE_TYPE_DEFAULT.<br>Sound type initial value set to SOUND_TYPE_SYSTEM.");
}
