/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "user_callbacks.h"
#include "main.h"
#include <glib.h>
#include <storage.h>

#include <sys/stat.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <net/if.h>

#define SIZE 256

Evas_Object *ap_name_entry = NULL;
Evas_Object *password_entry = NULL;
connection_h connection = NULL;

char *_connection_type_to_string(connection_type_e net_state)
{
    switch (net_state) {
    case CONNECTION_TYPE_DISCONNECTED:
        return "disconnected";

    case CONNECTION_TYPE_WIFI:
        return "WiFi";

    case CONNECTION_TYPE_CELLULAR:
        return "cellular";

    case CONNECTION_TYPE_ETHERNET:
        return "ethernet";

    default:
        return "unknown";
    }
}

/* Getting the network connection details */
void _connection_details_cb(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    int error_code;
    connection_type_e net_state;
    char *state = NULL;

    /* Get the type of the current profile for data connection */
    error_code = connection_get_type(connection, &net_state);
    state = _connection_type_to_string(net_state);
    if (error_code == CONNECTION_ERROR_NONE) {
        PRINT_MSG("Network connection type : %s", state);
        dlog_print(DLOG_DEBUG, LOG_TAG, "Network connection type : %s", state);
    }

    if (net_state == CONNECTION_TYPE_DISCONNECTED)
        return;

    /* Get the connection IPv4 address */
    char *ip_addr = NULL;
    error_code = connection_get_ip_address(connection, CONNECTION_ADDRESS_FAMILY_IPV4, &ip_addr);
    if (error_code == CONNECTION_ERROR_NONE) {
        PRINT_MSG("IP address : %s", ip_addr);
        dlog_print(DLOG_DEBUG, LOG_TAG, "IP address : %s", ip_addr);
        free(ip_addr);
    }

    /* Get the connection proxy information */
    char *proxy_addr = NULL;
    error_code = connection_get_proxy(connection, CONNECTION_ADDRESS_FAMILY_IPV4, &proxy_addr);
    if (error_code == CONNECTION_ERROR_NONE) {
        PRINT_MSG("Proxy address : %s", proxy_addr);
        dlog_print(DLOG_DEBUG, LOG_TAG, "Proxy address : %s", proxy_addr);
        free(proxy_addr);
    }
}

/* Getting the connection information */
void _connection_information_cb(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    int error_code = 0;

    /* Get the cellular connection state */
    connection_cellular_state_e cellular_state;
    connection_get_cellular_state(connection, &cellular_state);

    switch (cellular_state) {
    case CONNECTION_CELLULAR_STATE_OUT_OF_SERVICE:
        PRINT_MSG("Connection cellular state: Out of service");
        dlog_print(DLOG_DEBUG, LOG_TAG, "Out of service");
        break;

    case CONNECTION_CELLULAR_STATE_FLIGHT_MODE:
        PRINT_MSG("Connection cellular state: Flight mode");
        dlog_print(DLOG_DEBUG, LOG_TAG, "Flight mode");
        break;

    case CONNECTION_CELLULAR_STATE_ROAMING_OFF:
        PRINT_MSG("Connection cellular state: Roaming is turned off");
        dlog_print(DLOG_DEBUG, LOG_TAG, "Roaming is turned off");
        break;

    case CONNECTION_CELLULAR_STATE_CALL_ONLY_AVAILABLE:
        PRINT_MSG("Connection cellular state: Call only");
        dlog_print(DLOG_DEBUG, LOG_TAG, "Call only");
        break;

    case CONNECTION_CELLULAR_STATE_AVAILABLE:
        PRINT_MSG("Connection cellular state: Available");
        dlog_print(DLOG_DEBUG, LOG_TAG, "Available");
        break;

    case CONNECTION_CELLULAR_STATE_CONNECTED:
        PRINT_MSG("Connection cellular state: Connected");
        dlog_print(DLOG_DEBUG, LOG_TAG, "Connected");
        break;

    default:
        PRINT_MSG("Connection cellular state: error");
        dlog_print(DLOG_DEBUG, LOG_TAG, "error");
        break;
    }

    /* Get the Wi-Fi connection state */
    connection_wifi_state_e wifi_state;
    connection_get_wifi_state(connection, &wifi_state);

    switch (wifi_state) {
    case CONNECTION_WIFI_STATE_DEACTIVATED:
        PRINT_MSG("Connection wifi state: Deactivated");
        dlog_print(DLOG_DEBUG, LOG_TAG, "Deactivated");
        break;

    case CONNECTION_WIFI_STATE_DISCONNECTED:
        PRINT_MSG("Connection wifi state: Disconnected");
        dlog_print(DLOG_DEBUG, LOG_TAG, "Disconnected");
        break;

    case CONNECTION_WIFI_STATE_CONNECTED:
        PRINT_MSG("Connection wifi state: Connected");
        dlog_print(DLOG_DEBUG, LOG_TAG, "Connected");
        break;

    default:
        PRINT_MSG("Connection wifi state: Unavailable");
        dlog_print(DLOG_DEBUG, LOG_TAG, "error");
        break;
    }

    /* Get the connection statistics */
    PRINT_MSG("Cellular details:");
    long long last_received_size;
    error_code =
        connection_get_statistics(connection, CONNECTION_TYPE_CELLULAR,
                                  CONNECTION_STATISTICS_TYPE_LAST_RECEIVED_DATA,
                                  &last_received_size);
    if (error_code != CONNECTION_ERROR_NONE)
        PRINT_MSG("[connection_get_statistics] failed");
    else
        PRINT_MSG(" last_received_size = %lld", last_received_size);

    long long last_sent_size;
    error_code =
        connection_get_statistics(connection, CONNECTION_TYPE_CELLULAR,
                                  CONNECTION_STATISTICS_TYPE_LAST_SENT_DATA, &last_sent_size);
    if (error_code != CONNECTION_ERROR_NONE)
        PRINT_MSG("[connection_get_statistics] failed");
    else
        PRINT_MSG(" last_sent_size = %lld", last_sent_size);

    long long total_received_size;
    error_code =
        connection_get_statistics(connection, CONNECTION_TYPE_CELLULAR,
                                  CONNECTION_STATISTICS_TYPE_TOTAL_RECEIVED_DATA,
                                  &total_received_size);
    if (error_code != CONNECTION_ERROR_NONE)
        PRINT_MSG("[connection_get_statistics] failed");
    else
        PRINT_MSG(" total_received_size = %lld", total_received_size);

    long long total_sent_size;
    error_code =
        connection_get_statistics(connection, CONNECTION_TYPE_CELLULAR,
                                  CONNECTION_STATISTICS_TYPE_TOTAL_SENT_DATA, &total_sent_size);
    if (error_code != CONNECTION_ERROR_NONE)
        PRINT_MSG("[connection_get_statistics] failed");
    else
        PRINT_MSG(" total_sent_size = %lld", total_sent_size);

    PRINT_MSG("Wifi details:");
    error_code =
        connection_get_statistics(connection, CONNECTION_TYPE_WIFI,
                                  CONNECTION_STATISTICS_TYPE_LAST_RECEIVED_DATA,
                                  &last_received_size);
    if (error_code != CONNECTION_ERROR_NONE)
        PRINT_MSG("[connection_get_statistics] failed");
    else
        PRINT_MSG(" last_received_size = %lld", last_received_size);

    error_code =
        connection_get_statistics(connection, CONNECTION_TYPE_WIFI,
                                  CONNECTION_STATISTICS_TYPE_LAST_SENT_DATA, &last_sent_size);
    if (error_code != CONNECTION_ERROR_NONE)
        PRINT_MSG("[connection_get_statistics] failed");
    else
        PRINT_MSG(" last_sent_size = %lld", last_sent_size);

    error_code =
        connection_get_statistics(connection, CONNECTION_TYPE_WIFI,
                                  CONNECTION_STATISTICS_TYPE_TOTAL_RECEIVED_DATA,
                                  &total_received_size);
    if (error_code != CONNECTION_ERROR_NONE)
        PRINT_MSG("[connection_get_statistics] failed");
    else
        PRINT_MSG(" total_received_size = %lld", total_received_size);

    error_code =
        connection_get_statistics(connection, CONNECTION_TYPE_WIFI,
                                  CONNECTION_STATISTICS_TYPE_TOTAL_SENT_DATA, &total_sent_size);
    if (error_code != CONNECTION_ERROR_NONE)
        PRINT_MSG("[connection_get_statistics] failed");
    else
        PRINT_MSG(" total_sent_size = %lld", total_sent_size);
}

/* Called if IP changed */
static void __ip_changed_cb(const char *ipv4_address, const char *ipv6_address, void *user_data)
{
    PRINT_MSG("%s callback, IPv4 address : %s, IPv6 address : %s", (char *)user_data,
              ipv4_address, (ipv6_address ? ipv6_address : "NULL"));
    dlog_print(DLOG_DEBUG, LOG_TAG, "%s callback, IPv4 address : %s, IPv6 address : %s",
               (char *)user_data, ipv4_address, (ipv6_address ? ipv6_address : "NULL"));
}

/* Called if proxy changed */
static void __proxy_changed_cb(const char *ipv4_address, const char *ipv6_address, void *user_data)
{
    PRINT_MSG("%s callback, IPv4 address : %s, IPv6 address : %s", (char *)user_data,
              ipv4_address, (ipv6_address ? ipv6_address : "NULL"));
    dlog_print(DLOG_DEBUG, LOG_TAG, "%s callback, IPv4 address : %s, IPv6 address : %s",
               (char *)user_data, ipv4_address, (ipv6_address ? ipv6_address : "NULL"));
}

/* Register the Property Change Callbacks */
void _connection_property_cb(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    int error_code;

    error_code =
        connection_set_ip_address_changed_cb(connection, __ip_changed_cb, "IP addr changed:");
    if (error_code != CONNECTION_ERROR_NONE)
        PRINT_MSG("[connection_set_ip_address_changed_cb] failed");

    error_code =
        connection_set_proxy_address_changed_cb(connection, __proxy_changed_cb,
                "Proxy IP addr changed:");
    if (error_code != CONNECTION_ERROR_NONE)
        PRINT_MSG("[connection_set_proxy_address_changed_cb] failed");

    PRINT_MSG("Waiting for ip or proxy change.");
}

static void _server_thread(void *data, Ecore_Thread *thread)
{
    appdata_s *ad = (appdata_s *)data;
    dlog_print(DLOG_DEBUG, LOG_TAG, "_server_thread() begun");
    int sockfd, newsockfd, portno = 8080;
    socklen_t clilen;
    char buffer[SIZE];
    struct sockaddr_in serv_addr, cli_addr;
    int n;
    sockfd = socket(AF_INET, SOCK_STREAM, 0);

    if (sockfd < 0) {
        dlog_print(DLOG_ERROR, LOG_TAG, "socket() failed = %s", strerror(errno));
        PRINT_MSG("socket() failed =  %s", strerror(errno));
    }

    bzero((char *)&serv_addr, sizeof(serv_addr));
    serv_addr.sin_family = AF_INET;
    serv_addr.sin_addr.s_addr = INADDR_ANY;
    serv_addr.sin_port = htons(portno);
    int option = 1;

    if (setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, (char *)&option, sizeof(option)) < 0) {
        dlog_print(DLOG_ERROR, LOG_TAG, "setsockopt() failed = %s", strerror(errno));
        PRINT_MSG("setsockopt() failed = %s", strerror(errno));
    }

    if (bind(sockfd, (struct sockaddr *)&serv_addr, sizeof(serv_addr)) < 0) {
        dlog_print(DLOG_ERROR, LOG_TAG, "bind() failed = %s", strerror(errno));
        PRINT_MSG("bind() failed = %s", strerror(errno));
    }

    listen(sockfd, 5);
    clilen = sizeof(cli_addr);

    /* Accept all connections until the app is terminated */
    ad->server_running = true;

    while (ad->server_running) {
        newsockfd = accept(sockfd, (struct sockaddr *)&cli_addr, &clilen);

        if (newsockfd < 0) {
            dlog_print(DLOG_ERROR, LOG_TAG, "accept() failed = %s", strerror(errno));
            PRINT_MSG("accept() failed = %s", strerror(errno));
        }

        bzero(buffer, SIZE);
        n = read(newsockfd, buffer, SIZE-1);

        if (n < 0) {
            dlog_print(DLOG_ERROR, LOG_TAG, "read() failed = %s", strerror(errno));
            PRINT_MSG("read() failed = %s", strerror(errno));
        }

        dlog_print(DLOG_DEBUG, LOG_TAG, "Server got the message: %s", buffer);
        PRINT_MSG("Server got the message: %s", buffer);
        char *message_from_server = "This is a message send from the server";
        n = write(newsockfd, message_from_server, strlen(message_from_server));

        if (n < 0) {
            dlog_print(DLOG_ERROR, LOG_TAG, "write() failed = %s", strerror(errno));
            PRINT_MSG("write() failed = %s", strerror(errno));
        }

        close(newsockfd);
    }

    close(sockfd);
    return;
}

static void _server_thread_end_cb(void *data, Ecore_Thread *thread)
{
    dlog_print(DLOG_DEBUG, LOG_TAG, "_server_thread_end_cb()");
}

static void _server_thread_cancel_cb(void *data, Ecore_Thread *thread)
{
    dlog_print(DLOG_DEBUG, LOG_TAG, "_server_thread_cancel_cb()");
}

/* Socket for use and check the default connection  */
void _ip_sockets_cb(appdata_s *ad, Evas_Object *obj, void *event_info)
{
    int rv = 0;

    // Initialize a socket
    int ip_type = -1;
    char *user_url = "127.0.0.1";
    char *user_port = "8080";
    char *user_msg = "User message";
    char *local_ipv4 = NULL;
    char *local_ipv6 = NULL;
    char *interface_name = NULL;

    connection_type_e net_state;
    connection_profile_h profile_h = NULL;

    // Check whether the default connection is available
    rv = connection_get_type(connection, &net_state);
    if (rv != CONNECTION_ERROR_NONE) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "Failed to get connection type %d", rv);
        PRINT_MSG("Failed to get connection type %d", rv);
        return;
    }

    if (net_state == CONNECTION_TYPE_DISCONNECTED) {
        PRINT_MSG("Turn on WiFi");
        return;
    }

    // Check the address type of the default connection
    rv = connection_get_current_profile(connection, &profile_h);

    if (rv != CONNECTION_ERROR_NONE) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "Failed to get profile handle %d", rv);
        PRINT_MSG("Failed to get profile handle %d", rv);
        return;
    }

    rv = connection_profile_get_ip_address(profile_h, CONNECTION_ADDRESS_FAMILY_IPV6, &local_ipv6);
    if (rv == CONNECTION_ERROR_NONE && g_strcmp0(local_ipv6, "::") != 0) {
        ip_type = CONNECTION_ADDRESS_FAMILY_IPV6;
        dlog_print(DLOG_DEBUG, LOG_TAG, "IPv6 address : %s", local_ipv6);
        PRINT_MSG("IPv6 address : %s", local_ipv6);
    }

    // If both IPv4 and IPv6 types are set, the IPv4 type is used as default here
    rv = connection_profile_get_ip_address(profile_h, CONNECTION_ADDRESS_FAMILY_IPV4, &local_ipv4);

    if (rv == CONNECTION_ERROR_NONE && g_strcmp0(local_ipv4, "0.0.0.0") != 0) {
        ip_type = CONNECTION_ADDRESS_FAMILY_IPV4;
        dlog_print(DLOG_DEBUG, LOG_TAG, "IPv4 address : %s", local_ipv4);
        PRINT_MSG("IPv4 address : %s", local_ipv4);
    }

    if (ip_type != CONNECTION_ADDRESS_FAMILY_IPV6 && ip_type != CONNECTION_ADDRESS_FAMILY_IPV4) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "No IP address");
        PRINT_MSG("No IP address");
    }

    connection_profile_get_network_interface_name(profile_h, &interface_name);
    dlog_print(DLOG_DEBUG, LOG_TAG, "Interface Name: %s", interface_name);
    PRINT_MSG("Interface Name: %s", interface_name);

    // Retrieving the Address Family
    struct sockaddr_in6 *addr6;
    struct addrinfo hints;
    struct addrinfo *result;

    memset(&hints, 0x00, sizeof(struct addrinfo));

    hints.ai_family = PF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_protocol = IPPROTO_TCP;

    if (getaddrinfo(user_url, user_port, &hints, &result) != 0) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "getaddrinfo() error");
        PRINT_MSG("getaddrinfo() error");
        return;
    }

    PRINT_MSG("getaddrinfo() succeeded");

    // Find the proper address family and create the socket
    int sockfd = -1;
    struct addrinfo *rp;

    for (rp = result; rp != NULL; rp = rp->ai_next) {
        if (rp->ai_family == AF_INET && ip_type == CONNECTION_ADDRESS_FAMILY_IPV4) {
            if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
                dlog_print(DLOG_DEBUG, LOG_TAG, "socket error");
                PRINT_MSG("socket error");
                freeaddrinfo(result);
                return;
            }

            dlog_print(DLOG_DEBUG, LOG_TAG, "IPv4");
            PRINT_MSG("IPv4");

            // Connect to the remote host
            if (connect(sockfd, rp->ai_addr, rp->ai_addrlen) < 0) {
                dlog_print(DLOG_DEBUG, LOG_TAG, "connect() error: %s", strerror(errno));
                PRINT_MSG("connect() error: %s", strerror(errno));
                freeaddrinfo(result);
                close(sockfd);
                return;
            }

            PRINT_MSG("connect() done");
        } else if (rp->ai_family == AF_INET6 && ip_type == CONNECTION_ADDRESS_FAMILY_IPV6) {
            if ((sockfd = socket(AF_INET6, SOCK_STREAM, 0)) < 0) {
                dlog_print(DLOG_DEBUG, LOG_TAG, "socket error");
                PRINT_MSG("socket error");
                freeaddrinfo(result);
                return;
            }

            dlog_print(DLOG_DEBUG, LOG_TAG, "IPv6");
            PRINT_MSG("IPv6");

            connection_profile_get_network_interface_name(profile_h, &interface_name);
            dlog_print(DLOG_DEBUG, LOG_TAG, "Interface Name: %s", interface_name);
            PRINT_MSG("Interface Name: %s", interface_name);

            addr6 = (struct sockaddr_in6 *)rp->ai_addr;
            addr6->sin6_scope_id = if_nametoindex(interface_name);

            if ((sockfd = connect(sockfd, (struct sockaddr *)addr6, rp->ai_addrlen)) < 0) {
                dlog_print(DLOG_DEBUG, LOG_TAG, "connect() error: %s", strerror(errno));
                PRINT_MSG("connect() error: %s", strerror(errno));
                freeaddrinfo(result);
                close(sockfd);
                return;
            }
        } else {
            if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
                dlog_print(DLOG_DEBUG, LOG_TAG, "socket error");
                PRINT_MSG("socket error");
                freeaddrinfo(result);
                return;
            }

            // Connect to the remote host
            if (connect(sockfd, rp->ai_addr, rp->ai_addrlen) < 0) {
                dlog_print(DLOG_DEBUG, LOG_TAG, "connect() error: %s", strerror(errno));
                PRINT_MSG("connect() error: %s", strerror(errno));
                freeaddrinfo(result);
                close(sockfd);
                return;
            }

            PRINT_MSG("connect() done");

        }
    }

    // Manage messages
    // Send a message to the remote host
    int count = 0;
    char buf[SIZE];
    memset(buf, 0x00, SIZE);

    if ((count = write(sockfd, user_msg, SIZE)) < 0) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "write() error: %s", strerror(errno));
        PRINT_MSG("write() error: %s", strerror(errno));
        freeaddrinfo(result);
        close(sockfd);
    }

    // Read a message from the remote host
    memset(buf, 0x00, SIZE);

    if ((count = read(sockfd, buf, SIZE)) < 0) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "read() error: %s", strerror(errno));
        PRINT_MSG("read() error: %s", strerror(errno));

        freeaddrinfo(result);
        close(sockfd);
    }

    buf[count] = '\0';
    dlog_print(DLOG_DEBUG, LOG_TAG, "Server: count: %d, msg: %s", count, buf);
    PRINT_MSG("Read count: %d, msg: %s", count, buf);

    // Close the socket and release the resources
    freeaddrinfo(result);
    close(sockfd);

    connection_profile_destroy(profile_h);

    free(local_ipv6);
    free(local_ipv4);
    free(interface_name);
}

void create_buttons_in_main_window(appdata_s *ad)
{
    /*
     * Start a thread which will communicate with the main thread
     * by calling ecore_thread_feedback(), which will run the server part
     */
    ad->thread = ecore_thread_run(_server_thread, _server_thread_end_cb,
                                  _server_thread_cancel_cb, ad);

    if (ad->thread == NULL)
        dlog_print(DLOG_ERROR, LOG_TAG, "Could not create thread");

    Evas_Object *display = _create_new_cd_display(ad, "Connection", _pop_cb);

    _new_button(ad, display, "Network connection details", _connection_details_cb);
    _new_button(ad, display, "Network informations", _connection_information_cb);
    _new_button(ad, display, "Network property", _connection_property_cb);
    _new_button(ad, display, "IP sockets", _ip_sockets_cb);

    /* Create the connection handle */
    int error_code;
    error_code = connection_create(&connection);
    if (error_code != CONNECTION_ERROR_NONE) {
        dlog_print(DLOG_DEBUG, LOG_TAG, "[connection_create] Failed.");
        PRINT_MSG("[connection_create] Failed.");
        return;
    }
}
