/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "user_callbacks.h"
#include "main.h"

#include <storage.h>

static int internal_storage_id;
static bool storage_info_cb(int storage_id, storage_type_e type, storage_state_e state,
                            const char *path, void *user_data)
{
    if (type == STORAGE_TYPE_INTERNAL) {
        internal_storage_id = storage_id;
        PRINT_MSG("storage id: %d", storage_id);
        dlog_print(DLOG_DEBUG, LOG_TAG, "storage id: %d", storage_id);
        return false;
    }

    return true;
}

void _storage_cb(void *data, Evas_Object *obj, void *event_info)
{
    // Using Statvfs Wrapper
    struct statvfs s;
    int error = storage_get_internal_memory_size(&s);

    if (error != STORAGE_ERROR_NONE) {
        PRINT_MSG("storage_get_internal_memory_size: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "storage_get_internal_memory_size: %d", error);
        return;
    }

    PRINT_MSG("Internal memory");
    PRINT_MSG("Total bytes: %llu, available bytes: %llu",
              (unsigned long long int)s.f_frsize * s.f_blocks,
              (unsigned long long int)s.f_bsize * s.f_bavail);

    error = storage_get_external_memory_size(&s);

    if (error != STORAGE_ERROR_NONE) {
        PRINT_MSG("storage_get_external_memory_size: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "storage_get_external_memory_size: %d", error);
        return;
    }

    PRINT_MSG("External memory");
    PRINT_MSG("Total bytes: %llu, available bytes: %llu",
              (unsigned long long int)s.f_frsize * s.f_blocks,
              (unsigned long long int)s.f_bsize * s.f_bavail);

    // Retrieving Storage Information
    error = storage_foreach_device_supported(storage_info_cb, NULL);

    if (error != STORAGE_ERROR_NONE) {
        PRINT_MSG("storage_foreach_device_supported: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "storage_foreach_device_supported: %d", error);
        return;
    }

    char *path = NULL;
    error = storage_get_root_directory(internal_storage_id, &path);

    if (error != STORAGE_ERROR_NONE) {
        PRINT_MSG("storage_get_root_directory: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "storage_get_root_directory: %d", error);
    } else {
        PRINT_MSG("path: %s", path);
        dlog_print(DLOG_ERROR, LOG_TAG, "path: %s", path);
        free(path);
        path = NULL;
    }

    error = storage_get_directory(internal_storage_id, STORAGE_DIRECTORY_IMAGES, &path);

    if (error != STORAGE_ERROR_NONE) {
        PRINT_MSG("storage_get_directory: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "storage_get_directory: %d", error);
    } else {
        PRINT_MSG("Images directory path: %s", path);
        dlog_print(DLOG_ERROR, LOG_TAG, "Images directory path: %s", path);
        free(path);
    }

    storage_type_e type;
    error = storage_get_type(internal_storage_id, &type);

    if (error != STORAGE_ERROR_NONE) {
        PRINT_MSG("storage_get_type: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "storage_get_type: %d", error);
    }

    storage_state_e state;
    error = storage_get_state(internal_storage_id, &state);

    if (error != STORAGE_ERROR_NONE) {
        PRINT_MSG("storage_get_state: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "storage_get_state: %d", error);
    }

    // Retrieving Storage Space Information
    unsigned long long bytes = 0;
    error = storage_get_total_space(internal_storage_id, &bytes);

    if (error != STORAGE_ERROR_NONE) {
        PRINT_MSG("storage_get_total_space: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "storage_get_total_space: %d", error);
    } else {
        PRINT_MSG("Total space(bytes): %llu", bytes);
        dlog_print(DLOG_ERROR, LOG_TAG, "total space bytes: %llu", bytes);
    }

    error = storage_get_available_space(internal_storage_id, &bytes);

    if (error != STORAGE_ERROR_NONE) {
        PRINT_MSG("storage_get_available_space: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "storage_get_available_space: %d", error);
    } else {
        PRINT_MSG("Available space(bytes): %llu", bytes);
        dlog_print(DLOG_ERROR, LOG_TAG, "available space bytes: %llu", bytes);
    }
}

void create_buttons_in_main_window(appdata_s *ad)
{
    Evas_Object *display = _create_new_cd_display(ad, "Storage", _pop_cb);
    _new_button(ad, display, "Start", _storage_cb);
}
