/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "user_callbacks.h"
#include "main.h"

tts_h tts;

void _tts_state_changed_cb(tts_h tts, tts_state_e previous, tts_state_e current, void *user_data)
{
    appdata_s *ad = (appdata_s *)user_data;
    if (current == TTS_STATE_PLAYING || current == TTS_STATE_PAUSED)
        elm_object_disabled_set(ad->pause_resume, EINA_FALSE);
    else
        elm_object_disabled_set(ad->pause_resume, EINA_TRUE);

    PRINT_MSG("State has changed %s", current == TTS_STATE_READY ? "TTS_STATE_READY" :
              current == TTS_STATE_CREATED ? "TTS_STATE_CREATED" :
              current == TTS_STATE_PAUSED ? "TTS_STATE_PAUSED" : "TTS_STATE_PLAYING");
}

void _tts_error_cb(tts_h tts, int utt_id, tts_error_e reason, void *user_data)
{
    dlog_print(DLOG_ERROR, LOG_TAG, "An error occured");
}

void default_voice_changed_cb(tts_h tts,
                              const char *previous_language, int previous_voice_type,
                              const char *current_language, int current_voice_type, void *user_data)
{
    PRINT_MSG("Default voice has change");
}

void utterance_started_cb(tts_h tts, int utt_id, void *user_data)
{
    PRINT_MSG("Utterance");
}

void utterance_completed_cb(tts_h tts, int utt_id, void *user_data)
{
    PRINT_MSG("Utterance completed");

    // Stopping Playback
    int ret = tts_stop(tts);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_stop] failed = %d", ret);
        PRINT_MSG("[tts_stop] failed = %d", ret);
    }
}

void _tutorial_TTS_play(void *data, Evas_Object *obj, void *event_info)
{
    //Get the current state
    tts_state_e current_state;
    int ret = tts_get_state(tts, &current_state);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_get_state] failed = %d", ret);
        PRINT_MSG("[tts_get_state] failed = %d", ret);
    }

    if (current_state != TTS_STATE_READY) {
        PRINT_MSG("Bad state = %d", current_state);
        dlog_print(DLOG_ERROR, LOG_TAG, "Bad state = %d", current_state);
        return;
    }

    // Adding text
    const char *text = elm_object_text_get(((appdata_s *)data)->entry);    // Text to read
    const char *language = NULL; // Language
    int voice_type = TTS_VOICE_TYPE_FEMALE; // Voice type
    int speed = TTS_SPEED_AUTO; // Read speed
    int utt_id;                 // Utterance ID for the requested text

    unsigned int count;
    ret = tts_get_max_text_size(tts, &count);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_get_max_text_size] failed = %d", ret);
        PRINT_MSG("[tts_get_max_text_size] failed = %d", ret);
    }

    if (strlen(text) > count) {
        dlog_print(DLOG_ERROR, LOG_TAG, "Maximum text length is %u", count);
        PRINT_MSG("Maximum text length is %u", count);
    }

    ret = tts_add_text(tts, text, language, voice_type, speed, &utt_id);
    if (TTS_ERROR_NONE != ret) {
        PRINT_MSG("[tts_add_text] failed = %d", ret);
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_add_text] failed = %d", ret);
    }

    // Playing Playback
    ret = tts_play(tts);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_play] failed = %d", ret);
        PRINT_MSG("[tts_play] failed = %d", ret);
    }
}

void _tutorial_TTS_pause(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s*)data;
    // Check state
    tts_state_e current_state;
    int ret = tts_get_state(tts, &current_state);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_get_state] failed = %d", ret);
        PRINT_MSG("[tts_get_state] failed = %d", ret);
    }

    if (current_state == TTS_STATE_PAUSED) {
        // Playing Playback
        ret = tts_play(tts);
        if (TTS_ERROR_NONE != ret) {
            dlog_print(DLOG_ERROR, LOG_TAG, "[tts_play] failed = %d", ret);
            PRINT_MSG("[tts_play] failed = %d", ret);
        } else {
            elm_object_text_set(ad->pause_resume, "Pause");
        }
    } else if (current_state == TTS_STATE_PLAYING) {
        // Pausing Playback
        int ret = tts_pause(tts);
        if (TTS_ERROR_NONE != ret) {
            dlog_print(DLOG_ERROR, LOG_TAG, "[tts_pause] failed = %d", ret);
            PRINT_MSG("[tts_pause] failed = %d", ret);
        } else {
            elm_object_text_set(ad->pause_resume, "Resume");
        }
    }
}

bool supported_voice_cb(tts_h tts, const char *language, int voice_type, void *user_data)
{
    PRINT_MSG("Language %s", language);
    return true;
}

void create_buttons_in_main_window(appdata_s *ad)
{
    Evas_Object *display = _create_new_cd_display(ad, "Text To Speech", _pop_cb);

    _new_entry(ad, display, "Example", _tutorial_TTS_play);
    _new_button(ad, display, "Text to speach", _tutorial_TTS_play);
    ad->pause_resume = _new_button(ad, display, "Pause", _tutorial_TTS_pause);
    elm_object_disabled_set(ad->pause_resume, EINA_TRUE);

    // Creating TTS Handle
    int ret = tts_create(&tts);
    if (TTS_ERROR_NONE != ret) {
        PRINT_MSG("[tts_create] failed %s", ret == TTS_ERROR_NOT_SUPPORTED ? "Not supported" : "");
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_create] failed = %d", ret);
    }

    // Setting Callbacks
    ret = tts_set_error_cb(tts, _tts_error_cb, NULL);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_set_error_cb] failed = %d", ret);
        PRINT_MSG("[tts_set_error_cb] failed = %d", ret);
    }

    ret = tts_set_state_changed_cb(tts, _tts_state_changed_cb, (void *)ad);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_set_state_changed_cb] failed = %d", ret);
        PRINT_MSG("[tts_set_state_changed_cb] failed = %d", ret);
    }

    ret = tts_set_default_voice_changed_cb(tts, default_voice_changed_cb, NULL);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_set_default_voice_changed_cb] failed = %d", ret);
        PRINT_MSG("[tts_set_default_voice_changed_cb] failed = %d", ret);
    }

    ret = tts_set_utterance_started_cb(tts, utterance_started_cb, NULL);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_set_utterance_started_cb] failed = %d", ret);
        PRINT_MSG("[tts_set_utterance_started_cb] failed = %d", ret);
    }

    ret = tts_set_utterance_completed_cb(tts, utterance_completed_cb, NULL);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_set_utterance_completed_cb] failed = %d", ret);
        PRINT_MSG("[tts_set_utterance_completed_cb] failed = %d", ret);
    }

    // Getting and Setting the Mode
    ret = tts_set_mode(tts, TTS_MODE_NOTIFICATION);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_set_mode] failed = %d", ret);
        PRINT_MSG("[tts_set_mode] failed = %d", ret);
    }

    tts_mode_e mode;
    ret = tts_get_mode(tts, &mode);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_get_mode] failed = %d", ret);
        PRINT_MSG("[tts_get_mode] failed = %d", ret);
    }

    // Obtain a list of voices supported by TTS
    ret = tts_foreach_supported_voices(tts, supported_voice_cb, NULL);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_foreach_supported_voices] failed = %d", ret);
        PRINT_MSG("[tts_foreach_supported_voices] failed = %d", ret);
    }

    // Get the default voice
    int default_voice_type;
    char *default_lang = NULL;
    ret = tts_get_default_voice(tts, &default_lang, &default_voice_type);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_get_default_voice] failed = %d", ret);
        PRINT_MSG("[tts_get_default_voice] failed = %d", ret);
    }

    PRINT_MSG("Default language: %s", default_lang);

    // Connecting TTS
    ret = tts_prepare(tts);
    if (TTS_ERROR_NONE != ret) {
        dlog_print(DLOG_ERROR, LOG_TAG, "[tts_prepare] failed = %d", ret);
        PRINT_MSG("[tts_prepare] failed = %d", ret);
    } else
        PRINT_MSG("TTS prepared");
}
