/*
 * Copyright (c) 2013 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "$(appName).h"

static char *image_data[6][3]  = {
	{"movie1.jpg", "movie2.jpg", "movie3.jpg"},
	{"game1.jpg", "game2.jpg", "game3.jpg"},
	{"music1.jpg","music2.jpg","music3.jpg"},
	{"book1.jpg", "book2.jpg", "book3.jpg"},
	{"magazine1.jpg", "magazine2.jpg", "magazine3.jpg"},
	{"app1.jpg", "app2.jpg", "app3.jpg"}
};

typedef struct appdata {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *layout;
	Evas_Object *nf;
	Evas_Object *mainview;
} appdata_s;

static void
win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	ui_app_exit();
}

static void
mainview_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_lower(ad->win);
}

static void
clicked_cb(void *user_data, Evas_Object *obj, void *event_info)
{
	// Add your application logic.
}

static void
pressed_cb(void *user_data, Evas_Object *obj, void *event_info)
{
	evas_object_color_set(obj, 128, 128, 128, 128);
}

static void
unpressed_cb(void *user_data, Evas_Object *obj, void *event_info)
{
	evas_object_color_set(obj, 255, 255, 255, 255);
}

static Evas_Object*
create_label(Evas_Object *parent, const char *text)
{
	Evas_Object *text_obj = elm_label_add(parent);
	elm_object_text_set(text_obj, text);
	return text_obj;
}

static Evas_Object*
create_image(Evas_Object *parent, const char *image_path, int min_height)
{
	char path[100] = {0};
	snprintf(path, sizeof(path), ICON_DIR"/%s", image_path);
	Evas_Object *ic = elm_image_add(parent);
	elm_image_file_set(ic, path, NULL);
	evas_object_size_hint_min_set(ic, 0, min_height);
	return ic;
}

static Evas_Object*
create_button(Evas_Object *parent, Evas_Object *content)
{
	Evas_Object *button = elm_button_add(parent);
	elm_object_style_set(button, "transparent");
	evas_object_size_hint_weight_set(button, EVAS_HINT_EXPAND, 0.0);
	evas_object_size_hint_align_set(button, EVAS_HINT_FILL, 0.0);

	elm_object_content_set(button, content);

	evas_object_smart_callback_add(button, "pressed", pressed_cb, content);
	evas_object_smart_callback_add(button, "unpressed", unpressed_cb, content);
	evas_object_smart_callback_add(button, "clicked", clicked_cb, content);

	return button;
}

static Evas_Object*
create_item_header(Evas_Object *parent, const char *item_name)
{
	Evas_Object *label = create_label(parent, item_name);
	Evas_Object *item_header = create_button(parent, label);
	return item_header;
}

static Evas_Object*
create_item_content(Evas_Object *parent, int content_height, int image_index)
{
	Evas_Object *content_box, *content, *ic;

	content_box = elm_box_add(parent);
	elm_box_horizontal_set(content_box, EINA_TRUE);
	evas_object_size_hint_align_set(content_box, EVAS_HINT_FILL, 0.0);

	evas_object_show(content_box);
	elm_box_padding_set(content_box, ELM_SCALE_SIZE(5), ELM_SCALE_SIZE(10));
	ic = create_image(parent, image_data[image_index][0], content_height);
	content = create_button(content_box, ic);
	evas_object_show(content);
	elm_box_pack_end(content_box, content);

	ic = create_image(parent, image_data[image_index][1], content_height);
	content = create_button(content_box, ic);
	evas_object_show(content);
	elm_box_pack_end(content_box, content);

	ic = create_image(parent, image_data[image_index][2], content_height);
	content = create_button(content_box, ic);
	evas_object_show(content);
	elm_box_pack_end(content_box, content);

	return content_box;
}

static Evas_Object*
add_item_table(Evas_Object *parent, int content_height)
{
	Evas_Object *content_table, *content, *ic;

	content_table = elm_table_add(parent);
	elm_table_homogeneous_set(content_table, EINA_TRUE);
	evas_object_size_hint_align_set(content_table, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_table_padding_set(content_table, ELM_SCALE_SIZE(5), ELM_SCALE_SIZE(5));
	evas_object_show(content_table);

	ic = create_image(parent, "apps.jpg", content_height);
	content = create_button(content_table, ic);
	evas_object_show(content);
	elm_table_pack(content_table, content, 0, 0, 1, 1);

	ic = create_image(parent, "games.jpg", content_height);
	content = create_button(content_table, ic);
	evas_object_show(content);
	elm_table_pack(content_table, content, 1, 0, 1, 1);

	ic = create_image(parent, "movies.jpg", content_height);
	content = create_button(content_table, ic);
	evas_object_show(content);
	elm_table_pack(content_table, content, 0, 1, 1, 1);

	ic = create_image(parent, "music.jpg", content_height);
	content = create_button(content_table, ic);
	evas_object_show(content);
	elm_table_pack(content_table, content, 1, 1, 1, 1);

	ic = create_image(parent, "books.jpg", content_height);
	content = create_button(content_table, ic);
	evas_object_show(content);
	elm_table_pack(content_table, content, 0, 2, 1, 1);

	ic = create_image(parent, "magazines.jpg", content_height);
	content = create_button(content_table, ic);
	evas_object_show(content);
	elm_table_pack(content_table, content, 1, 2, 1, 1);

	return content_table;
}

static Evas_Object*
add_main_view(appdata_s *ad)
{
	Evas_Object *scroller, *main_view, *item_table, *item_header, *item_content;
	int content_height = ELM_SCALE_SIZE(240);
	int grid_item_height = ELM_SCALE_SIZE(60);

	scroller = elm_scroller_add(ad->win);
	elm_scroller_bounce_set(scroller, EINA_FALSE, EINA_TRUE);
	evas_object_size_hint_weight_set(scroller, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(scroller, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_show(scroller);

	main_view = elm_box_add(ad->win);
	elm_box_align_set(main_view, 0, 0);
	evas_object_size_hint_weight_set(main_view, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_box_padding_set(main_view, ELM_SCALE_SIZE(10), ELM_SCALE_SIZE(15));
	elm_object_content_set(scroller, main_view);

	item_table = add_item_table(main_view, grid_item_height);
	evas_object_show(item_table);
	elm_box_pack_end(main_view, item_table);

	item_header = create_item_header(main_view, "<font_size=30>New Movie Releases</font_size>");
	evas_object_show(item_header);
	elm_box_pack_end(main_view, item_header);

	item_content = create_item_content(main_view, content_height, 0);
	evas_object_show(item_content);
	elm_box_pack_end(main_view, item_content);

	item_header = create_item_header(main_view, "<font_size=30>Top Grossing Games</font_size>");
	evas_object_show(item_header);
	elm_box_pack_end(main_view, item_header);

	item_content = create_item_content(main_view, content_height, 1);
	evas_object_show(item_content);
	elm_box_pack_end(main_view, item_content);

	item_header = create_item_header(main_view, "<font_size=30>Chart Buster Music</font_size>");
	evas_object_show(item_header);
	elm_box_pack_end(main_view, item_header);

	item_content = create_item_content(main_view, content_height, 2);
	evas_object_show(item_content);
	elm_box_pack_end(main_view, item_content);


	item_header = create_item_header(main_view, "<font_size=30>Books</font_size>");
	evas_object_show(item_header);
	elm_box_pack_end(main_view, item_header);

	item_content = create_item_content(main_view, content_height, 3);
	evas_object_show(item_content);
	elm_box_pack_end(main_view, item_content);

	item_header = create_item_header(main_view, "<font_size=30>Magazine</font_size>");
	evas_object_show(item_header);
	elm_box_pack_end(main_view, item_header);

	item_content = create_item_content(main_view, content_height, 4);
	evas_object_show(item_content);
	elm_box_pack_end(main_view, item_content);

	item_header = create_item_header(main_view, "<font_size=30>App of the week</font_size>");
	evas_object_show(item_header);
	elm_box_pack_end(main_view, item_header);

	item_content = create_item_content(main_view, content_height, 5);
	evas_object_show(item_content);
	elm_box_pack_end(main_view, item_content);

	return scroller;
}

static void
create_base_gui(appdata_s *ad)
{
	Evas_Object *bg = NULL;

	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_app_base_scale_set(1.8);

	elm_win_conformant_set(ad->win, EINA_TRUE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	if (elm_win_wm_rotation_supported_get(ad->win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(ad->win, (const int *)(&rots), 4);
	}

	evas_object_smart_callback_add(ad->win, "delete,request", win_delete_request_cb, NULL);

	/* Conformant */
	ad->conform = elm_conformant_add(ad->win);
	elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
	evas_object_size_hint_weight_set(ad->conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(ad->win, ad->conform);
	evas_object_show(ad->conform);

	/* Indicator BG */
	bg = elm_bg_add(ad->conform);
	elm_object_style_set(bg, "indicator/headerbg");
	elm_object_part_content_set(ad->conform, "elm.swallow.indicator_bg", bg);
	evas_object_show(bg);

	/* Base Layout */
	ad->layout = elm_layout_add(ad->conform);
	evas_object_size_hint_weight_set(ad->layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_layout_theme_set(ad->layout, "layout", "application", "default");
	evas_object_show(ad->layout);

	elm_object_content_set(ad->conform, ad->layout);

	/* Naviframe */
	ad->nf = elm_naviframe_add(ad->layout);

	/* Main View */
	ad->mainview = add_main_view(ad);

	elm_naviframe_item_push(ad->nf, "Application Store", NULL, NULL, ad->mainview, NULL);

	elm_object_part_content_set(ad->layout, "elm.swallow.content", ad->nf);

	/* Show window after base gui is set up */
	evas_object_show(ad->win);

	eext_object_event_callback_add(ad->mainview, EEXT_CALLBACK_BACK, mainview_back_cb, ad);
}

static bool
app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
		Initialize UI resources and application's data
		If this function returns true, the main loop of application starts
		If this function returns false, the application is terminated */
	appdata_s *ad = data;

	create_base_gui(ad);

	return true;
}

static void
app_control(app_control_h app_control, void *data)
{
	// Handle the launch request
}

static void
app_pause(void *data)
{
	// Take necessary actions when application becomes invisible
}

static void
app_resume(void *data)
{
	// Take necessary actions when application becomes visible.
}

static void
app_terminate(void *data)
{
	// Release all resources
	appdata_s *ad;

	if (!data)
		return;

	ad = data;

	if (ad->win)
		evas_object_del(ad->win);
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

static void
ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_DEVICE_ORIENTATION_CHANGED*/
	return;
}

static void
ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_REGION_FORMAT_CHANGED*/
}

static void
ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_BATTERY*/
}

static void
ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_orient_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_changed, &ad);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() is failed. err = %d", ret);
	}

	return ret;
}

