/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "$(appName).h"

static char* user_email = "example@tizen.com";

void in_box_cb(void *data, Evas_Object *obj, void *event_info);
void out_box_cb(void *data, Evas_Object *obj, void *event_info);

typedef struct appdata {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *nf;
	Evas_Object *base_ly;
	Evas_Object *panel;
	Evas_Object *panel_list;
	Evas_Object *genlist;
} appdata_s;

typedef struct compose_data {
	appdata_s *ad;
	Evas_Object *to;
	Evas_Object *subject;
	Evas_Object *content;
	Evas_Object *popup;
} cdata_s;

typedef struct item_data
{
	int index;
	Elm_Object_Item *item;
} item_data_s;

static void
win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	ui_app_exit();
}

static Evas_Object*
create_favorite_icon(Evas_Object *parent)
{
	Evas_Object *img = elm_image_add(parent);
	char buf[PATH_MAX];

	snprintf(buf, sizeof(buf), "%s/favorite.png", ICON_DIR);
	elm_image_file_set(img, buf, NULL);
	evas_object_color_set(img, 255, 180, 0, 255);
	evas_object_size_hint_align_set(img, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(img, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	return img;
}

static Eina_Bool
nf_it_pop_cb(void *data, Elm_Object_Item *it)
{
	/* Don't pop the last view but hide window */
	appdata_s *ad = data;
	elm_win_lower(ad->win);
	return EINA_FALSE;
}

static void
back_button_clicked_cb(void *data, Evas_Object * obj, void *event_info)
{
	cdata_s *cd = data;
	appdata_s *ad = cd->ad;
	elm_naviframe_item_pop(ad->nf);
}

static void
ok_cb(void *data, Evas_Object *obj, void *event_info)
{
	evas_object_del(data);
}

static void
cancel_cb(void *data, Evas_Object *obj, void *event_info)
{
	evas_object_del(data);
}

static void
timeout_cb(void *data, Evas_Object *obj, void *event_info)
{
	evas_object_del(obj);
}

static Evas_Object *
create_popup(Evas_Object *parent, char *style, char *text)
{
	Evas_Object *popup;

	popup = elm_popup_add(parent);
	if (style)
		elm_object_style_set(popup, style);
	if (text)
		elm_object_text_set(popup, text);
	evas_object_show(popup);

	eext_object_event_callback_add(popup, EEXT_CALLBACK_BACK, eext_popup_back_cb, NULL);

	return popup;
}

static Evas_Object *
create_button(Evas_Object *parent, char *style, char *text)
{
	Evas_Object *button;

	button = elm_button_add(parent);
	if (style)
		elm_object_style_set(button, style);
	if (text)
		elm_object_text_set(button, text);
	evas_object_show(button);

	return button;
}
static void
send_cb(void *data, Evas_Object *obj, void *event_info)
{
	cdata_s *cd = data;
	appdata_s *ad = cd->ad;
	Evas_Object *popup;
	evas_object_del(cd->popup);
	elm_naviframe_item_pop(ad->nf);

	popup = create_popup(ad->win, "toast", "Sending message ...");
	evas_object_size_hint_weight_set(popup, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_popup_timeout_set(popup, 2.0);
	evas_object_smart_callback_add(popup, "timeout", timeout_cb, NULL);
}

static void
send_button_clicked_cb(void *data, Evas_Object * obj, void *event_info)
{
	cdata_s *cd = data;
	appdata_s *ad = cd->ad;
	Evas_Object *popup, *btn1, *btn2;
	const char *to, *subject, *content;

	to = elm_entry_entry_get(cd->to);
	if (!strcmp(to, "")) {
		popup = create_popup(ad->win, NULL, "Add at least one recipient.");
		evas_object_size_hint_weight_set(popup, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

		btn1 = create_button(popup, "popup", "OK");
		evas_object_smart_callback_add(btn1, "clicked", ok_cb, popup);
		elm_object_part_content_set(popup, "button1", btn1);
	}
	else {
		subject = elm_entry_entry_get(cd->subject);
		if (!strcmp(subject, "")) {
			popup = create_popup(ad->win, NULL, "There's no text in the message subject.");
			evas_object_size_hint_weight_set(popup, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
			cd->popup = popup;

			btn1 = create_button(popup, "popup", "Cancel");
			evas_object_smart_callback_add(btn1, "clicked", cancel_cb, popup);
			elm_object_part_content_set(popup, "button1", btn1);

			btn2 = create_button(popup, "popup", "Send");
			evas_object_smart_callback_add(btn2, "clicked", send_cb, cd);
			elm_object_part_content_set(popup, "button2", btn2);
		}
		else {
			content = elm_entry_entry_get(cd->content);
			if (!strcmp(content, "")) {
				popup = create_popup(ad->win, NULL, "There's no text in the message body.");
				evas_object_size_hint_weight_set(popup, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
				cd->popup = popup;

				btn1 = create_button(popup, "popup", "Cancel");
				evas_object_smart_callback_add(btn1, "clicked", cancel_cb, popup);
				elm_object_part_content_set(popup, "button1", btn1);

				btn2 = create_button(popup, "popup", "Send");
				evas_object_smart_callback_add(btn2, "clicked", send_cb, cd);
				elm_object_part_content_set(popup, "button2", btn2);
			}
			else {
				elm_naviframe_item_pop(ad->nf);

				popup = create_popup(ad->win, "toast", "Sending message ...");
				evas_object_size_hint_weight_set(popup, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
				elm_popup_timeout_set(popup, 2.0);
				evas_object_smart_callback_add(popup, "timeout", timeout_cb, NULL);
			}
		}
	}
}

static Evas_Object*
create_scroller(Evas_Object *parent)
{
	Evas_Object *scroller;

	scroller = elm_scroller_add(parent);
	elm_scroller_policy_set(scroller, ELM_SCROLLER_POLICY_OFF, ELM_SCROLLER_POLICY_AUTO);
	evas_object_show(scroller);

	return scroller;
}

static Evas_Object*
create_box(Evas_Object *parent)
{
	Evas_Object *box;
	box = elm_box_add(parent);
	evas_object_show(box);

	return box;
}

static Evas_Object*
create_entry(Evas_Object *parent, char *text, char *guide)
{
	Evas_Object *entry;
	entry = elm_entry_add(parent);

	if (text) {
		elm_object_text_set(entry, text);
	}
	if (guide) {
		elm_object_part_text_set(entry, "elm.guide", guide);
	}
	evas_object_show(entry);

	return entry;
}

static void
composer_button_clicked_cb(void *data, Evas_Object * obj, void *event_info)
{
	appdata_s *ad = data;
	cdata_s *cd = calloc(1, sizeof(cdata_s));
	cd->ad = ad;
	Evas_Object *btn, *scroller, *main_box, *myemail, *to, *subject, *content;
	Elm_Object_Item *nf_it;

	scroller = create_scroller(ad->nf);

	/* append box */
	main_box = create_box(scroller);
	evas_object_size_hint_weight_set(main_box, EVAS_HINT_EXPAND, 0.0);
	evas_object_size_hint_align_set(main_box, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_box_padding_set(main_box, 0, 30);
	elm_object_content_set(scroller, main_box);

	/* use email */
	myemail = create_entry(main_box, user_email, NULL);
	elm_entry_editable_set(myemail, EINA_FALSE);
	elm_entry_single_line_set(myemail, EINA_TRUE);
	evas_object_size_hint_weight_set(myemail, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(myemail, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_box_pack_end(main_box, myemail);

	/* to */
	to = create_entry(main_box, NULL, "To");
	evas_object_size_hint_weight_set(to, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(to, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_box_pack_end(main_box, to);
	cd->to = to;

	/* subject */
	subject = create_entry(main_box, NULL, "Subject");
	evas_object_size_hint_weight_set(subject, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(subject, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_box_pack_end(main_box, subject);
	cd->subject = subject;

	/* content */
	content = create_entry(main_box, NULL, "Compose email");
	evas_object_size_hint_weight_set(content, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(content, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_box_pack_end(main_box, content);
	cd->content = content;

	nf_it = elm_naviframe_item_push(ad->nf, "Compose", NULL, NULL, scroller, NULL);

	/* back button */
	btn = create_button(ad->nf, "naviframe/title_cancel", NULL);
	evas_object_smart_callback_add(btn, "clicked", back_button_clicked_cb, cd);
	elm_object_item_part_content_set(nf_it, "title_left_btn", btn);

	/* send button */
	btn = create_button(ad->nf, "naviframe/title_done", NULL);
	evas_object_smart_callback_add(btn, "clicked", send_button_clicked_cb, cd);
	elm_object_item_part_content_set(nf_it, "title_right_btn", btn);
}

static void
left_panel_button_clicked_cb(void *data, Evas_Object * obj, void *event_info)
{
	appdata_s *ad = data;
	if (!elm_object_disabled_get(ad->panel)) elm_panel_toggle(ad->panel);
}

static void
panel_scroll_cb(void *data, Evas_Object *obj, void *event_info)
{
	Elm_Panel_Scroll_Info *ev = event_info;
	Evas_Object *bg = data;
	int col = 127 * ev->rel_x;
	evas_object_color_set(bg, 0, 0, 0, col);
}

static void
list_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	Elm_Object_Item *it = (Elm_Object_Item *) elm_list_selected_item_get(obj);
	if(it == NULL) return;
	elm_object_item_part_text_set(elm_naviframe_top_item_get(ad->nf), NULL, elm_object_item_part_text_get(it, "default"));
	elm_list_item_selected_set(it, EINA_FALSE);
}

static void
list_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *panel = data;
	elm_panel_hidden_set(panel, EINA_TRUE);
}

void in_box_cb(void *data, Evas_Object *obj, void *event_info)
{
	/* Add your application logic. */
}

void out_box_cb(void *data, Evas_Object *obj, void *event_info)
{
	/* Add your application logic. */
}

static void
gl_del_cb(void *data, Evas_Object *obj EINA_UNUSED)
{
	/* FIXME: Unrealized callback can be called after this. */
	/* Accessing item_data_s can be dangerous on unrealized callback. */
	item_data_s *id = data;
	free(id);
}

static char *
gl_text_get_cb(void *data, Evas_Object *obj, const char *part)
{
	item_data_s *id = data;
	const Elm_Genlist_Item_Class *itc = elm_genlist_item_item_class_get(id->item);
	char buf[1024];

	if (!strcmp(itc->item_style, "groupindex")) {
		if (!strcmp(part, "elm.text.main")) {
			snprintf(buf, 1023, "%s %d", "Group Index", id->index);
			return strdup(buf);
		}
	}
	else if (!strcmp(itc->item_style, "2line.top.3")) {
		if (!strcmp(part, "elm.text.main.left.top")) {
			snprintf(buf, 1023, "[%d]%s:%s", id->index, part, "Subject");
			return strdup(buf);
		}
		else if (!strcmp(part, "elm.text.sub.left.bottom")) {
			snprintf(buf, 1023, "%s", "Content text will be written here");
			return strdup(buf);
		}
		else if (!strcmp(part, "elm.text.sub.right.bottom")) {
			int month = 7;
			int date = 14;
			snprintf(buf, 1023, "[%d/%d]", month, date);
			return strdup(buf);
		}
	}

	return NULL;
}

static Evas_Object*
gl_content_get_cb(void *data, Evas_Object *obj, const char *part)
{
	item_data_s *id = data;
	const Elm_Genlist_Item_Class *itc = elm_genlist_item_item_class_get(id->item);
	Evas_Object *content = NULL;

	if (!strcmp(itc->item_style, "2line.top.3")) {
		if (!strcmp(part, "elm.icon.right.top")) {
			content = elm_layout_add(obj);
			elm_layout_theme_set(content, "layout", "list/A/right.icon", "default");
			Evas_Object *img = create_favorite_icon(content);
			elm_layout_content_set(content, "elm.swallow.content", img);
		}
	}

	return content;
}

static void
gl_loaded_cb(void *data EINA_UNUSED, Evas_Object *obj EINA_UNUSED, void *event_info EINA_UNUSED)
{

}

static void
gl_realized_cb(void *data EINA_UNUSED, Evas_Object *obj EINA_UNUSED, void *event_info)
{
	/* If you need, you can get item's data. */
	// Elm_Object_Item *it = event_info;
	// item_data_s *id = elm_object_item_data_get(it);
}

static void
gl_longpressed_cb(void *data EINA_UNUSED, Evas_Object *obj EINA_UNUSED, void *event_info)
{
	/* If you need, you can get item's data. */
	// Elm_Object_Item *it = event_info;
	// item_data_s *id = elm_object_item_data_get(it);
}

static void
gl_selected_cb(void *data EINA_UNUSED, Evas_Object *obj EINA_UNUSED, void *event_info)
{
	Elm_Object_Item *it = event_info;
	item_data_s *id = elm_object_item_data_get(it);

	/* Unhighlight item */
	elm_genlist_item_selected_set(id->item, EINA_FALSE);
}

static Evas_Object*
create_base_layout(Evas_Object *parent)
{
	Evas_Object *layout;
	layout = elm_layout_add(parent);
	elm_layout_theme_set(layout, "layout", "drawer", "panel");
	evas_object_show(layout);

	return layout;
}

static Evas_Object*
create_panel(Evas_Object *parent)
{
	Evas_Object *panel, *bg;

	panel = elm_panel_add(parent);
	elm_panel_scrollable_set(panel, EINA_TRUE);
	elm_panel_hidden_set(panel, EINA_TRUE);
	elm_panel_orient_set(panel, ELM_PANEL_ORIENT_LEFT);

	bg = evas_object_rectangle_add(evas_object_evas_get(parent));
	evas_object_color_set(bg, 0, 0, 0, 0);
	elm_object_part_content_set(parent, "elm.swallow.bg", bg);
	evas_object_show(bg);

	evas_object_smart_callback_add(panel, "scroll", panel_scroll_cb, bg);
	evas_object_show(panel);

	return panel;
}

static Evas_Object*
create_panel_list(Evas_Object *parent)
{
	Evas_Object *list;

	list = elm_list_add(parent);
	elm_list_mode_set(list, ELM_LIST_COMPRESS);
	elm_list_item_append(list, "In Box", NULL, NULL, in_box_cb, NULL);
	elm_list_item_append(list, "Out Box", NULL, NULL, out_box_cb, NULL);
	evas_object_show(list);

	return list;
}

static Evas_Object*
create_genlist(Evas_Object *parent)
{
	Evas_Object *genlist;
	Elm_Object_Item *git, *it;
	int n_items = NUM_OF_ITEMS;
	int index;

	/* Create item class */
	Elm_Genlist_Item_Class *gitc = elm_genlist_item_class_new();
	Elm_Genlist_Item_Class *itc = elm_genlist_item_class_new();

	gitc->item_style = "groupindex";
	gitc->func.text_get = gl_text_get_cb;
	gitc->func.content_get = gl_content_get_cb;
	gitc->func.del = gl_del_cb;

	itc->item_style = "2line.top.3";
	itc->func.text_get = gl_text_get_cb;
	itc->func.content_get = gl_content_get_cb;
	itc->func.del = gl_del_cb;

	genlist = elm_genlist_add(parent);

	elm_object_style_set(genlist, "handler");
	elm_scroller_single_direction_set(genlist, ELM_SCROLLER_SINGLE_DIRECTION_HARD);

	/* HOMOGENEOUS MODE
		If item height is same when each style name is same, Use homogeneous mode. */
	elm_genlist_homogeneous_set(genlist, EINA_TRUE);

	/* COMPRESS MODE
		For the Mobile view, because it has full window, compresse mode should be used. */
	elm_genlist_mode_set(genlist, ELM_LIST_COMPRESS);

	/* Add Smart Callback */
	evas_object_smart_callback_add(genlist, "realized", gl_realized_cb, NULL);
	evas_object_smart_callback_add(genlist, "loaded", gl_loaded_cb, NULL);
	evas_object_smart_callback_add(genlist, "longpressed", gl_longpressed_cb, NULL);

	int group_count = 0;
	for (index = 0; index < n_items; index++) {
		item_data_s *id = calloc(sizeof(item_data_s), 1);
		id->index = index;
		if (index % 10 == 0) {
			group_count++;
			id->index = group_count;
			git = elm_genlist_item_append(genlist,							/* genlist object */
											  gitc,							/* item class */
											  id,							/* item class user data */
											  NULL,
											  ELM_GENLIST_ITEM_GROUP,		/* item type */
											  gl_selected_cb,				/* select smart callback */
											  id);						    /* smart callback user data */
			id->item = git;
		}
		else {
			it = elm_genlist_item_append(genlist,							/* genlist object */
											  itc,							/* item class */
											  id,							/* item class user data */
											  git,							/* parent item */
											  ELM_GENLIST_ITEM_NONE,		/* item type */
											  gl_selected_cb,				/* select smart callback */
											  id);							/* smart callback user data */
			id->item = it;
		}
	}
	elm_genlist_item_class_free(gitc);
	elm_genlist_item_class_free(itc);
	evas_object_show(genlist);

	return genlist;
}

static Evas_Object*
create_main_view(appdata_s *ad)
{
	Evas_Object *layout;
	/* Drawer layout */
	layout = create_base_layout(ad->conform);

	/* Panel */
	ad->panel = create_panel(layout);
	elm_object_part_content_set(layout, "elm.swallow.left", ad->panel);

	/* Panel list */
	ad->panel_list = create_panel_list(ad->panel);
	evas_object_size_hint_weight_set(ad->panel_list, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(ad->panel_list, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_smart_callback_add(ad->panel_list, "selected", list_clicked_cb, ad);
	eext_object_event_callback_add(ad->panel_list, EEXT_CALLBACK_BACK, list_back_cb, ad->panel);
	elm_object_content_set(ad->panel, ad->panel_list);

	return layout;
}

static void
create_base_gui(appdata_s *ad)
{
	Evas_Object *btn1, *btn2, *bg;
	Elm_Object_Item *nf_it;

	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_conformant_set(ad->win, EINA_TRUE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	if (elm_win_wm_rotation_supported_get(ad->win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(ad->win, (const int *)(&rots), 4);
	}

	evas_object_smart_callback_add(ad->win, "delete,request", win_delete_request_cb, NULL);

	/* Conformant */
	ad->conform =  elm_conformant_add(ad->win);
	evas_object_size_hint_weight_set(ad->conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
	elm_win_resize_object_add(ad->win, ad->conform);
	evas_object_show(ad->conform);

	/* Indicator BG */
	bg = elm_bg_add(ad->conform);
	elm_object_style_set(bg, "indicator/headerbg");
	elm_object_part_content_set(ad->conform, "elm.swallow.indicator_bg", bg);
	evas_object_show(bg);

	/* Main View */
	ad->base_ly = create_main_view(ad);
	elm_object_content_set(ad->conform, ad->base_ly);

	/* Naviframe */
	ad->nf = elm_naviframe_add(ad->base_ly);
	eext_object_event_callback_add(ad->nf, EEXT_CALLBACK_BACK, eext_naviframe_back_cb, ad);
	elm_object_part_content_set(ad->base_ly, "elm.swallow.content", ad->nf);
	evas_object_show(ad->nf);

	/* Genlist */
	ad->genlist = create_genlist(ad->base_ly);

	nf_it = elm_naviframe_item_push(ad->nf, "In Box", NULL, NULL, ad->genlist, NULL);
	elm_naviframe_item_pop_cb_set(nf_it, nf_it_pop_cb, ad);
	elm_object_item_part_text_set(nf_it, "subtitle", user_email);

	/* left panel toggle button */
	btn1 = create_button(ad->nf, "naviframe/drawers", NULL);
	evas_object_smart_callback_add(btn1, "clicked", left_panel_button_clicked_cb, ad);
	elm_object_item_part_content_set(nf_it, "title_left_btn", btn1);

	/* right compose button */
	btn2 = create_button(ad->nf, NULL, "<font_size=20>+</font_size>");
	evas_object_size_hint_min_set(btn2, ELM_SCALE_SIZE(36), ELM_SCALE_SIZE(32));
	evas_object_smart_callback_add(btn2, "clicked", composer_button_clicked_cb, ad);
	elm_object_item_part_content_set(nf_it, "title_right_btn", btn2);

	/* Show window after base gui is set up */
	evas_object_show(ad->win);
}

static bool
app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
		Initialize UI resources and application's data
		If this function returns true, the main loop of application starts
		If this function returns false, the application is terminated */
	appdata_s *ad = data;

	create_base_gui(ad);

	return true;
}

static void
app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
	/* Release all resources. */
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

static void
ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_DEVICE_ORIENTATION_CHANGED*/
	return;
}

static void
ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_REGION_FORMAT_CHANGED*/
}

static void
ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_BATTERY*/
}

static void
ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_orient_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_changed, &ad);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() is failed. err = %d", ret);
	}

	return ret;
}
