/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "$(appName).h"

typedef struct appdata {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *nf;
	Evas_Object *genlist;
	Evas_Object *popup;
} appdata_s;

typedef struct item_data
{
	int index;
	Elm_Object_Item *item;
} item_data_s;

const char *list_names[] = {
	"Network connections", "Wi-Fi", "Bluetooth", "SIM card manager", "Network",
	"Sound and display", "Sound", "Display",
	"Personalisation", "Wallpaper", "Color Theme",
	"User and backup", "Accounts", "Reset",
	"System", "Language and input", "Date and time", "Safety assistance", "Storage", "Security", "Privacy", "About Device"};

const char *icon_names[] = {
	"", "wifi", "bluetooth", "SIM_card_manager", "network",
	"", "sound", "display",
	"", "wallpaper", "color_theme",
	"", "account", "restart",
	"", "language_and_input", "date_and_time", "safetyassistance", "storage", "security", "privacy", "about_pdevice"};

static Evas_Object *
create_menu_popup(appdata_s *ad);

static void
dismissed_popup_cb(void *data, Evas_Object *obj, void *event_info);

static Evas_Object *
create_label(Evas_Object *parent, char *text);

static void
win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	ui_app_exit();
}

static Eina_Bool
nf_it_pop_cb(void *data, Elm_Object_Item *it)
{
	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_lower(ad->win);
	return EINA_FALSE;
}

static void
nf_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	elm_naviframe_item_pop(obj);
}

static void
nf_more_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;

	if (!ad->popup)
		ad->popup = create_menu_popup(ad);
	else {
		evas_object_del(ad->popup);
		ad->popup = NULL;
	}
}

static Evas_Object*
create_icon(Evas_Object *parent, int count)
{
	char buf[255];
	Evas_Object *img = elm_image_add(parent);

	snprintf(buf, sizeof(buf), "%s/settings_%s.png", ICON_DIR, icon_names[count]);
	elm_image_file_set(img, buf, NULL);
	evas_object_color_set(img, 110, 162, 185, 255);
	evas_object_size_hint_align_set(img, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(img, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	return img;
}

static void
gl_del_cb(void *data, Evas_Object *obj EINA_UNUSED)
{
	/* FIXME: Unrealized callback can be called after this. */
	/* Accessing item_data_s can be dangerous on unrealized callback. */
	item_data_s *id = data;
	free(id);
}

static char *
gl_text_get_cb(void *data, Evas_Object *obj, const char *part)
{
	item_data_s *id = data;
	const Elm_Genlist_Item_Class *itc = elm_genlist_item_item_class_get(id->item);
	char buf[1024];

	if (itc->item_style && !strcmp(itc->item_style, "groupindex")) {
		if (part && !strcmp(part, "elm.text.main")) {
			snprintf(buf, 1023, "%s", list_names[id->index]);
			return strdup(buf);
		}
	}
	else if (itc->item_style && !strcmp(itc->item_style, "1line")) {
		if (part && !strcmp(part, "elm.text.main.left")) {
			snprintf(buf, 1023, "%s", list_names[id->index]);
			return strdup(buf);
		}
	}
	return NULL;
}

static Evas_Object*
gl_content_get_cb(void *data, Evas_Object *obj, const char *part)
{
	item_data_s *id = data;
	const Elm_Genlist_Item_Class *itc = elm_genlist_item_item_class_get(id->item);
	Evas_Object *content = NULL;

	if (itc->item_style && !strcmp(itc->item_style, "1line")) {
		if (part && !strcmp(part, "elm.icon.1")) {
			content = elm_layout_add(obj);
			elm_layout_theme_set(content, "layout", "list/B/type.3", "default");
			Evas_Object *icon = create_icon(content, id->index);
			elm_layout_content_set(content, "elm.swallow.content", icon);
		}
	}

	return content;
}

static void
gl_loaded_cb(void *data EINA_UNUSED, Evas_Object *obj, void *event_info EINA_UNUSED)
{
}

static void
gl_realized_cb(void *data EINA_UNUSED, Evas_Object *obj EINA_UNUSED, void *event_info)
{
	/* If you need, you can get item's data. */
	// Elm_Object_Item *it = event_info;
	// item_data_s *id = elm_object_item_data_get(it);
}

static void
gl_longpressed_cb(void *data EINA_UNUSED, Evas_Object *obj EINA_UNUSED, void *event_info)
{
	/* If you need, you can get item's data. */
	// Elm_Object_Item *it = event_info;
	// item_data_s *id = elm_object_item_data_get(it);
}

static void
gl_selected_cb(void *data, Evas_Object *obj EINA_UNUSED, void *event_info)
{
	Evas_Object *nf = (Evas_Object *)data, *label;
	Elm_Object_Item *it = event_info;
	item_data_s *id = elm_object_item_data_get(it);

	/* Unhighlight item */
	elm_genlist_item_selected_set(id->item, EINA_FALSE);

	/* Create second view */
	label = create_label(nf, "Second View");
	elm_naviframe_item_push(nf, "Second View", NULL, NULL, label, NULL);
}

static void
gl_contracted_cb(void *data EINA_UNUSED, Evas_Object *obj EINA_UNUSED, void *event_info)
{
   Elm_Object_Item *it = event_info;
   elm_genlist_item_subitems_clear(it);
}

static Evas_Object *
create_genlist(Evas_Object *parent)
{
	Evas_Object *genlist;
	Elm_Object_Item *git, *it;
	int n_items = NUM_OF_ITEMS;
	int index;

	/* Create item class */
	Elm_Genlist_Item_Class *gitc = elm_genlist_item_class_new();
	Elm_Genlist_Item_Class *itc = elm_genlist_item_class_new();

	gitc->item_style = "groupindex";
	gitc->func.text_get = gl_text_get_cb;
	gitc->func.content_get = gl_content_get_cb;
	gitc->func.del = gl_del_cb;

	itc->item_style = "1line";
	itc->func.text_get = gl_text_get_cb;
	itc->func.content_get = gl_content_get_cb;
	itc->func.del = gl_del_cb;

	genlist = elm_genlist_add(parent);

	elm_scroller_single_direction_set(genlist, ELM_SCROLLER_SINGLE_DIRECTION_HARD);

	/* HOMOGENEOUS MODE
		If item height is same when each style name is same, Use homogeneous mode. */
	elm_genlist_homogeneous_set(genlist, EINA_TRUE);

	/* COMPRESS MODE
		For the Mobile view, because it has full window, compresse mode should be used. */
	elm_genlist_mode_set(genlist, ELM_LIST_COMPRESS);

	/* Add Smart Callback */
	evas_object_smart_callback_add(genlist, "realized", gl_realized_cb, NULL);
	evas_object_smart_callback_add(genlist, "loaded", gl_loaded_cb, NULL);
	evas_object_smart_callback_add(genlist, "longpressed", gl_longpressed_cb, NULL);
	evas_object_smart_callback_add(genlist, "contracted", gl_contracted_cb, NULL);

	for (index = 0; index < n_items; index++) {
		item_data_s *id = calloc(sizeof(item_data_s), 1);
		id->index = index;
		if ((index == 0) || (index == 5) || (index == 8) || (index == 11) || (index == 14)) {
			git = elm_genlist_item_append(genlist,							/* genlist object */
											gitc,							/* item class */
											id,								/* item class user data */
											NULL,							/* parent item */
											ELM_GENLIST_ITEM_TREE,			/* item type */
											NULL,							/* select smart callback */
											id);							/* smart callback user data */
			id->item = git;
		}
		else {
			id->index = index;
			it = elm_genlist_item_append(genlist,							/* genlist object */
											itc,							/* item class */
											id,								/* item class user data */
											git,							/* parent item */
											ELM_GENLIST_ITEM_NONE,			/* item type */
											gl_selected_cb,					/* select smart callback */
											parent);						/* smart callback user data */
			id->item = it;
		}
	}
	elm_genlist_item_class_free(gitc);
	elm_genlist_item_class_free(itc);
	evas_object_show(genlist);

	return genlist;
}

static Evas_Object *
create_label(Evas_Object *parent, char *text)
{
	Evas_Object *label;
	label = elm_label_add(parent);

	if (text) {
		elm_object_text_set(label, text);
	}
	evas_object_show(label);

	return label;
}

static void
popup_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	Evas_Object *label;
	Elm_Object_Item *nf_it = elm_naviframe_top_item_get(ad->nf);
	const char *text = elm_object_item_text_get((Elm_Object_Item *) event_info);
	const char *title = elm_object_item_text_get(nf_it);

	evas_object_del(ad->popup);
	ad->popup = NULL;

	if (text && !strcmp(text, "Second View")) {
		if (title && strcmp(title, text)) {

			label = create_label(ad->nf, "Second View");
			elm_naviframe_item_push(ad->nf, "Second View", NULL, NULL, label, NULL);
		}
	} else if (text && !strcmp(text, "Settings")) {
		if (title && strcmp(title, text)) {
			elm_naviframe_item_pop(ad->nf);
		}
	}
}

static void
dismissed_popup_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	evas_object_del(obj);
	ad->popup = NULL;
}

static void
move_menu_popup(Evas_Object *parent, Evas_Object *obj)
{
	Evas_Coord w, h;
	int pos = -1;

	elm_win_screen_size_get(parent, NULL, NULL, &w, &h);
	pos = elm_win_rotation_get(parent);

	switch (pos) {
		case 0:
		case 180:
			evas_object_move(obj, 0, h);
			break;
		case 90:
			evas_object_move(obj, 0, w);
			break;
		case 270:
			evas_object_move(obj, h, w);
			break;
	}
}

static Evas_Object *
create_menu_popup(appdata_s *ad)
{
	Evas_Object *popup;

	popup = elm_ctxpopup_add(ad->win);
	elm_object_style_set(popup, "more/default");
	elm_ctxpopup_auto_hide_disabled_set(popup, EINA_TRUE);
	evas_object_smart_callback_add(popup, "dismissed", dismissed_popup_cb, ad);

	elm_ctxpopup_item_append(popup, "Settings", NULL, popup_cb, ad);
	elm_ctxpopup_item_append(popup, "Second View", NULL, popup_cb, ad);

	move_menu_popup(ad->win, popup);
	evas_object_show(popup);

	return popup;
}

static void
create_base_gui(appdata_s *ad)
{
	Evas_Object *bg;
	Elm_Object_Item *nf_it;

	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_conformant_set(ad->win, EINA_TRUE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	if (elm_win_wm_rotation_supported_get(ad->win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(ad->win, (const int *)(&rots), 4);
	}

	evas_object_smart_callback_add(ad->win, "delete,request", win_delete_request_cb, NULL);

	/* Conformant */
	ad->conform =  elm_conformant_add(ad->win);
	elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
	elm_win_resize_object_add(ad->win, ad->conform);
	evas_object_show(ad->conform);

	/* Indicator BG */
	bg = elm_bg_add(ad->conform);
	elm_object_style_set(bg, "indicator/headerbg");
	elm_object_part_content_set(ad->conform, "elm.swallow.indicator_bg", bg);
	evas_object_show(bg);

	/* Naviframe */
	ad->nf = elm_naviframe_add(ad->conform);
	eext_object_event_callback_add(ad->nf, EEXT_CALLBACK_BACK, nf_back_cb, ad);
	eext_object_event_callback_add(ad->nf, EEXT_CALLBACK_MORE, nf_more_cb, ad);
	elm_object_content_set(ad->conform, ad->nf);
	evas_object_show(ad->nf);

	/* Genlist */
	ad->genlist = create_genlist(ad->nf);

	nf_it = elm_naviframe_item_push(ad->nf, "Settings", NULL, NULL, ad->genlist, NULL);
	elm_naviframe_item_pop_cb_set(nf_it, nf_it_pop_cb, ad);

	/* Show window after base gui is set up */
	evas_object_show(ad->win);
}

static bool
app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
		Initialize UI resources and application's data
		If this function returns true, the main loop of application starts
		If this function returns false, the application is terminated */
	appdata_s *ad = data;

	create_base_gui(ad);

	return true;
}

static void
app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
	/* Release all resources. */
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

static void
ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_DEVICE_ORIENTATION_CHANGED*/
	return;
}

static void
ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_REGION_FORMAT_CHANGED*/
}

static void
ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_BATTERY*/
}

static void
ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_orient_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_changed, &ad);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() is failed. err = %d", ret);
	}

	return ret;
}
