/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "vc-elm.h"

/* Command and hint names */
const char *first_view_cmd[] =
	{"button", "yes", "no", "check", "switch", "horizontal", "vertical"};

const char *tabbar_view_cmd[] =
	{"first", "second"};

static int val;

/* vc_elm_h objects used to create commands and hints for widgets */
static vc_elm_h first_view[7];
static vc_elm_h tabbar_view[2];

/* created in another file, but destroyed in naviframe pop callback */
vc_elm_h vertical;
vc_elm_h horizontal;
vc_elm_h genlist_h;

static Evas_Object *
create_basic(appdata_s *ad);

static Evas_Object *
create_toolbar(appdata_s *ad);

static void win_delete_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

static Eina_Bool nf_it_pop_cb(void *data, Elm_Object_Item *it)
{
	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_lower(ad->win);
	return EINA_FALSE;
}

static void nf_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	Elm_Object_Item *nf_it = elm_naviframe_top_item_get(ad->nf);
	const char *title = elm_object_item_text_get(nf_it);
	int index = 0;

	if (title == NULL) {
		/* Basic Widgets view disappears so delete all vc_elm_h objects */
		for (index = 0; index < 7; ++index)
			vc_elm_destroy(first_view[index]);
		/* More Widgets view disappears so delete all vc_elm_h object */
		for (index = 0; index < 2; ++index)
			vc_elm_destroy(tabbar_view[index]);
	}

	elm_naviframe_item_pop(obj);
}

static void gl_toolbar_cb(void *data, Evas_Object *obj EINA_UNUSED, void *event_info EINA_UNUSED)
{
	appdata_s *ad = (appdata_s *)data;
	Evas_Object *nf = ad->nf, *toolbar;
	Elm_Object_Item *nf_it;

	nf_it = elm_naviframe_item_push(nf, NULL, NULL, NULL, NULL, "tabbar/icon/notitle");
	elm_naviframe_item_pop_cb_set(nf_it, nf_it_pop_cb, ad);
	/* Create second view */
	toolbar = create_toolbar(ad);
	elm_object_item_part_content_set(nf_it, "tabbar", toolbar);
}

/* Function responsible for switching content in toolbar */
static void toolbar_it_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	Evas_Object *sub_view;
	Elm_Object_Item *it;
	const char *str = NULL;

	it = event_info;

	str = elm_object_item_text_get(it);

	if (str == NULL || !strcmp(str, "First")) {
		sub_view = create_basic(ad);
	} else if (!strcmp(str, "Second")) {
		sub_view = create_genlist(ad);
	} else {
		sub_view = create_basic(ad);
	}

	elm_object_content_set(ad->nf, sub_view);
}

/* Create toolbar */
static Evas_Object *
create_toolbar(appdata_s *ad)
{
	Evas_Object *toolbar;
	Elm_Object_Item *it;
	toolbar = elm_toolbar_add(ad->nf);
	elm_object_style_set(toolbar, "tabbar");
	elm_toolbar_shrink_mode_set(toolbar, ELM_TOOLBAR_SHRINK_EXPAND);
	elm_toolbar_transverse_expanded_set(toolbar, EINA_TRUE);

	/* Create first tab, add command and hint */
	it = elm_toolbar_item_append(toolbar, NULL, "First", toolbar_it_cb, ad);
	vc_elm_create_item(it, &tabbar_view[0]);
	vc_elm_set_command(tabbar_view[0], tabbar_view_cmd[0]);
	vc_elm_set_command_hint(tabbar_view[0], tabbar_view_cmd[0]);

	/* Create second tab, add command and hint */
	it = elm_toolbar_item_append(toolbar, NULL, "Second", toolbar_it_cb, ad);
	vc_elm_create_item(it, &tabbar_view[1]);
	vc_elm_set_command(tabbar_view[1], tabbar_view_cmd[1]);
	vc_elm_set_command_hint(tabbar_view[1], tabbar_view_cmd[1]);

	elm_toolbar_select_mode_set(toolbar, ELM_OBJECT_SELECT_MODE_ALWAYS);

	return toolbar;
}

/* Create UI with voice commands and hints */
static Evas_Object *
create_basic(appdata_s *ad)
{
	Evas_Object *bg;
	Evas_Object *main_box;
	Evas_Object *box_horiz;
	Evas_Object *button;
	Evas_Object *radio_group;
	Evas_Object *radio;
	Evas_Object *sliderH;
	Evas_Object *sliderV;
	Evas_Object *check;
	Evas_Object *switcher;
	double min;
	double max;

	bg = elm_bg_add(ad->win);
	evas_object_size_hint_align_set(bg, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_show(bg);

	main_box = elm_box_add(ad->win);
	elm_object_content_set(bg, main_box);
	elm_box_horizontal_set(main_box, EINA_FALSE);
	evas_object_size_hint_align_set(main_box, EVAS_HINT_FILL, EVAS_HINT_FILL);

	/* Create button */
	button = elm_button_add(ad->win);
	elm_object_text_set(button, "Regular Button");
	elm_box_pack_end(main_box, button);
	evas_object_size_hint_align_set(button, EVAS_HINT_FILL, 0.5);
	evas_object_size_hint_weight_set(button, EVAS_HINT_EXPAND, 1.0);
	evas_object_show(button);

	/* Create voice-control-elm object and set command with hint for button */
	vc_elm_create_object(button, &first_view[0]);
	vc_elm_set_command(first_view[0], first_view_cmd[0]);
	vc_elm_set_command_hint(first_view[0], first_view_cmd[0]);

	/* Create horizontal box to hold radio buttons next to one another */
	box_horiz = elm_box_add(ad->win);
	elm_box_horizontal_set(box_horiz, EINA_TRUE);
	elm_box_pack_end(main_box, box_horiz);
	evas_object_size_hint_align_set(box_horiz, EVAS_HINT_FILL, 0.5);
	evas_object_size_hint_weight_set(box_horiz, EVAS_HINT_EXPAND, 1.0);
	evas_object_show(box_horiz);

	/* Create radio group and first radio button */
	radio_group = radio = elm_radio_add(ad->win);
	elm_object_text_set(radio, "Radio 1");
	elm_radio_state_value_set(radio, 0);
	elm_radio_value_pointer_set(radio, &val);
	elm_box_pack_end(box_horiz, radio);

	/* Set command for first radio button and move hint to left */
	vc_elm_create_object(radio, &first_view[1]);
	vc_elm_set_command(first_view[1], first_view_cmd[1]);
	vc_elm_set_command_hint(first_view[1], first_view_cmd[1]);
	vc_elm_set_command_hint_direction(first_view[1], VC_ELM_DIRECTION_LEFT);

	/* Create second radio button for the same group */
	radio = elm_radio_add(ad->win);
	elm_radio_group_add(radio, radio_group);
	elm_object_text_set(radio, "Radio 2");
	elm_radio_state_value_set(radio, 1);
	elm_radio_value_pointer_set(radio, &val);
	elm_box_pack_end(box_horiz, radio);

	/* Set command for second radio button and move hint to right */
	vc_elm_create_object(radio, &first_view[2]);
	vc_elm_set_command(first_view[2], first_view_cmd[2]);
	vc_elm_set_command_hint(first_view[2], first_view_cmd[2]);
	vc_elm_set_command_hint_direction(first_view[2], VC_ELM_DIRECTION_RIGHT);

	evas_object_show(radio);
	evas_object_show(radio_group);

	/* Create checkbox widget */
	check = elm_check_add(ad->win);
	elm_box_pack_end(main_box, check);
	evas_object_size_hint_weight_set(check, EVAS_HINT_EXPAND, 1.0);
	evas_object_show(check);

	/* Set command and hint for checkbox */
	vc_elm_create_object(check, &first_view[3]);
	vc_elm_set_command(first_view[3], first_view_cmd[3]);
	vc_elm_set_command_hint(first_view[3], first_view_cmd[3]);

	/* Create checkbox widget with on&off style (switch) */
	switcher = elm_check_add(ad->win);
	elm_object_style_set(switcher, "on&off");
	elm_box_pack_end(main_box, switcher);
	evas_object_size_hint_weight_set(switcher, EVAS_HINT_EXPAND, 1.0);
	evas_object_show(switcher);

	/* Set command and hint for switch button */
	vc_elm_create_object(switcher, &first_view[4]);
	vc_elm_set_command(first_view[4], first_view_cmd[4]);
	vc_elm_set_command_hint(first_view[4], first_view_cmd[4]);

	/* Create slider widget aligned horizontaly */
	sliderH = elm_slider_add(ad->win);
	elm_slider_horizontal_set(sliderH, EINA_TRUE);
	elm_slider_min_max_get(sliderH, &min, &max);
	elm_slider_value_set(sliderH, max / 2);
	evas_object_size_hint_align_set(sliderH, EVAS_HINT_FILL, 0.5);
	evas_object_size_hint_weight_set(sliderH, EVAS_HINT_EXPAND, 1.0);
	elm_box_pack_end(main_box, sliderH);
	evas_object_show(sliderH);

	/* Set command and hint for horizontal slider and move hint to top */
	vc_elm_create_object(sliderH, &first_view[5]);
	vc_elm_set_command(first_view[5], first_view_cmd[5]);
	vc_elm_set_command_hint(first_view[5], first_view_cmd[5]);
	vc_elm_set_command_hint_direction(first_view[5], VC_ELM_DIRECTION_TOP);

	/* Create slider widget aligned vertically */
	sliderV = elm_slider_add(ad->win);
	elm_slider_horizontal_set(sliderV, EINA_FALSE);
	evas_object_size_hint_align_set(sliderV, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(sliderV, EVAS_HINT_EXPAND, 1.0);
	elm_box_pack_end(main_box, sliderV);
	evas_object_show(sliderV);

	/* Set command and hint for vertical slider */
	vc_elm_create_object(sliderV, &first_view[6]);
	vc_elm_set_command(first_view[6], first_view_cmd[6]);
	vc_elm_set_command_hint(first_view[6], first_view_cmd[6]);

	evas_object_show(main_box);

	return bg;
}

static void create_base_gui(appdata_s *ad)
{
	Evas_Object *bg;

	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_conformant_set(ad->win, EINA_TRUE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	if (elm_win_wm_rotation_supported_get(ad->win)) {
		int rots[4] =
			{0, 90, 180, 270};
		elm_win_wm_rotation_available_rotations_set(ad->win, (const int *)(&rots), 4);
	}

	evas_object_smart_callback_add(ad->win, "delete,request", win_delete_request_cb, NULL);

	/* Conformant */
	ad->conform = elm_conformant_add(ad->win);
	elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
	elm_win_resize_object_add(ad->win, ad->conform);
	evas_object_show(ad->conform);

	/* Indicator BG */
	bg = elm_bg_add(ad->conform);
	elm_object_style_set(bg, "indicator/headerbg");
	elm_object_part_content_set(ad->conform, "elm.swallow.indicator_bg", bg);
	evas_object_show(bg);

	/* Naviframe */
	ad->nf = elm_naviframe_add(ad->conform);
	eext_object_event_callback_add(ad->nf, EEXT_CALLBACK_BACK, nf_back_cb, ad);
	/*eext_object_event_callback_add(ad->nf, EEXT_CALLBACK_MORE, nf_more_cb, ad);*/
	elm_object_content_set(ad->conform, ad->nf);
	evas_object_show(ad->nf);

	gl_toolbar_cb((void *)ad, NULL, NULL);

	/* Show window after base gui is set up */
	evas_object_show(ad->win);
}

static bool app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
	Initialize UI resources and application's data
	If this function returns true, the main loop of application starts
	If this function returns false, the application is terminated */
	appdata_s *ad = data;

	vc_elm_initialize();
	create_base_gui(ad);

	return true;
}

static void app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void app_terminate(void *data)
{
	/* Release all resources. */
	vc_elm_deinitialize();
}

static void ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

static void ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_DEVICE_ORIENTATION_CHANGED*/
	return;
}

static void ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_REGION_FORMAT_CHANGED*/
}

static void ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_BATTERY*/
}

static void ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}

int main(int argc, char *argv[])
{
	appdata_s ad =
		{0, };
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback =
		{0, };
	app_event_handler_h handlers[5] =
		{NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_orient_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_changed, &ad);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() is failed. err = %d", ret);
	}

	return ret;
}
