/*
 * Samsung API
 * Copyright (c) 2009-2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "main.h"

typedef struct appdata {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *naviframe;
	Evas_Object *genlist;

	Elm_Object_Item *command;
	Elm_Object_Item *result;
} appdata_s;

typedef enum {
	SET_COMMAND_TITLE = 0,
	SELECT_COMMAND_BUTTON,
	RESULT_TITLE,
	RESULT_BUTTON
} item_index_e;

typedef enum {
	MP3 = 0,
	MEMO,
	CAMERA
} cmd_list_index_e;

static Elm_Genlist_Item_Class *g_itc_group_title = NULL;
static Elm_Genlist_Item_Class *g_itc_button_1line = NULL;
static Elm_Genlist_Item_Class *g_itc_button_2line = NULL;

#define MAX_COMMAND_NUM 4

static char * g_player_command[MAX_COMMAND_NUM] = {"play", "stop", "pause", "rewind"};
static char * g_memo_command[MAX_COMMAND_NUM] = {"copy", "paste", "underline", "italic"};
static char * g_camera_command[MAX_COMMAND_NUM] = {"capture", "zoom in", "zoom out", "gallery"};

static vc_cmd_list_h g_cmd_list;
static char *g_result = NULL;
static int current_cmd_list_idx = -1;

static Eina_Bool
__naviframe_item_pop_cb(void *data, Elm_Object_Item *it)
{
	dlog_print(DLOG_DEBUG, LOG_TAG, "");
	ui_app_exit();
	return EINA_TRUE;
}

static char *
__genlist_text_get(void *data, Evas_Object *obj, const char *part)
{
	item_index_e idx = (item_index_e *)data;

	if (SELECT_COMMAND_BUTTON == idx) {
		if (!strcmp("elm.text", part)) {
			return strdup("Select Command List");
		} else if (!strcmp("elm.text.sub", part)) {
			char command[64] = {'\0',};
			if (MP3 == current_cmd_list_idx) {
				snprintf(command, 64, "%s, %s, %s, %s", g_player_command[0], g_player_command[1], g_player_command[2], g_player_command[3]);
			} else if (MEMO == current_cmd_list_idx) {
				snprintf(command, 64, "%s, %s, %s, %s", g_memo_command[0], g_memo_command[1], g_memo_command[2], g_memo_command[3]);
			} else if (CAMERA == current_cmd_list_idx) {
				snprintf(command, 64, "%s, %s, %s, %s", g_camera_command[0], g_camera_command[1], g_camera_command[2], g_camera_command[3]);
			} else {
				snprintf(command, 64, "%s", "No commands");
			}
			return strdup(command);
		}
	} else if (!strcmp("elm.text", part)) {
		if (RESULT_BUTTON == idx) {
			char command[64] = {'\0',};
			if (NULL == g_result)
				snprintf(command, 64, "%s", "No result");
			else
				snprintf(command, 64, "%s", g_result);

			if (NULL != g_result) {
				free(g_result);
				g_result = NULL;
			}
			return strdup(command);
		}
	} else if (!strcmp("elm.text.main", part)) {
		if (SET_COMMAND_TITLE == idx) {
			return strdup("Set Command");
		} else if (RESULT_TITLE == idx) {
			return strdup("Result");
		}
	}

	return NULL;
}

static Eina_Bool
__naviframe_sub_item_pop_cb(void *data, Elm_Object_Item *it)
{
	dlog_print(DLOG_DEBUG, LOG_TAG, "");
	return EINA_TRUE;
}

static void
__list_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	elm_naviframe_item_pop(ad->naviframe);
	elm_genlist_item_update(ad->command);
}

static void
__vc_destroy_command_list(void)
{
	dlog_print(DLOG_DEBUG, LOG_TAG, "Destroy Command List");

	if (0 != vc_cmd_list_destroy(g_cmd_list, true)) {
		dlog_print(DLOG_DEBUG, LOG_TAG, "[WARNING] Fail to destroy list");
	}
}

static void
__vc_create_command_list(cmd_list_index_e cmd_list_idx)
{
	dlog_print(DLOG_DEBUG, LOG_TAG, "Create Command List");

	if (0 != vc_cmd_list_create(&g_cmd_list)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[ERROR] Fail to cmd list create");
	}

	char *temp[MAX_COMMAND_NUM];

	if (MP3 == cmd_list_idx)
		memcpy(temp, g_player_command, sizeof(g_player_command));
	else if (MEMO == cmd_list_idx)
		memcpy(temp, g_memo_command, sizeof(g_memo_command));
	else
		memcpy(temp, g_camera_command, sizeof(g_camera_command));

	int i;
	for (i = 0; i < MAX_COMMAND_NUM; i++) {
		vc_cmd_h cmd;
		if (0 != vc_cmd_create(&cmd)) {
			dlog_print(DLOG_ERROR, LOG_TAG, "[ERROR] Fail to cmd create");
		}
		if (0 != vc_cmd_set_command(cmd, _(temp[i]))) {
			dlog_print(DLOG_ERROR, LOG_TAG, "[ERROR] Fail to set command");
			vc_cmd_destroy(cmd);
		}
		if (0 != vc_cmd_set_type(cmd, VC_COMMAND_TYPE_FOREGROUND)) {
			dlog_print(DLOG_ERROR, LOG_TAG, "[ERROR] Fail to set type");
			vc_cmd_destroy(cmd);
		}
		if (0 != vc_cmd_list_add(g_cmd_list, cmd)) {
			dlog_print(DLOG_ERROR, LOG_TAG, "[ERROR] Fail to list add");
			vc_cmd_destroy(cmd);
		}
	}

	if ( 0 != vc_set_command_list(g_cmd_list, VC_COMMAND_TYPE_FOREGROUND)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[ERROR] Fail to set command");
		vc_cmd_list_destroy(g_cmd_list, true);
	}
}


static void
__command_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	Elm_Object_Item *it = event_info;
	elm_list_item_selected_set(it, EINA_FALSE);

	cmd_list_index_e cmd_list_idx = (int *)data;

	current_cmd_list_idx = cmd_list_idx;
	__vc_create_command_list(cmd_list_idx);
}

static void
__show_command_list(appdata_s *ad)
{
	Evas_Object *list = elm_list_add(ad->naviframe);
	elm_list_mode_set(list, ELM_LIST_COMPRESS);
	evas_object_smart_callback_add(list, "selected", __list_clicked_cb, ad);

	char command[64] = {'\0',};
	snprintf(command, 64, "%s, %s, %s, %s", g_player_command[0], g_player_command[1], g_player_command[2], g_player_command[3]);
	elm_list_item_append(list, command, NULL, NULL, __command_selected_cb, (void *)0);
	snprintf(command, 64, "%s, %s, %s, %s", g_memo_command[0], g_memo_command[1], g_memo_command[2], g_memo_command[3]);
	elm_list_item_append(list, command, NULL, NULL, __command_selected_cb, (void *)1);
	snprintf(command, 64, "%s, %s, %s, %s", g_camera_command[0], g_camera_command[1], g_camera_command[2], g_camera_command[3]);
	elm_list_item_append(list, command, NULL, NULL, __command_selected_cb, (void *)2);

	elm_list_go(list);

	Elm_Object_Item *voice_item = elm_naviframe_item_push(ad->naviframe, "Voice", NULL, NULL, list, NULL);
	elm_naviframe_item_pop_cb_set(voice_item, __naviframe_sub_item_pop_cb, ad);
}

static void
__select_command_item_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	Elm_Object_Item *item = (Elm_Object_Item *)event_info;
	elm_genlist_item_selected_set(item, EINA_FALSE);

	appdata_s *ad = (appdata_s *)data;
	__show_command_list(ad);

	elm_genlist_item_update(item);
}

static void
__result_item_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	Elm_Object_Item *item = (Elm_Object_Item *)event_info;

	elm_genlist_item_selected_set(item, EINA_FALSE);

	elm_genlist_item_update(item);
}

static void
create_contents(appdata_s *ad)
{
	/* Genlist */
	ad->genlist = elm_genlist_add(ad->naviframe);
	elm_genlist_mode_set(ad->genlist, ELM_LIST_COMPRESS);
	elm_genlist_homogeneous_set(ad->genlist, EINA_TRUE);

	g_itc_group_title = elm_genlist_item_class_new();
	if (NULL == g_itc_group_title) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Fail to item class new");
		return;
	}
	g_itc_group_title->item_style = "groupindex";
	g_itc_group_title->func.text_get = __genlist_text_get;
	g_itc_group_title->func.content_get = NULL;
	g_itc_group_title->func.state_get = NULL;
	g_itc_group_title->func.del = NULL;

	g_itc_button_1line = elm_genlist_item_class_new();
	if (NULL == g_itc_button_1line) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Fail to item class new");
		return;
	}
	g_itc_button_1line->item_style = "type1";
	g_itc_button_1line->func.text_get = __genlist_text_get;
	g_itc_button_1line->func.content_get = NULL;
	g_itc_button_1line->func.state_get = NULL;
	g_itc_button_1line->func.del = NULL;

	g_itc_button_2line = elm_genlist_item_class_new();
	if (NULL == g_itc_button_2line) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Fail to item class new");
		return;
	}
	g_itc_button_2line->item_style = "type1";
	g_itc_button_2line->func.text_get = __genlist_text_get;
	g_itc_button_2line->func.content_get = NULL;
	g_itc_button_2line->func.state_get = NULL;
	g_itc_button_2line->func.del = NULL;

	/* Set Command - Title */
	elm_genlist_item_append(ad->genlist, g_itc_group_title, (void *)SET_COMMAND_TITLE, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);

	/* Select command - 2line */
	ad->command = elm_genlist_item_append(ad->genlist, g_itc_button_2line, (void *)SELECT_COMMAND_BUTTON, NULL, ELM_GENLIST_ITEM_NONE, __select_command_item_selected_cb, ad);

	/* Result - Title */
	elm_genlist_item_append(ad->genlist, g_itc_group_title, (void *)RESULT_TITLE, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);

	/* Result - button */
	ad->result = elm_genlist_item_append(ad->genlist, g_itc_button_1line, (void *)RESULT_BUTTON, NULL, ELM_GENLIST_ITEM_NONE, __result_item_selected_cb, ad);

	elm_genlist_select_mode_set(ad->genlist, ELM_OBJECT_SELECT_MODE_DEFAULT);

	evas_object_show(ad->genlist);
}

static void
create_base_gui(appdata_s *ad)
{
	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	if (elm_win_wm_rotation_supported_get(ad->win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(ad->win, (const int *)(&rots), 4);
	}

	/* Conformant */
	ad->conform = elm_conformant_add(ad->win);
	elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
	evas_object_size_hint_weight_set(ad->conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(ad->win, ad->conform);
	evas_object_show(ad->conform);

	/* Naviframe */
	ad->naviframe = elm_naviframe_add(ad->conform);
	eext_object_event_callback_add(ad->naviframe, EEXT_CALLBACK_BACK, eext_naviframe_back_cb, NULL);
	evas_object_size_hint_weight_set(ad->naviframe, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(ad->naviframe, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_object_part_content_set(ad->conform, "elm.swallow.content", ad->naviframe);
	evas_object_show(ad->naviframe);

	/* Contents */
	create_contents(ad);
	Elm_Object_Item *main_item = elm_naviframe_item_push(ad->naviframe, "Voice Control", NULL, NULL, ad->genlist, NULL);
	elm_naviframe_item_title_enabled_set(main_item, EINA_TRUE, EINA_TRUE);
	elm_naviframe_item_pop_cb_set(main_item, __naviframe_item_pop_cb, ad);
	elm_object_content_set(ad->conform, ad->naviframe);

	/* Show window after base gui is set up */
	evas_object_show(ad->win);
}

static void
__vc_service_state_changed_cb(vc_service_state_e previous, vc_service_state_e current, void* user_data)
{
	appdata_s *ad = (appdata_s *)user_data;

	if (VC_SERVICE_STATE_READY == current) {
		elm_genlist_select_mode_set(ad->genlist, ELM_OBJECT_SELECT_MODE_DEFAULT);
	} else {
		elm_genlist_select_mode_set(ad->genlist, ELM_OBJECT_SELECT_MODE_NONE);
	}
}

static void
__vc_result_cb(vc_result_event_e event, vc_cmd_list_h vc_cmd_list, const char* result, void *user_data)
{
	appdata_s *ad = (appdata_s *)user_data;

	int len = 0;
	len = strlen(result);
	g_result = (char*)calloc(len, sizeof(char));
	strncpy(g_result, result, len);

	elm_genlist_item_update(ad->result);
}

static int
init_vc(appdata_s *ad)
{
	int ret;
	dlog_print(DLOG_DEBUG, LOG_TAG, "Initialize");

	ret = vc_initialize();
	if (VC_ERROR_NONE != ret) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Fail to initialize");
		return -1;
	}

	ret = vc_set_result_cb(__vc_result_cb, ad);
	if (VC_ERROR_NONE != ret) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Fail to set all result cb");
		return -1;
	}

	ret = vc_set_service_state_changed_cb(__vc_service_state_changed_cb, ad);
	if (VC_ERROR_NONE != ret) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Fail to set service state changed cb");
		return -1;
	}

	ret = vc_prepare();
	if (VC_ERROR_NONE != ret) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Fail to prepare");
		return -1;
	}

	return 0;
}

static void
deinit_vc(appdata_s *ad)
{
	__vc_destroy_command_list();

	if (0 != vc_deinitialize()) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Fail to deinitialize");
	}
}

static bool
app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
		Initialize UI resources and application's data
		If this function returns true, the main loop of application starts
		If this function returns false, the application is terminated */
	appdata_s *ad = data;

	create_base_gui(ad);

	if (0 != init_vc(ad)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Fail to init vc");
		return false;
	}

	return true;
}

static void
app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
	appdata_s *ad = data;
	/* Release all resources. */
	deinit_vc(ad);
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

static void
ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_DEVICE_ORIENTATION_CHANGED*/
	return;
}

static void
ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_REGION_FORMAT_CHANGED*/
}

static void
ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_BATTERY*/
}

static void
ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_orient_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_changed, &ad);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_main() is failed. err = %d", ret);
	}

	return ret;
}
