#include <tizen.h>
#include "ship.h"

typedef struct widget_instance_data {
	Evas_Object *win;
	Evas_Object *bg;
	Evas_Object *conformant;
	Evas_Object *ship;
} widget_instance_data_s;

#define ANIM_TIME 100  // in seconds
#define MAX_ROTATION_ANGLE 180

static Ecore_Pos_Map animation_type;
static int screen_width;
static int screen_height;

static int
calculate_x_position(double frame) {
	return 0.25 * screen_width * atan((0.5 - frame) * 1.2 * M_PI)
			+ 0.45 * screen_width;
}

static int
calculate_y_position(double frame) {
	return -0.75 * screen_height * tan((0.5 - frame)) + 0.45 * screen_height;
}

static int
calculate_rotation(double frame) {
	return MAX_ROTATION_ANGLE * cos((0.5 - frame) * M_PI);
}

static void
evas_object_rotate(Evas_Object *object, double degree) {
	Evas_Map *m;
	Evas_Coord x, y, w, h;
	m = evas_map_new(4);
	evas_object_geometry_get(object, &x, &y, &w, &h);
	evas_map_util_points_populate_from_object(m, object);
	evas_map_util_rotate(m, degree, x + (w / 2), y + (h / 2));
	evas_object_map_set(object, m);
	evas_object_map_enable_set(object, EINA_TRUE);
	evas_map_free(m);
}

static Eina_Bool
_do_animation(void *data, double pos) {
	double frame = pos;
	int x, y;
	double rotation_angle;
	animation_type = ECORE_POS_MAP_LINEAR;

	frame = ecore_animator_pos_map(pos, animation_type, 0.0, 0.0);

	/* calculate position on curve  */
	x = calculate_x_position(frame);
	y = calculate_y_position(frame);
	rotation_angle = calculate_rotation(frame);

	evas_object_rotate(data, rotation_angle);

	return EINA_TRUE;
}

static int
widget_instance_create(widget_context_h context, bundle *content,
		int w, int h, void *user_data) {
	widget_instance_data_s *wid = (widget_instance_data_s*) malloc(
			sizeof(widget_instance_data_s));
	int ret;

	if (content != NULL) {
		/* Recover the previous status with the bundle object. */

	}

	/* Window */
	ret = widget_app_get_elm_win(context, &wid->win);
	if (ret != WIDGET_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to get window. err = %d", ret);
		return WIDGET_ERROR_FAULT;
	}

	evas_object_resize(wid->win, w, h);

	// Label
	Evas_Object* lb = elm_label_add(wid->win);
	elm_object_text_set(lb, "<center>Animation</center>");
	evas_object_move(lb, 0, 0);
	evas_object_resize(lb, w, 50);
	evas_object_show(lb);

	// Animation
	wid->ship = evas_object_image_filled_add(evas_object_evas_get(wid->win));

	char buf[256];

	snprintf(buf, sizeof(buf), "%s/ship.png", app_get_resource_path());
	evas_object_image_file_set(wid->ship, buf, NULL);
	evas_object_resize(wid->ship, w, h);
	evas_object_move(wid->ship, 0, 0);
	evas_object_show(wid->ship);

	Ecore_Animator *animator = ecore_animator_timeline_add(ANIM_TIME,
			_do_animation, wid->ship);

	/* Show window after base gui is set up */
	evas_object_show(wid->win);

	widget_app_context_set_tag(context, wid);
	return WIDGET_ERROR_NONE;
}

static int
widget_instance_destroy(widget_context_h context,
		widget_app_destroy_type_e reason, bundle *content_info, void *user_data) {
	widget_instance_data_s *wid = NULL;
	widget_app_context_get_tag(context, (void**) &wid);
	evas_object_del(wid->win);
	free(wid);

	return WIDGET_ERROR_NONE;
}

static int
widget_instance_pause(widget_context_h context, void *user_data) {
	/* Take necessary actions when widget instance becomes invisible. */
	return WIDGET_ERROR_NONE;

}

static int
widget_instance_resume(widget_context_h context, void *user_data) {
	/* Take necessary actions when widget instance becomes visible. */
	return WIDGET_ERROR_NONE;
}

static int
widget_instance_update(widget_context_h context, bundle *content,
		int force, void *user_data) {
	/* Take necessary actions when widget instance should be updated. */
	return WIDGET_ERROR_NONE;
}

static int
widget_instance_resize(widget_context_h context, int w, int h, void *user_data) {
	widget_instance_data_s *wid = NULL;

	widget_app_context_get_tag(context, (void**) &wid);
	evas_object_resize(wid->win, w, h);
	evas_object_resize(wid->bg, w, h);
	evas_object_resize(wid->conformant, w, h);

	return WIDGET_ERROR_NONE;
}

static void
widget_app_lang_changed(app_event_info_h event_info,
		void *user_data) {
	/* APP_EVENT_LANGUAGE_CHANGED */
	char *locale = NULL;
	app_event_get_language(event_info, &locale);
	elm_language_set(locale);
	free(locale);
}

static void
widget_app_region_changed(app_event_info_h event_info,
		void *user_data) {
	/* APP_EVENT_REGION_FORMAT_CHANGED */
}

static widget_class_h
widget_app_create(void *user_data) {
	app_event_handler_h handlers[5] = { NULL, };

	widget_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED],
			APP_EVENT_LANGUAGE_CHANGED, widget_app_lang_changed, user_data);
	widget_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED],
			APP_EVENT_REGION_FORMAT_CHANGED, widget_app_region_changed,
			user_data);

	widget_instance_lifecycle_callback_s ops = { .create =
			widget_instance_create, .destroy = widget_instance_destroy, .pause =
			widget_instance_pause, .resume = widget_instance_resume, .update =
			widget_instance_update, .resize = widget_instance_resize, };

	return widget_app_class_create(ops, user_data);
}

static void
widget_app_terminate(void *user_data) {
	/* Release all resources. */
}

int
main(int argc, char *argv[]) {
	widget_app_lifecycle_callback_s ops = { 0, };
	int ret;

	ops.create = widget_app_create;
	ops.terminate = widget_app_terminate;

	ret = widget_app_main(argc, argv, &ops, NULL);
	if (ret != WIDGET_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "widget_app_main() is failed. err = %d",
				ret);
	}

	return ret;
}

