/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <app_common.h>
#include <Elementary.h>
#include <efl_extension.h>
#include <notification.h>
#include <dlog.h>
#include "$(appName).h"
#include "data.h"

#define NOTIFICATION_DATA_NUM 3

/*
 * This sample provide three notification data.
 */
noti_data_s noti_data[NOTIFICATION_DATA_NUM] = {
	{
		"images/icon.png",
		"Geunsun Lee",
		"Lorem ipsum dolor sit amet, consectetur adipiscing elit. Aliquam sit amet ligula non libero sodales vestibulum vel non metus. Nullam vulputate posuere quam eget fermentum. In nec nulla enim. Proin dictum ultricies ex, quis vulputate sem sodales non. Pellentesque egestas mauris eu mauris bibendum, sed semper odio venenatis. Aenean dignissim magna feugiat, cursus ipsum et, aliquam risus. Cras sit amet metus tempus, aliquam purus finibus, placerat nulla. Suspendisse felis sapien, suscipit sed elementum a, vehicula non metus. Aliquam ullamcorper mauris quam, ut dignissim purus congue vel. Morbi congue eros nec neque dignissim, nec commodo turpis interdum. Praesent at sagittis tellus. Mauris dolor nibh, vestibulum nec sem vel, ultricies imperdiet magna. Maecenas lorem libero, tincidunt ac vehicula sodales, egestas non sapien. Nullam malesuada facilisis consectetur.",
		"12:01 AM",
	},
	{
		"images/icon.png",
		"Moonkyung Park",
		"Nunc eleifend commodo pellentesque. Sed maximus iaculis urna, in faucibus est faucibus non. Integer at libero nisi. In consectetur luctus ipsum sit amet lobortis. Aenean diam ipsum, pulvinar quis ante fringilla, cursus porta est. Aliquam maximus, nunc sed faucibus laoreet, lorem nibh tincidunt nisi, ac porta mauris leo eu arcu. Integer eros augue, porta tempus diam sit amet, pulvinar placerat augue. Nulla egestas sem eget felis gravida ultricies. Praesent a libero iaculis sem sagittis mollis. Quisque accumsan augue quam, vitae pretium ante sagittis sed.",
		"03:15 PM",
	},
	{
		"images/icon.png",
		"Junkyu Han",
		"Quisque et leo at nulla ultricies posuere. Maecenas dignissim mi eget aliquam ornare. Nulla lobortis sollicitudin egestas. In eget gravida ante. Aenean ac massa purus. Interdum et malesuada fames ac ante ipsum primis in faucibus. Donec enim elit, tincidunt efficitur enim a, finibus placerat libero. Duis ultricies cursus augue, id consequat tortor convallis viverra. Duis condimentum ultricies nibh, ut pulvinar magna congue in. Nam nulla felis, faucibus ut fermentum non, molestie sit amet sem. Sed facilisis facilisis tortor, vitae tincidunt lectus finibus quis. Fusce quis lorem et lacus pharetra porttitor. Mauris odio nulla, auctor eu venenatis lacinia, vestibulum ac tellus. Duis a purus massa.",
		"05:32 PM",
	},
};

static struct _s_info {
	Eina_List *noti_list;
} s_info = {
	.noti_list = NULL,
};

/**
 * @brief Initialization function for data module.
 */
void data_initialize(void)
{
	/*
	 * If you need to initialize managing data,
	 * please use this function.
	 */
	Eina_List *noti_list = NULL;
	int i = 0;

	for (i = 0; i < NOTIFICATION_DATA_NUM; i++) {
		noti_data_s *noti_item = malloc(sizeof(noti_data_s));

		noti_item->icon_path = strdup(noti_data[i].icon_path);
		noti_item->title = strdup(noti_data[i].title);
		noti_item->body_text = strdup(noti_data[i].body_text);
		noti_item->date = strdup(noti_data[i].date);

		noti_list = eina_list_append(noti_list, noti_item);
	}

	s_info.noti_list = noti_list;
}

/**
 * @brief Finalization function for data module.
 */
void data_finalize(void)
{
	/*
	 * If you need to finalize managing data,
	 * please use this function.
	 */
	noti_data_s *noti_item = NULL;

	if (s_info.noti_list == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "notification list is NULL");
		return;
	}

	EINA_LIST_FREE(s_info.noti_list, noti_item) {
		if (noti_item == NULL) {
			continue;
		}

		free(noti_item->icon_path);
		free(noti_item->title);
		free(noti_item->body_text);
		free(noti_item->date);
	}

	s_info.noti_list = NULL;
}

/**
 * @brief Gets the list that is consists of notification data.
 */
Eina_List *data_get_noti_list(void)
{
	return s_info.noti_list;
}

/**
 * @brief Gets path of resource.
 * @param[in] file_in File name
 * @param[in] file_path_out The point to which save full path of the resource
 * @param[in] file_path_max Size of file name include path
 */
void data_get_resource_path(const char *file_in, char *file_path_out, int file_path_max)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(file_path_out, file_path_max, "%s%s", res_path, file_in);
		free(res_path);
	}
}

/**
 * @brief Gets notification count.
 */
int data_get_notification_count(void)
{
	int count = 0;

	if (s_info.noti_list == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "notification list is NULL");
		return 0;
	}

	count = eina_list_count(s_info.noti_list);

	return count;
}

/**
 * @brief Gets notification item from notification list by index.
 * @param[in] index The order you want to get notification data
 */
noti_data_s *data_get_noti_item_by_index(int index)
{
	noti_data_s *noti_item = NULL;

	if (s_info.noti_list == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "notification list is NULL");
		return NULL;
	}

	noti_item = eina_list_nth(s_info.noti_list, index);
	if (noti_item == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get notification item");
		return NULL;
	}

	return noti_item;
}

/**
 * @brief Gets notification icon path by index.
 * @param[in] index The order you want to get notification data
 */
char *data_get_noti_icon_path(int index)
{
	noti_data_s *noti_item = NULL;
	char icon_path[PATH_MAX] = { 0, };

	noti_item = data_get_noti_item_by_index(index);
	if (noti_item == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get notification item");
		return NULL;
	}

	if (noti_item->icon_path == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "icon path is NULL");
		return NULL;
	}

	data_get_resource_path(noti_item->icon_path, icon_path, sizeof(icon_path));

	return strdup(icon_path);
}

/**
 * @brief Gets notification title by index.
 * @param[in] index The order you want to get notification data
 */
char *data_get_noti_title(int index)
{
	noti_data_s *noti_item = NULL;

	noti_item = data_get_noti_item_by_index(index);
	if (noti_item == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get notification item");
		return NULL;
	}

	return (noti_item->title) ? strdup(noti_item->title) : NULL;
}

/**
 * @brief Gets notification text by index.
 * @param[in] index The order you want to get notification data
 */
char *data_get_noti_body_text(int index)
{
	noti_data_s *noti_item = NULL;

	noti_item = data_get_noti_item_by_index(index);
	if (noti_item == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get notification item");
		return NULL;
	}

	return (noti_item->body_text) ? strdup(noti_item->body_text) : NULL;
}

/**
 * @brief Gets notification date by index.
 * @param[in] index The order you want to get notification data
 */
char *data_get_noti_date(int index)
{
	noti_data_s *noti_item = NULL;

	noti_item = data_get_noti_item_by_index(index);
	if (noti_item == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get notification item");
		return NULL;
	}

	return (noti_item->date) ? strdup(noti_item->date) : NULL;
}
