/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <system_settings.h>
#include <Elementary.h>
#include <efl_extension.h>
#include <app.h>
#include <dlog.h>
#include "$(appName).h"
#include "view.h"
#include "data.h"

#define IMG_PATH_NO_CONTENTS "images/no_items.png"
#define WIN_WIDTH 360
#define WIN_HEIGHT 360

static Evas_Object *_create_cover_view(Evas_Object *parent);
static Evas_Object *_create_card_view(Evas_Object *parent);
static Evas_Object *_create_expanded_title_layout(Evas_Object *parent, int index);
static Evas_Object *_create_expanded_body_text_label(Evas_Object *parent, int index);
static Eina_Bool _view_pop_cb(void *user_data, Elm_Object_Item *it);
static void _more_btn_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _txt_body_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source);

/**
 * @brief Hook to take necessary actions before main event loop starts.
 * Initialize UI resources and application's data
 * If this function returns true, the main loop of application starts
 * If this function returns false, the application is terminated
 */
static bool app_create(void *user_data)
{
	Evas_Object *layout_cover_view = NULL;
	Evas_Object *naviframe = NULL;

	/*
	 * Create essential objects for UI application.
	 * Window, Conformant, layout
	 */
	view_create();

	/*
	 * Create additional objects for Notification viewer UI.
	 * Circle surface, Naviframe
	 */
	view_notification_viewer_create();

	/*
	 * Initialize notification data.
	 */
	data_initialize();

	/*
	 * Get naviframe.
	 */
	naviframe = view_get_naviframe();
	if (naviframe == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to get naviframe", __func__, __LINE__);
		return false;
	}

	/*
	 * Create cover view of notification.
	 */
	layout_cover_view = _create_cover_view(naviframe);
	if (layout_cover_view == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to create cover view", __func__, __LINE__);
		return false;
	}

	/*
	 * Register event callback function to display detail view.
	 */
	view_push_item_to_naviframe(naviframe, layout_cover_view, _view_pop_cb, (void *) VIEW_TYPE_COVER);

	return true;
}

/**
 * @brief This callback function is called when another application.
 * sends the launch request to the application
 */
static void app_control(app_control_h app_control, void *user_data)
{
	/* Handle the launch request. */
}

/**
 * @brief This callback function is called each time.
 * the application is completely obscured by another application
 * and becomes invisible to the user
 */
static void app_pause(void *user_data)
{
	/* Take necessary actions when application becomes invisible. */
}

/**
 * @brief This callback function is called each time.
 * the application becomes visible to the user
 */
static void app_resume(void *user_data)
{
	/* Take necessary actions when application becomes visible. */
}

/**
 * @brief This callback function is called once after the main loop of the application exits.
 */
static void app_terminate(void *user_data)
{
	/*
	 * Finalize notification data.
	 */
	data_finalize();

	/*
	 * Release all resources.
	 */
	view_destroy();
}

/**
 * @brief This function will be called when the language is changed.
 */
static void ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/* APP_EVENT_LANGUAGE_CHANGED */
	char *locale = NULL;

	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);

	if (locale != NULL) {
		elm_language_set(locale);
		free(locale);
	}
	return;
}

/**
 * @brief Main function of the application.
 */
int main(int argc, char *argv[])
{
	int ret;

	ui_app_lifecycle_callback_s event_callback = {0, };
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	/*
	 * If you want to handling more events,
	 * please check the application lifecycle guide
	 */
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, NULL);

	ret = ui_app_main(argc, argv, &event_callback, NULL);
	if (ret != APP_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() failed. err = %d", ret);

	return ret;
}

/**
 * @brief Creates cover view displayed.
 * @param[in] parent The object to which you want to add this layout
 */
static Evas_Object *_create_cover_view(Evas_Object *parent)
{
	Evas_Object *layout_cover_view = NULL;
	Evas_Object *layout_more_btn = NULL;
	noti_data_s *noti_item = NULL;
	char edje_path[PATH_MAX] = { 0, };
	int count = 0;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] parent is NULL", __func__, __LINE__);
		return NULL;
	}

	count = data_get_notification_count();
	dlog_print(DLOG_INFO, LOG_TAG, "[%s:%d] notification count : %d", __func__, __LINE__, count);

	/*
	 * If there is no notification, create no contents layout.
	 */
	if (count == 0) {
		char icon_path[PATH_MAX] = { 0, };

		data_get_resource_path(IMG_PATH_NO_CONTENTS, icon_path, sizeof(icon_path));

		layout_cover_view = view_create_no_contents_layout(parent, "Notification", "No items", icon_path);
		if (layout_cover_view == NULL) {
			dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create no contents layout");
			return NULL;
		}

		return layout_cover_view;
	}

	data_get_resource_path(EDJ_FILE, edje_path, sizeof(edje_path));
	dlog_print(DLOG_INFO, LOG_TAG, " [%s:%d] edje path : %s", __func__, __LINE__, edje_path);

	layout_cover_view = view_create_layout(parent, edje_path, "layout_cover_view", NULL, NULL);
	if (layout_cover_view == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to create cover view layout", __func__, __LINE__);
		return NULL;
	}

	/*
	 * Get the first notification item by index.
	 */
	noti_item = data_get_noti_item_by_index(0);
	if (noti_item == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to get notification item", __func__, __LINE__);

		/*
		 * You can add the code to display no contents layout here.
		 */

		return NULL;
	}

	/*
	 * Create a cover view icon.
	 */
	if (noti_item->icon_path) {
		char icon_path[PATH_MAX] = { 0, };

		data_get_resource_path(noti_item->icon_path, icon_path, sizeof(icon_path));
		dlog_print(DLOG_INFO, LOG_TAG, "[%s:%d] icon path : %s", __func__, __LINE__, icon_path);

		view_set_image(layout_cover_view, "sw.icon", icon_path);
	}

	/*
	 * Set title for cover view.
	 */
	if (noti_item->title) {
		view_set_text(layout_cover_view, "txt.title", noti_item->title);
	}

	/*
	 * Set date for cover view.
	 */
	if (noti_item->date) {
		view_set_text(layout_cover_view, "txt.date", noti_item->date);
	}

	/*
	 * Set body text for cover view.
	 */
	if (noti_item->body_text) {
		view_set_text(layout_cover_view, "txt.body", noti_item->body_text);

		/*
		 * Register clicked event callback function.
		 */
		elm_object_signal_callback_add(layout_cover_view, "txt,body,clicked", "txt.body", _txt_body_clicked_cb, (void *) 0);
	}

	/*
	 * If there are several notification, create a more button.
	 */
	if (count > 1) {
		layout_more_btn = view_create_layout(layout_cover_view, edje_path, "layout_more_btn", NULL, NULL);
		if (layout_more_btn) {
			view_notification_set_more_button(layout_cover_view, "sw.more.btn", layout_more_btn, count);

			/*
			 * Register clicked event callback function.
			 */
			elm_object_signal_callback_add(layout_more_btn, "more,btn,clicked", "img.bg", _more_btn_clicked_cb, NULL);
		}
	}

	return layout_cover_view;
}

/**
 * @brief Creates collapsed view to display.
 * @param[in] parent The object to which you want to add this layout
 * @param[in] index The order you want to get notification data
 */
static Evas_Object *_create_collapsed_layout(Evas_Object *parent, int index)
{
	Evas_Object *layout_collapsed = NULL;
	noti_data_s *noti_item = NULL;
	char edje_path[PATH_MAX] = { 0, };

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] parent is NULL", __func__, __LINE__);
		return NULL;
	}

	noti_item = data_get_noti_item_by_index(index);
	if (noti_item == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to get notification item");
		return NULL;
	}

	data_get_resource_path(EDJ_FILE, edje_path, sizeof(edje_path));
	dlog_print(DLOG_INFO, LOG_TAG, " [%s:%d] edje path : %s", __func__, __LINE__, edje_path);

	layout_collapsed = view_create_layout(parent, edje_path, "layout_collapsed", NULL, NULL);
	if (layout_collapsed == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to create collapsed layout", __func__, __LINE__);
		return NULL;
	}

	evas_object_resize(layout_collapsed, WIN_WIDTH, WIN_HEIGHT);

	/*
	 * Set title for collapsed layout.
	 */
	if (noti_item->title) {
		view_set_text(layout_collapsed, "txt.title", noti_item->title);
	}

	/*
	 * Set date for collapsed layout.
	 */
	if (noti_item->date) {
		view_set_text(layout_collapsed, "txt.date", noti_item->date);
	}

	/*
	 * Set body text for collapsed layout.
	 */
	if (noti_item->body_text) {
		view_set_text(layout_collapsed, "txt.body", noti_item->body_text);

		/*
		 * Register clicked event callback function.
		 */
		elm_object_signal_callback_add(layout_collapsed, "txt,body,clicked", "txt.body", _txt_body_clicked_cb, (void *) index);
	}

	return layout_collapsed;
}

/**
 * @brief Creates expanded title layout.
 * @param[in] parent The object to which you want to add this layout
 * @param[in] index The order you want to get notification data
 */
static Evas_Object *_create_expanded_title_layout(Evas_Object *parent, int index)
{
	Evas_Object *layout_expanded = NULL;
	noti_data_s *noti_item = NULL;
	char edje_path[PATH_MAX] = { 0, };

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] parent is NULL", __func__, __LINE__);
		return NULL;
	}

	data_get_resource_path(EDJ_FILE, edje_path, sizeof(edje_path));

	layout_expanded = view_create_layout(parent, edje_path, "layout_expanded", NULL, NULL);
	if (layout_expanded == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to create expanded layout", __func__, __LINE__);
		return NULL;
	}

	/*
	 * Get the notification item by index.
	 */
	noti_item = data_get_noti_item_by_index(index);
	if (noti_item == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to get notification item", __func__, __LINE__);
		evas_object_del(layout_expanded);
		return NULL;
	}

	/*
	 * Set title for expanded view.
	 */
	if (noti_item->title) {
		view_set_text(layout_expanded, "txt.title", noti_item->title);
	}

	/*
	 * Set date for expanded view.
	 */
	if (noti_item->date) {
		view_set_text(layout_expanded, "txt.date", noti_item->date);
	}

	return layout_expanded;
}

/**
 * @brief Creates expanded body text label.
 * @param[in] parent The object to which you want to add this layout
 * @param[in] index The order you want to get notification data
 */
static Evas_Object *_create_expanded_body_text_label(Evas_Object *parent, int index)
{
	Evas_Object *label = NULL;
	noti_data_s *noti_item = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] parent is NULL", __func__, __LINE__);
		return NULL;
	}

	/*
	 * Get the notification item by index.
	 */
	noti_item = data_get_noti_item_by_index(index);
	if (noti_item == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to get notification item", __func__, __LINE__);
		return NULL;
	}

	if (noti_item->body_text == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] body text is NULL");
		return NULL;
	}

	label = view_notification_create_expanded_label(parent);
	if (label == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to create expanded label");
		return NULL;
	}

	view_notification_set_expanded_label_text(label, noti_item->body_text);

	return label;
}

/**
 * @brief Creates card view to display.
 * @param[in] parent The object to which you want to add this layout
 */
static Evas_Object *_create_card_view(Evas_Object *parent)
{
	Evas_Object *layout_card_view = NULL;
	Evas_Object *scroller = NULL;
	Evas_Object *box = NULL;
	char edje_path[PATH_MAX] = { 0, };

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] parent is NULL", __func__, __LINE__);
		return NULL;
	}

	data_get_resource_path(EDJ_FILE, edje_path, sizeof(edje_path));
	dlog_print(DLOG_INFO, LOG_TAG, " [%s:%d] edje path : %s", __func__, __LINE__, edje_path);

	/*
	 * Create card view of notification.
	 */
	layout_card_view = view_create_layout(parent, edje_path, "layout_card_view", NULL, NULL);
	if (layout_card_view == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to create expanded card view layout", __func__, __LINE__);
		return NULL;
	}

	/*
	 * Create scroller for card view.
	 */
	scroller = view_create_scroller(layout_card_view);
	if (scroller == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to create scroller", __func__, __LINE__);
		evas_object_del(layout_card_view);
		return NULL;
	}

	/*
	 * Create circle scroller for card view.
	 */
	view_create_circle_scroller(scroller);

	/*
	 * Create box for scroller.
	 */
	box = view_create_box(layout_card_view);
	if (box == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to create box", __func__, __LINE__);
		evas_object_del(scroller);
		evas_object_del(layout_card_view);
		return NULL;
	}

	elm_object_content_set(scroller, box);
	elm_object_part_content_set(layout_card_view, "sw.scroller", scroller);

	return layout_card_view;
}

/**
 * @brief The function is called when main view is popped at naviframe.
 * @param[in] user_data User data to pass to callback function
 * @param[in] it The item that set the callback on
 */
static Eina_Bool _view_pop_cb(void *user_data, Elm_Object_Item *it)
{
	int view_type = VIEW_TYPE_COVER;

	view_type = (view_type_e) user_data;
	dlog_print(DLOG_INFO, LOG_TAG, "[%s:%d] view type : %d", __func__, __LINE__, view_type);

	switch (view_type) {
	case VIEW_TYPE_COVER:
		ui_app_exit();
		break;
	case VIEW_TYPE_COLLAPSED:
	case VIEW_TYPE_EXPANDED:
		break;
	default:
		break;
	}

	return EINA_TRUE;
}

/**
 * @brief The function is called when more button is clicked.
 * @param[in] data A pointer to the data to pass to the callback function
 * @param[in] obj The edje object where the signal comes from
 * @param[in] emission The signal name
 * @param[in] source The signal source
 */
static void _more_btn_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	Evas_Object *naviframe = NULL;
	Evas_Object *layout_card_view = NULL;
	Evas_Object *box = NULL;
	Evas_Object *padding_bottom = NULL;
	int i = 0, count = 0;

	dlog_print(DLOG_INFO, LOG_TAG, "[%s:%d] More button is clicked", __func__, __LINE__);

	/*
	 * Get naviframe.
	 */
	naviframe = view_get_naviframe();
	if (naviframe == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to get naviframe.", __func__, __LINE__);
		return;
	}

	count = data_get_notification_count();
	dlog_print(DLOG_INFO, LOG_TAG, "[%s:%d] notification count : %d", __func__, __LINE__, count);

	/*
	 * Create card view to use scroller.
	 */
	layout_card_view = _create_card_view(naviframe);
	if (layout_card_view == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to create collapsed layout", __func__, __LINE__);
		return;
	}

	/*
	 * Get the box.
	 */
	box = view_get_box(layout_card_view);
	if (box == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to get box", __func__, __LINE__);
		evas_object_del(layout_card_view);
		return;
	}

	/*
	 * Create collapsed layout
	 */
	for (i = 0; i < count; i++) {
		Evas_Object *layout_collapsed = NULL;

		layout_collapsed = _create_collapsed_layout(layout_card_view, i);
		if (layout_collapsed == NULL) {
			dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to create collapsed layout", __func__, __LINE__);
			continue;
		}

		/*
		 * Append a item to box.
		 */
		view_append_item_to_box(box, layout_collapsed);

		/*
		 * Create bottom padding and append it to box.
		 */
		if (i != count - 1) {
			padding_bottom = view_create_scroller_padding(layout_card_view, 48);
			if (padding_bottom) {
				view_append_item_to_box(box, padding_bottom);
			}
		}
	}

	/*
	 * Push layout to naviframe and register event callback function to display expanded card.
	 */
	view_push_item_to_naviframe(naviframe, layout_card_view, _view_pop_cb, (void *) VIEW_TYPE_EXPANDED);
}

/**
 * @brief The function is called when body text is clicked.
 * @param[in] data A pointer to the data to pass to the callback function
 * @param[in] obj The edje object where the signal comes from
 * @param[in] emission The signal name
 * @param[in] source The signal source
 */
static void _txt_body_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	Evas_Object *naviframe = NULL;
	Evas_Object *layout_card_view = NULL;
	Evas_Object *layout_expanded_title = NULL;
	Evas_Object *label_body_text = NULL;
	Evas_Object *padding_bottom = NULL;
	Evas_Object *box = NULL;
	int index = 0;

	index = (int) data;

	dlog_print(DLOG_INFO, LOG_TAG, "[%s:%d] Text body is clicked", __func__, __LINE__);

	/*
	 * Get naviframe.
	 */
	naviframe = view_get_naviframe();
	if (naviframe == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to get naviframe.", __func__, __LINE__);
		return;
	}

	/*
	 * Create card view to use scroller.
	 */
	layout_card_view = _create_card_view(naviframe);
	if (layout_card_view == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to create card view", __func__, __LINE__);
		return;
	}

	/*
	 * Create title layout of expanded card.
	 */
	layout_expanded_title = _create_expanded_title_layout(layout_card_view, index);
	if (layout_expanded_title == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to create expanded layout", __func__, __LINE__);
		evas_object_del(layout_card_view);
		return;
	}

	/*
	 * Get the box.
	 */
	box = view_get_box(layout_card_view);
	if (box == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to get box", __func__, __LINE__);
		evas_object_del(layout_card_view);
		evas_object_del(layout_expanded_title);
		return;
	}

	/*
	 * Append a item to box.
	 */
	view_append_item_to_box(box, layout_expanded_title);

	/*
	 * Create body text label.
	 */
	label_body_text = _create_expanded_body_text_label(layout_card_view, index);
	if (label_body_text == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to create body text label", __func__, __LINE__);
		evas_object_del(layout_expanded_title);
		evas_object_del(layout_card_view);
		return;
	}

	/*
	 * Append a body text label to box.
	 */
	view_append_item_to_box(box, label_body_text);

	/*
	 * Create bottom padding and append it to box.
	 */
	padding_bottom = view_create_scroller_padding(layout_card_view, 100);
	if (padding_bottom) {
		view_append_item_to_box(box, padding_bottom);
	}

	/*
	 * Push layout to naviframe and register event callback function to display expanded card.
	 */
	view_push_item_to_naviframe(naviframe, layout_card_view, _view_pop_cb, (void *) VIEW_TYPE_EXPANDED);
}
