/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <Elementary.h>
#include <efl_extension.h>
#include <app.h>
#include <dlog.h>
#include "$(appName).h"
#include "view.h"
#include "data.h"

#define LABEL_STYLE_START "<font=Tizen:style=Regular><font_size=28><align=center><color=#FAFAFA><wrap=mixed>"
#define LABEL_STYLE_END "</wrap></color></align></font_size></font>"

static struct view_info {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *layout;
	Evas_Object *nf;
	Eext_Circle_Surface *circle_surface;
} s_info = {
	.win = NULL,
	.conform = NULL,
	.layout = NULL,
	.nf = NULL,
	.circle_surface = NULL,
};

static void _dummy_callback_cb(void *data, Evas_Object *obj, void *event_info);

/**
 * @brief Gets window.
 */
Evas_Object *view_get_window(void)
{
	return s_info.win;
}

/**
 * @brief Gets conformant.
 */
Evas_Object *view_get_conformant(void)
{
	return s_info.conform;
}

/**
 * @brief Gets naviframe.
 */
Evas_Object *view_get_naviframe(void)
{
	return s_info.nf;
}

/**
 * @brief Gets scroller.
 * @param[in] parent The object to which you want to add this layout
 */
Evas_Object *view_get_scroller(Evas_Object *parent)
{
	Evas_Object *scroller = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL");
		return NULL;
	}

	scroller = elm_object_part_content_get(parent, "sw.scroller");
	if (scroller == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get scroller");
		return NULL;
	}

	return scroller;
}

/**
 * @brief Gets box.
 * @param[in] parent The object to which you want to add this layout
 */
Evas_Object *view_get_box(Evas_Object *parent)
{
	Evas_Object *scroller = NULL;
	Evas_Object *box = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL");
		return NULL;
	}

	scroller = view_get_scroller(parent);
	if (scroller == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get scroller");
		return NULL;
	}

	box = elm_object_content_get(scroller);
	if (box == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get box");
		return NULL;
	}

	return box;
}

/**
 * @brief Creates essential objects: window, conformant and layout.
 */
Eina_Bool view_create(void)
{
	/* Create the window. */
	s_info.win = view_create_win(PACKAGE);
	if (s_info.win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a window.");
		return EINA_FALSE;
	}

	/* Create the conformant. */
	s_info.conform = view_create_conformant_without_indicator(s_info.win);
	if (s_info.conform == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a conformant");
		return EINA_FALSE;
	}

	/* Show the window after main view is set up. */
	evas_object_show(s_info.win);
	return EINA_TRUE;
}

/**
 * @brief Creates a basic window named package.
 * @param[in] pkg_name Name of the window
 */
Evas_Object *view_create_win(const char *pkg_name)
{
	Evas_Object *win = NULL;

	/*
	 * Window
	 * Create and initialize elm_win.
	 * elm_win is mandatory to manipulate the window.
	 */
	win = elm_win_util_standard_add(pkg_name, pkg_name);
	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_autodel_set(win, EINA_TRUE);

	/* Rotation setting. */
	if (elm_win_wm_rotation_supported_get(win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(win, (const int *)(&rots), 4);
	}

	evas_object_smart_callback_add(win, "delete,request", _dummy_callback_cb, NULL);

	return win;
}

/**
 * @brief Creates a conformant without indicator for wearable app.
 * @param[in] win The object to which you want to set this conformant
 * Conformant is mandatory for base GUI to have proper size
 */
Evas_Object *view_create_conformant_without_indicator(Evas_Object *win)
{
	/*
	 * Conformant
	 * Create and initialize elm_conformant.
	 * elm_conformant is mandatory for base GUI to have proper size
	 * when indicator or virtual keypad is visible.
	 */
	Evas_Object *conform = NULL;

	if (win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "window is NULL.");
		return NULL;
	}

	conform = elm_conformant_add(win);
	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(win, conform);

	evas_object_show(conform);

	return conform;
}

/**
 * @brief Creates a layout to target parent object with edje file.
 * @param[in] parent The object to which you want to add this layout
 * @param[in] file_path File path of the EDJ file will be used
 * @param[in] group_name Name of group in EDJ you want to set to
 * @param[in] cb_function The function called when back event is detected
 * @param[in] user_data The user data to be passed to the callback function
 */
Evas_Object *view_create_layout(Evas_Object *parent, const char *file_path, const char *group_name, Eext_Event_Cb cb_function, void *user_data)
{
	Evas_Object *layout = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return NULL;
	}

	/* Create layout using EDC(an edje file). */
	layout = elm_layout_add(parent);
	elm_layout_file_set(layout, file_path, group_name);

	/* Layout size setting. */
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	if (cb_function)
		eext_object_event_callback_add(layout, EEXT_CALLBACK_BACK, cb_function, user_data);

	evas_object_show(layout);

	return layout;
}

/**
 * @brief Creates and sets a layout to conformant.
 * @param[in] parent Target conformant object
 * @param[in] file_path File path of EDJ will be used
 * @param[in] group_name Group name in EDJ you want to set to layout
 * @param[in] cb_function The function will be called when the back event is detected
 * @param[in] user_data The user data to be passed to the callback functions
 */
Evas_Object *view_create_layout_for_conformant(Evas_Object *parent, const char *file_path, const char *group_name, Eext_Event_Cb cb_function, void *user_data)
{
	Evas_Object *layout = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return NULL;
	}

	/* Create layout for conformant. */
	if (file_path == NULL)
		layout = view_create_layout_by_theme(parent, "layout", "application", "default");
	else
		layout = view_create_layout(parent, file_path, group_name, cb_function, user_data);

	if (layout == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "layout is NULL.");
		return NULL;
	}

	elm_object_content_set(parent, layout);

	return layout;
}

/**
 * @brief Creates a layout with theme.
 * @param[in] parent Object to which you want to add this layout
 * @param[in] class_name The class of the group
 * @param[in] group_name Group name in EDJ that you want to set to layout
 * @param[in] style The style to use
 */
Evas_Object *view_create_layout_by_theme(Evas_Object *parent, const char *class_name, const char *group_name, const char *style)
{
	/*
	 * Layout
	 * Create and initialize elm_layout.
	 * view_create_layout_by_theme() is used to create layout by using premade edje file.
	 */
	Evas_Object *layout = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return NULL;
	}

	layout = elm_layout_add(parent);
	elm_layout_theme_set(layout, class_name, group_name, style);
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	evas_object_show(layout);

	return layout;
}

/**
 * @brief Destroys window and frees its resources.
 */
void view_destroy(void)
{
	if (s_info.win == NULL)
		return;

	evas_object_del(s_info.win);
}

/**
 * @brief Sets image to given part.
 * @param[in] parent Object has part to which you want to set this image
 * @param[in] part_name Part name to which you want to set this image
 * @param[in] image_path Path of the image file
 */
void view_set_image(Evas_Object *parent, const char *part_name, const char *image_path)
{
	Evas_Object *image = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return;
	}

	image = elm_object_part_content_get(parent, part_name);
	if (image == NULL) {
		image = elm_image_add(parent);
		if (image == NULL) {
			dlog_print(DLOG_ERROR, LOG_TAG, "failed to create an image object.");
			return;
		}

		elm_object_part_content_set(parent, part_name, image);
	}

	if (EINA_FALSE == elm_image_file_set(image, image_path, NULL)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to set image: %s", image_path);
		return;
	}

	evas_object_show(image);

	return;
}

/**
 * @brief Sets text to the part.
 * @param[in] parent Object has part to which you want to set text
 * @param[in] part_name Part name to which you want to set the text
 * @param[in] text Text you want to set to the part
 */
void view_set_text(Evas_Object *parent, const char *part_name, const char *text)
{
	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return;
	}

	/* Set text of target part object. */
	elm_object_part_text_set(parent, part_name, text);
}

/**
 * @brief Creates a naviframe and sets to parent.
 * @param[in] parent Object to which you want to set naviframe
 * Add callback function will be operated when back key is pressed
 */
Evas_Object *view_create_naviframe(Evas_Object *parent)
{
	/* Naviframe
	 * Create and initialize elm_naviframe.
	 * Naviframe stands for navigation frame.
	 * elm_naviframe is a views manager for applications.
	 */
	Evas_Object *nf = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return NULL;
	}

	nf = elm_naviframe_add(parent);

	elm_object_part_content_set(parent, "elm.swallow.content", nf);
	eext_object_event_callback_add(nf, EEXT_CALLBACK_BACK, eext_naviframe_back_cb, NULL);
	eext_object_event_callback_add(nf, EEXT_CALLBACK_MORE, eext_naviframe_more_cb, NULL);

	evas_object_show(nf);

	return nf;
}

/**
 * @brief Pushes item to naviframe.
 * @param[in] nf Naviframe has several views
 * @param[in] item Object will be added to naviframe
 * @param[in] pop_cb Function will be operated when this item is popped from naviframe
 * @param[in] user_data Data passed to the 'pop_cb' function
 * Naviframe make changing of view is easy and effectively
 */
Elm_Object_Item* view_push_item_to_naviframe(Evas_Object *nf, Evas_Object *item, Elm_Naviframe_Item_Pop_Cb pop_cb, void *user_data)
{
	Elm_Object_Item* nf_it = NULL;

	if (nf == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "naviframe is NULL.");
		return NULL;
	}

	if (item == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "item is NULL.");
		return NULL;
	}

	nf_it = elm_naviframe_item_push(nf, NULL, NULL, NULL, item, "empty");

	if (pop_cb != NULL)
		elm_naviframe_item_pop_cb_set(nf_it, pop_cb, user_data);

	return nf_it;
}

/**
 * @brief Creates a circle scroller.
 * @param[in] scroller The scroller to which you want to add
 */
void view_create_circle_scroller(Evas_Object *scroller)
{
	Evas_Object *circle_scroller = NULL;

	if (s_info.circle_surface == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "circle surface is NULL");
		return;
	}

	if (scroller == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "scroller is NULL");
		return;
	}

	circle_scroller = eext_circle_object_scroller_add(scroller, s_info.circle_surface);
	eext_circle_object_scroller_policy_set(circle_scroller, ELM_SCROLLER_POLICY_OFF, ELM_SCROLLER_POLICY_AUTO);
	eext_rotary_object_event_activated_set(circle_scroller, EINA_TRUE);
}

/**
 * @brief Creates padding for scroller.
 * @param[in] parent The object to which you want to add this layout
 * @param[in] height The height size to which you want to set
 */
Evas_Object *view_create_scroller_padding(Evas_Object *parent, int height)
{
	Evas_Object *padding = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] parent is NULL");
		return NULL;
	}

	padding = evas_object_rectangle_add(evas_object_evas_get(parent));
	if (padding == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to add rectangle", __func__, __LINE__);
		return NULL;
	}

	evas_object_color_set(padding, 0, 0, 0, 255);
	evas_object_size_hint_min_set(padding, 360, height);
	evas_object_show(padding);

	return padding;
}

/**
 * @brief Creates a scroller.
 * @param[in] parent The object to which you want to add
 */
Evas_Object *view_create_scroller(Evas_Object *parent)
{
	Evas_Object *scroller = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return NULL;
	}

	scroller = elm_scroller_add(parent);
	if (scroller == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to add scroller");
		return NULL;
	}

	elm_scroller_bounce_set(scroller, EINA_FALSE, EINA_FALSE);
	elm_scroller_policy_set(scroller, ELM_SCROLLER_POLICY_OFF, ELM_SCROLLER_POLICY_OFF);
	elm_scroller_single_direction_set(scroller, ELM_SCROLLER_SINGLE_DIRECTION_HARD);
	evas_object_show(scroller);

	return scroller;
}

/**
 * @brief Creates a box.
 * @param[in] parent The object to which you want to add
 */
Evas_Object *view_create_box(Evas_Object *parent)
{
	Evas_Object *box = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return NULL;
	}

	box = elm_box_add(parent);
	if (box == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to add box");
		return NULL;
	}

	evas_object_size_hint_weight_set(box, 0.0, 0.0);
	evas_object_size_hint_align_set(box, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_min_set(box, 360, 360);
	evas_object_size_hint_max_set(box, 360, -1);
	elm_box_horizontal_set(box, EINA_FALSE);
	evas_object_show(box);

	return box;
}

/**
 * @brief Adds an object at the end of the packed list.
 * @param[in] box The box object
 * @param[in] item The object to add to the box
 */
void view_append_item_to_box(Evas_Object *box, Evas_Object *item)
{
	if (box == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "box is NULL");
		return;
	}

	if (item == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "item is NULL");
		return;
	}

	elm_box_pack_end(box, item);
}

/**
 * @brief Creates a no contents layout.
 * @param[in] parent The parent object of this no contents layout
 * @param[in] title The text you want to display at top of the layout
 * @param[in] text The text you want to display at bottom of the layout
 * @param[in] icon_path The image path you want to display at center of the layout
 */
Evas_Object *view_create_no_contents_layout(Evas_Object *parent, const char *title, const char *text, const char *icon_path)
{
	Evas_Object *layout = NULL;

	layout = view_create_layout_by_theme(parent, "layout", "nocontents", "default");
	if (layout == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create no contents layout");
		return NULL;
	}

	if (title) {
		elm_object_part_text_set(layout, "elm.text.title", title);
	}

	if (text) {
		elm_object_part_text_set(layout, "elm.text", text);
	}

	if (icon_path) {
		view_set_image(layout, "elm.swallow.icon", icon_path);
	}

	return layout;
}

/**
 * @brief Creates additional objects for Notification viewer.
 */
void view_notification_viewer_create(void)
{
	/*
	 * Eext Circle Surface Creation.
	 */
	s_info.circle_surface = eext_circle_surface_conformant_add(s_info.conform);

	s_info.layout = view_create_layout_for_conformant(s_info.conform, NULL, NULL, NULL, NULL);

	s_info.nf = view_create_naviframe(s_info.layout);
	if (s_info.nf == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a naviframe.");
		evas_object_del(s_info.win);
		s_info.win = NULL;
		return;
	}
}

/**
 * @brief Creates a label and sets label options.
 * @param[in] parent The object to which you want to add this label
 */
Evas_Object *view_notification_create_expanded_label(Evas_Object *parent)
{
	Evas_Object *label = NULL;

	label = elm_label_add(parent);
	if (label == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to add label");
		return NULL;
	}

	evas_object_size_hint_weight_set(label, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_show(label);

	elm_label_line_wrap_set(label, ELM_WRAP_MIXED);
	elm_label_wrap_width_set(label, 360 - 33 - 33);
	elm_label_ellipsis_set(label, EINA_FALSE);

	return label;
}

/**
 * @brief Sets a text to label object.
 * @param[in] label The label Object has part to which you want to set text
 * @param[in] text Text you want to set to the part
 */
void view_notification_set_expanded_label_text(Evas_Object *label, const char *text)
{
	char *markup_text = NULL;
	char buf[1024] = { 0, };

	if (label == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "label is NULL.");
		return;
	}

	if (text == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "text is NULL.");
		return;
	}

	markup_text = elm_entry_utf8_to_markup(text);
	snprintf(buf, sizeof(buf), "%s%s%s", LABEL_STYLE_START, markup_text, LABEL_STYLE_END);
	free(markup_text);

	elm_object_text_set(label, buf);
}

/**
 * @brief Sets more button to cover view.
 * @param[in] parent The object to which you want to add this layout
 * @param[in] part_name Part name to which you want to set this layout
 * @param[in] layout_more_btn The more button layout
 * @param[in] count The number of notification
 */
void view_notification_set_more_button(Evas_Object *parent, const char *part_name, Evas_Object *layout_more_btn, int count)
{
	char buf[128] = { 0, };

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL");
		return;
	}

	if (part_name == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "part name is NULL");
		return;
	}

	if (layout_more_btn == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "more button is NULL");
		return;
	}

	elm_object_part_content_set(parent, part_name, layout_more_btn);

	snprintf(buf, sizeof(buf), "%d", count);
	elm_object_part_text_set(layout_more_btn, "txt.more.num", buf);
}

/**
 * Below functions are static functions.
 * You don't need to modify them.
 */

/**
 * @brief Function will be operated when registered event is triggered.
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void _dummy_callback_cb(void *data, Evas_Object *obj, void *event_info)
{
	/*
	 * Write your code here for smart callback.
	 */
	ui_app_exit();
}
