/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <Elementary.h>
#include <app.h>
#include <dlog.h>
#include <system_settings.h>
#include <efl_extension.h>

#include "$(appName).h"
#include "view.h"
#include "data.h"

#define MENU_ITEMS 9
#define DISPLAY_ITEMS 6

static Eina_Bool _naviframe_pop_cb(void *data, Elm_Object_Item *it);
static void _gl_display(void);
static void _create_sub_list(void *data, Evas_Object *obj, void *event_info);
static void _gl_clicked_cb(void *data, Evas_Object *obj, void *event_info);

/*
 * @brief: Hook to take necessary actions before main event loop starts
 * Initialize UI resources and application's data
 * If this function returns true, the main loop of application starts
 * If this function returns false, the application is terminated
 */
static bool app_create(void *user_data)
{
	/* Hook to take necessary actions before main event loop starts
	   Initialize UI resources and application's data
	   If this function returns true, the main loop of application starts
	   If this function returns false, the application is terminated */

	Evas_Object *nf = NULL;
	Evas_Object *genlist = NULL;
	int i = 0;

	/*
	 * Create essential objects.
	 */
	view_create();

	/*
	 * Create a genlist.
	 */
	nf = view_get_naviframe();
	genlist = view_create_circle_genlist(nf);

	/*
	 * Append a genlist's item as a title.
	 */
	view_append_item_to_genlist(genlist, "menu.title", NULL, NULL, NULL);

	/*
	 * Append menu items to genlist.
	 */
	for (i = 0; i < MENU_ITEMS; i++) {
		view_append_item_to_genlist(genlist, "1text.1icon", (void *)i, _create_sub_list, (void *)i);
	}

	/*
	 * Create a genlist's item as a padding item.
	 * Padding item makes genlist's items is located at the middle of the screen.
	 */
	view_append_item_to_genlist(genlist, "padding", NULL, NULL, NULL);

	/*
	 * Push the genlist at naviframe.
	 */
	view_push_item_to_naviframe(nf, genlist, _naviframe_pop_cb, NULL);

	view_set_genlist(genlist);

	return true;
}

/*
 * @brief: This callback function is called when another application
 * sends the launch request to the application
 */
static void app_control(app_control_h app_control, void *user_data)
{
	/* Handle the launch request. */
}

/*
 * @brief: This callback function is called each time
 * the application is completely obscured by another application
 * and becomes invisible to the user
 */
static void app_pause(void *user_data)
{
	/* Take necessary actions when application becomes invisible. */
}

/*
 * @brief: This callback function is called each time
 * the application becomes visible to the user
 */
static void app_resume(void *user_data)
{
	/* Take necessary actions when application becomes visible. */
}

/*
 * @brief: This callback function is called once after the main loop of the application exits
 */
static void app_terminate(void *user_data)
{
	/* Release all resources. */

	view_destroy();
}

/*
 * @brief: This function will be called when the language is changed
 */
static void ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;

	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);

	if (locale != NULL) {
		elm_language_set(locale);
		free(locale);
	}
	return;
}

/*
 * @brief: main function of the application
 */
int main(int argc, char *argv[])
{
	int ret;

	ui_app_lifecycle_callback_s event_callback = {0, };
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	/*
	 * If you want to handling more events,
	 * Please check the application lifecycle guide
	 */
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, NULL);

	ret = ui_app_main(argc, argv, &event_callback, NULL);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() is failed. err = %d", ret);
	}

	return ret;
}

/*
 * @note
 * Below functions are static functions.
 */

/*
 * @brief: This function will be operated when the first item of the naviframe is going to popped.
 * @param[data]: Data needed in this function
 * @param[it]: Item of naviframe
 */
static Eina_Bool _naviframe_pop_cb(void *data, Elm_Object_Item *it)
{
	ui_app_exit();

	return EINA_FALSE;
}

/*
 * @brief: This function will be operated when the first genlist's item is clicked
 */
static void _gl_display(void)
{
	Evas_Object *nf = NULL;
	Evas_Object *display_genlist = NULL;
	int i = 0;

	/*
	 * Create a genlist of display menu.
	 */
	nf = view_get_naviframe();
	display_genlist = view_create_circle_genlist(nf);

	view_append_item_to_genlist(display_genlist, "display.title", NULL, NULL, NULL);

	for (i = 0; i < DISPLAY_ITEMS; i++) {
		if (i == 0 || i == 3 || i == 5) {
			view_append_item_to_genlist(display_genlist, "1text", (void *)i, NULL, NULL);
		} else if (i == 1 || i == 2) {
			view_append_item_to_genlist(display_genlist, "1text.1icon.1", (void *)i, _gl_clicked_cb, (void *)i);
		} else {
			view_append_item_to_genlist(display_genlist, "2text", (void *)i, NULL, NULL);
		}
	}

	view_append_item_to_genlist(display_genlist, "padding", NULL, NULL, NULL);

	view_push_item_to_naviframe(nf, display_genlist, NULL, NULL);
}

/*
 * @brief: This function will be operated when one of genlist's items is clicked
 * @param[data]: Data passed from 'elm_genlist_item_append' as seventh parameter
 * @param[obj]: Genlist
 * @param[event_info]: A pointer to data which is totally dependent on the smart object's implementation and semantic for the given event
 */
static void _create_sub_list(void *data, Evas_Object *obj, void *event_info)
{
	int index = (int)data;

	switch (index) {
	case 0:
		/*
		 * You can do as you want by adding code here
		 * when you select the item of list.
		 */
		_gl_display();
		break;
	case 1:
		/*
		 * You can do as you want by adding code here
		 * when you select the item of list.
		 */
		break;
	case 2:
		/*
		 * You can do as you want by adding code here
		 * when you select the item of list.
		 */
		break;
	case 3:
		/*
		 * You can do as you want by adding code here
		 * when you select the item of list.
		 */
		break;
	case 4:
		/*
		 * You can do as you want by adding code here
		 * when you select the item of list.
		 */
		break;
	case 5:
		/*
		 * You can do as you want by adding code here
		 * when you select the item of list.
		 */
		break;
	case 6:
		/*
		 * You can do as you want by adding code here
		 * when you select the item of list.
		 */
		break;
	case 7:
		/*
		 * You can do as you want by adding code here
		 * when you select the item of list.
		 */
		break;
	case 8:
		/*
		 * You can do as you want by adding code here
		 * when you select the item of list.
		 */
		break;
	default:
		dlog_print(DLOG_ERROR, LOG_TAG, "wrong approach");
		break;
	}
}

/*
 * @brief: This function will be operated when one of genlist's items is clicked
 * @param[data]: Data passed from 'elm_genlist_item_append' as seventh parameter
 * @param[obj]: Genlist
 * @param[event_info]: A pointer to data which is totally dependent on the smart object's implementation and semantic for the given event
 */
static void _gl_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	Elm_Object_Item *it = (Elm_Object_Item *)event_info;
	Evas_Object *content = NULL;
	const char *w_type;

	/*
	 * Get content of genlist's item.
	 */
	content = elm_object_item_part_content_get(it, "elm.icon");

	/*
	 * Check whether type of content is elm_check or not.
	 */
	w_type = elm_object_widget_type_get(content);
	if (w_type == NULL) {
		return;
	}

	if (strcmp(w_type, "elm_check")) return;

	/*
	 * Control check state.
	 */
	if (content) {
		int index;
		Eina_Bool state;

		index = (int)data;
		state = elm_check_state_get(content);

		if (state == EINA_TRUE) {
			elm_check_state_set(content, EINA_FALSE);
			data_set_display_int_value(index, EINA_FALSE);
			dlog_print(DLOG_INFO, LOG_TAG, "selected check [%p] state [%d]", content, EINA_FALSE);
		} else {
			elm_check_state_set(content, EINA_TRUE);
			data_set_display_int_value(index, EINA_TRUE);
			dlog_print(DLOG_INFO, LOG_TAG, "selected check [%p] state [%d]", content, EINA_TRUE);
		}
	}
}

/* End of file */
