/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <tizen.h>
#include <app.h>
#include <Elementary.h>
#include <efl_extension.h>
#include <system_settings.h>
#include <dlog.h>

#include "$(appName).h"
#include "view.h"
#include "data.h"

#define EDJ_FILE "edje/main.edj"
#define GRP_MAIN "task-manager"
#define GRP_ITEM "item"
#define GRP_SCROLLER "task.item.list"
#define GRP_NOAPPS "group.noapps"
#define PART_NOAPPS "no.apps.page"

static void _clicked_cb(void *user_data, Evas_Object *obj, void *event_info);
static void _delete_btn_clicked_cb(void *user_data, Evas_Object *obj, const char *emission, const char *source);

/*
 * @brief: Hook to take necessary actions before main event loop starts
 * Initialize UI resources and application's data
 * If this function returns true, the main loop of application starts
 * If this function returns false, the application is terminated
 */
static bool app_create(void *user_data)
{
	int i;
	char image_path[PATH_MAX] = {0, };
	char file_path[PATH_MAX] = {0, };
	char appname[PATH_MAX] = {0, };
	Evas_Object *layout = NULL;
	Evas_Object *box = NULL;
	Evas_Object *no_apps_page = NULL;

	view_create();

	data_get_resource_path(EDJ_FILE, file_path, sizeof(file_path));

	/*
	 * Create layout, scroller for scrolling between several items,
	 * and box for management of several items
	 */
	view_task_manager_create(file_path, GRP_MAIN, GRP_SCROLLER);

	layout = view_get_layout();
	/* Create circular index for display the current showing page */
	view_create_circular_index(layout, "task.item.index");

	/* Create bottom and circular button */
	view_set_button(layout, "task.del.all.btn", "bottom", NULL, "CLOSE ALL", NULL, NULL, _clicked_cb, layout);

	box = view_get_box();
	/* Create items as many as you want */
	for (i = 0; i < data_get_screenshot_count(); i++) {
		Evas_Object *item = NULL;

		/* Get appname to make item will be displayed */
		if (data_get_appname(i, appname, sizeof(appname)) != 1) {
			continue;
		}

		/* Get image file path to make item will be displayed */
		if (data_get_screenshot(i, image_path, sizeof(image_path)) != 1) {
			continue;
		}

		/* Create a view consists of 1 title, 1 icon and delete button */
		item = view_create_1text_1image_delbutton_item(layout, file_path, GRP_ITEM);
		/* Set 'appname' to the part named 'recent.item.name' in EDJ file */
		view_set_text(item, "recent.item.name", appname);
		view_set_image(item, "recent.item.image", image_path);
		/* Register callback function will be called on "clicked" */
		view_set_customized_event_callback(item, "clicked", "delete_button", _delete_btn_clicked_cb, box);
		/* Add item to the box to manage easily */
		view_set_item_to_box(box, item);
		/* Create circular index to display the index of the items */
		view_add_circular_index(i, data_get_screenshot_count());
	}
	/* Highlight index which show the current item */
	view_sync_index_with_item(0);

	/* Create a view displayed when no apps is exist */
	no_apps_page = view_create_layout_for_part(layout, file_path, GRP_NOAPPS, PART_NOAPPS);

	return true;
}

/*
 * @brief: This callback function is called when another application
 * sends the launch request to the application
 */
static void app_control(app_control_h app_control, void *user_data)
{
	/* Handle the launch request. */
}

/*
 * @brief: This callback function is called each time
 * the application is completely obscured by another application
 * and becomes invisible to the user
 */
static void app_pause(void *user_data)
{
	/* Take necessary actions when application becomes invisible. */
}

/*
 * @brief: This callback function is called each time
 * the application becomes visible to the user
 */
static void app_resume(void *user_data)
{
	/* Take necessary actions when application becomes visible. */
}

/*
 * @brief: This callback function is called once after the main loop of the application exits
 */
static void app_terminate(void *user_data)
{
	/* Release all resources. */
	view_destroy();
}

/*
 * @brief: This function will be called when the language is changed
 */
static void ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;

	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);

	if (locale != NULL) {
		elm_language_set(locale);
		free(locale);
	}
	return;
}

/*
 * @brief: Main function of the application
 */
int main(int argc, char *argv[])
{
	int ret;

	ui_app_lifecycle_callback_s event_callback = {0, };
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	/*
	 * If you want to handling more events,
	 * Please check the application lifecycle guide
	 */
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, NULL);

	ret = ui_app_main(argc, argv, &event_callback, NULL);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() is failed. err = %d", ret);
	}

	return ret;
}

/*
 * @brief: The function will be called on click
 * @param[user_data]: The data passed to the function
 * @param[obj]: The object clicked
 * @param[event_info]: Information regarding this event
 */
static void _clicked_cb(void *user_data, Evas_Object *obj, void *event_info)
{
	elm_object_signal_emit(user_data, "del.all.item", "item.list");
	view_set_bottom_button_disabled(user_data, EINA_TRUE);
	elm_object_signal_emit(user_data, "noapps", "show,noapps");
}

/*
 * @brief: The function will be called each delete button is clicked
 * @param[user_data]: The data passed to the function
 * @param[obj]: The item clicked
 * @param[emission]: The appointed signal between C file and EDJ file
 * @param[source]: The appointed source that is regarding with this event
 */
static void _delete_btn_clicked_cb(void *user_data, Evas_Object *obj, const char *emission, const char *source)
{
	Evas_Object *box = user_data;
	Eina_List *children;
	Evas_Object *child;
	int idx = -1;
	int i = 0;
	int num_item = 0;

	children = elm_box_children_get(box);
	num_item = eina_list_count(children) - 2;

	EINA_LIST_FREE(children, child) {
		if (child == obj) {
			idx = i;
			break;
		}
		i++;
	}

	elm_box_unpack(box, obj);
	evas_object_del(obj);

	view_update_index();

	if (idx == num_item) {
		idx = idx - 2;
	} else {
		idx = idx - 1;
	}
	view_sync_index_with_item(idx);
}
