/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <tizen.h>
#include <app.h>
#include <Elementary.h>
#include <efl_extension.h>
#include <dlog.h>

#include "$(appName).h"
#include "view.h"

#define PACKAGE "org.example.$(appName)"

static struct view_info {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *scroller;
	Evas_Object *box;
	Evas_Object *layout;
	Evas_Object *index;
} s_info = {
	.win = NULL,
	.conform = NULL,
	.scroller = NULL,
	.box = NULL,
	.layout = NULL,
	.index = NULL,
};

static void _win_delete_request_cb(void *user_data, Evas_Object *obj, void *event_info);
static void _layout_back_cb(void *user_data, Evas_Object *obj, void *event_info);
static void _scrolled_cb(void *user_data, Evas_Object *scroller, void *event_info);

static char *style_even[11] = {
		"item/even_1",
		"item/even_2",
		"item/even_3",
		"item/even_4",
		"item/even_5",
		"item/even_6",
		"item/even_7",
		"item/even_8",
		"item/even_9",
		"item/even_10",
		"item/even_11",
};
static char *style_odd[11] = {
		"item/odd_1",
		"item/odd_2",
		"item/odd_3",
		"item/odd_4",
		"item/odd_5",
		"item/odd_6",
		"item/odd_7",
		"item/odd_8",
		"item/odd_9",
		"item/odd_10",
		"item/even_11",
};

/*
 * @brief: Get the window
 */
Evas_Object *view_get_win(void)
{
	return s_info.win;
}

/*
 * @brief: Get the layout
 */
Evas_Object *view_get_layout(void)
{
	return s_info.layout;
}

/*
 * @brief: Get the box
 */
Evas_Object *view_get_box(void)
{
	return s_info.box;
}

/*
 * @brief: Create Essential Object window, conformant and layout
 */
void view_create(void)
{
	/* Create window */
	s_info.win = view_create_win(PACKAGE);
	if (s_info.win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a window.");
		return;
	}

	/* Create conformant */
	s_info.conform = view_create_conformant_without_indicator(s_info.win);
	if (s_info.conform == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a conformant");
		return;
	}

	/* Show window after main view is set up */
	evas_object_show(s_info.win);
}

/*
 * @brief: Make a basic window named package
 * @param[pkg_name]: Name of the window
 */
Evas_Object *view_create_win(const char *pkg_name)
{
	Evas_Object *win = NULL;

	/* Window */
	/*
	 * Create and initialize elm_win.
	 * elm_win is mandatory to manipulate window
	 */
	win = elm_win_util_standard_add(pkg_name, pkg_name);
	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_autodel_set(win, EINA_TRUE);

	/* Rotation setting */
	if (elm_win_wm_rotation_supported_get(win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(win, (const int *)(&rots), 4);
	}

	evas_object_smart_callback_add(win, "delete,request", _win_delete_request_cb, NULL);

	return win;
}

/*
 * @brief: Make a conformant without indicator for wearable app
 * @param[win]: The object to which you want to set this conformant
 * Conformant is mandatory for base GUI to have proper size
 */
Evas_Object *view_create_conformant_without_indicator(Evas_Object *win)
{
	/*
	 * Conformant
	 * Create and initialize elm_conformant.
	 * elm_conformant is mandatory for base GUI to have proper size
	 * when indicator or virtual keypad is visible.
	 */
	Evas_Object *conform = NULL;

	if (win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "window is NULL.");
		return NULL;
	}

	conform = elm_conformant_add(win);
	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(win, conform);

	evas_object_show(conform);

	return conform;
}

/*
 * @brief: Make a layout to target parent object with edje file
 * @param[parent]: The object to which you want to add this layout
 * @param[file_path]: File path of EDJ file will be used
 * @param[group_name]: Name of group in EDJ you want to set to
 * @param[cb_function]: The function will be called when back event is detected
 * @param[user_data]: The user data to be passed to the callback functions
 */
Evas_Object *view_create_layout(Evas_Object *parent, const char *file_path, const char *group_name, Eext_Event_Cb cb_function, void *user_data)
{
	Evas_Object *layout = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return NULL;
	}

	/* Create layout by EDC(edje file) */
	layout = elm_layout_add(parent);
	elm_layout_file_set(layout, file_path, group_name);

	/* Layout size setting */
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	if (cb_function)
		eext_object_event_callback_add(layout, EEXT_CALLBACK_BACK, cb_function, user_data);

	evas_object_show(layout);

	return layout;
}

/*
 * @brief: Make and set a layout to the part
 * @param[parent]: Object to which you want to set this layout
 * @param[file_path]: File path of EDJ will be used
 * @param[group_name]: Group name in EDJ that you want to set to layout
 * @param[part_name]: Part name to which you want to set this layout
 */
Evas_Object *view_create_layout_for_part(Evas_Object *parent, char *file_path, char *group_name, char *part_name)
{
	Evas_Object *layout = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return NULL;
	}

	layout = elm_layout_add(parent);
	elm_layout_file_set(layout, file_path, group_name);
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	evas_object_show(layout);

	elm_object_part_content_set(parent, part_name, layout);

	return layout;
}

/*
 * @brief: Destroy window and free important data to finish this application
 */
void view_destroy(void)
{
	if (s_info.win == NULL) {
		return;
	}

	evas_object_del(s_info.win);
}

/*
 * @brief: Set image to given part
 * @param[parent]: Object has part to which you want to set this image
 * @param[part_name]: Part name to which you want to set this image
 * @param[image_path]: Path of the image file
 */
void view_set_image(Evas_Object *parent, const char *part_name, const char *image_path)
{
	Evas_Object *image = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return;
	}

	image = elm_object_part_content_get(parent, part_name);
	if (image == NULL) {
		image = elm_image_add(parent);
		if (image == NULL) {
			dlog_print(DLOG_ERROR, LOG_TAG, "failed to create an image object.");
			return;
		}

		elm_object_part_content_set(parent, part_name, image);
	}

	if (EINA_FALSE == elm_image_file_set(image, image_path, NULL)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to set image.");
		return;
	}

	evas_object_show(image);

	return;
}

/*
 * @brief: Set text to the part
 * @param[parent]: Object has part to which you want to set text
 * @param[part_name]: Part name to which you want to set the text
 * @param[text]: Text you want to set to the part
 */
void view_set_text(Evas_Object *parent, const char *part_name, const char *text)
{
	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return;
	}

	/* Set text of target part object */
	elm_object_part_text_set(parent, part_name, text);
}

/*
 * @brief: Make and set button.
 * @param[parent]: Object to which you want to set the button
 * @param[part_name]: Name of part to which you want to set the button
 * @param[style]: Style of the button
 * @param[image_path]: Path of image file will be used as button icon
 * @param[text]: The text will be written on the button
 * @param[down_cb]: Function will be operated when the button is pressed
 * @param[up_cb]: Function will be operated when the button is released
 * @param[clicked_cb]: Function will be operated when the button is clicked
 * @param[user_data]: Data passed to the 'clicked_cb' function
 */
void view_set_button(Evas_Object *parent, const char *part_name, const char *style, const char *image_path, const char *text,
		Evas_Object_Event_Cb down_cb, Evas_Object_Event_Cb up_cb, Evas_Smart_Cb clicked_cb, void *user_data)
{
	Evas_Object *btn = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return;
	}

	btn = elm_button_add(parent);
	if (btn == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create button.");
		return;
	}

	if (style)
		elm_object_style_set(btn, style);

	evas_object_size_hint_weight_set(btn, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(parent, part_name, btn);

	if (text)
		elm_object_text_set(btn, text);

	if (image_path)
		view_set_image(btn, NULL, image_path);

	if (down_cb)
		evas_object_event_callback_add(btn , EVAS_CALLBACK_MOUSE_DOWN, down_cb, user_data);
	if (up_cb)
		evas_object_event_callback_add(btn, EVAS_CALLBACK_MOUSE_UP, up_cb, user_data);
	if (clicked_cb)
		evas_object_smart_callback_add(btn, "clicked", clicked_cb, user_data);

	evas_object_show(btn);
}

/*
 * @brief: Set the function will be called when the appointed signal is occurred
 * @param[item]: The object triggered the signal
 * @param[signal]: The appointed signal to trigger the function
 * @param[source]: The appointed source that normally indicate the object triggered the event
 * @param[signal_cb]: The function will be called when the signal is detected
 * @param[user_data]: The data passed to the 'signal_cb' function
 */
void view_set_customized_event_callback(Evas_Object *item, char *signal, char *source, Edje_Signal_Cb signal_cb, void *user_data)
{
	elm_object_signal_callback_add(item, signal, source, signal_cb, user_data);
}

/*
 * @brief: Create additional object for voice memo
 * @param[file_path]: Name of the EDJ file
 * @param[group_name]: Name of group in EDJ file
 * @param[scroller_part]: Name of group in EDJ file
 */
void view_task_manager_create(char *file_path, char *group_name, char *scroller_part)
{
	s_info.layout = view_create_layout(s_info.conform, file_path, group_name, _layout_back_cb, NULL);

	elm_object_content_set(s_info.conform, s_info.layout);

	s_info.scroller = view_create_horizontal_scroller(s_info.layout, scroller_part, _scrolled_cb, NULL);

	s_info.box = view_create_horizontal_box(s_info.scroller, NULL);

	evas_object_show(s_info.layout);
}

/*
 * @brief: Create a horizontal scroller, has callback function, effect.
 * @param[parent] : Object has part which you want to put in this scroller
 * @param[part_name] : Name of part you want to put in
 * @param[scrolled_cb] : Function will be operated when scroller is scrolled
 * @param[user_data] : Data passed to the function
 */
Evas_Object *view_create_horizontal_scroller(Evas_Object *parent, char *part_name, Evas_Smart_Cb scrolled_cb, void *user_data)
{
	Evas_Object *scroller = NULL;

	scroller = elm_scroller_add(parent);
	elm_scroller_content_min_limit(scroller, EINA_FALSE, EINA_TRUE);
	elm_scroller_bounce_set(scroller, EINA_TRUE, EINA_TRUE);
	/* Hide horizontal & vertical scroll bar */
	elm_scroller_policy_set(scroller, ELM_SCROLLER_POLICY_OFF, ELM_SCROLLER_POLICY_OFF);
	evas_object_size_hint_weight_set(scroller, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(scroller, EVAS_HINT_FILL, EVAS_HINT_FILL);
	/* Can be scrolled horizontally */
	elm_object_scroll_lock_y_set(scroller, EINA_TRUE);
	/* You can show effect when you reach to end of scroller */
	elm_object_style_set(scroller, "effect");

	evas_object_smart_callback_add(scroller, "scroll", scrolled_cb, NULL);
	/* pass part name as a second argument for the part which you want to fill */
	elm_object_part_content_set(parent, part_name, scroller);
	elm_scroller_page_size_set(scroller, ICON_WIDTH, ICON_HEIGHT);

	evas_object_show(scroller);

	return scroller;
}

/*
 * @brief: Create a horizontal box and put into scroller, also has function operated when box size is changed
 * @param[parent] : Object which you want to add box to in this app is scroller
 * @param[user_data] : Data passed to the function
 */
Evas_Object *view_create_horizontal_box(Evas_Object *parent, void *user_data)
{
	Evas_Object *box = NULL;
	Evas_Object *padding_start = NULL;
	Evas_Object *padding_end = NULL;

	box = elm_box_add(parent);

	evas_object_size_hint_weight_set(box, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_fill_set(box, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_box_horizontal_set(box, EINA_TRUE);
	elm_box_homogeneous_set(box, EINA_FALSE);
	elm_box_padding_set(box, 25, 0);

	padding_start = evas_object_rectangle_add(evas_object_evas_get(parent));
	evas_object_size_hint_min_set(padding_start, 47, ITEM_LAYOUT_HEIGHT);
	evas_object_color_set(padding_start, 0, 0, 0, 0);
	evas_object_show(padding_start);
	evas_object_data_set(box, PADDING_START, padding_start);

	padding_end = evas_object_rectangle_add(evas_object_evas_get(parent));
	evas_object_size_hint_min_set(padding_end, 47, ITEM_LAYOUT_HEIGHT);
	evas_object_color_set(padding_end, 0, 0, 0, 0);
	evas_object_show(padding_end);

	elm_box_pack_start(box, padding_start);
	elm_box_pack_end(box, padding_end);

	elm_object_content_set(parent, box);

	evas_object_show(box);

	return box;
}

/*
 * @brief: Create a circular shaped index and put into given part, but not viewable yet
 * @param[parent] : Object that this index will be added
 * @param[part_name] : Name of part you want to put in
 */
void view_create_circular_index(Evas_Object *parent, char *part_name)
{
	s_info.index = elm_index_add(parent);
	elm_object_style_set(s_info.index, "circle");
	evas_object_size_hint_weight_set(s_info.index, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(s_info.index, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_index_horizontal_set(s_info.index, EINA_TRUE);
	elm_index_autohide_disabled_set(s_info.index, EINA_TRUE);
	elm_object_part_content_set(parent, part_name, s_info.index);
}

/*
 * @brief: Add item to box
 * @param[box] : Object that item will be added
 * @param[item] : Object that will be added to box
 */
void view_set_item_to_box(Evas_Object *box, Evas_Object *item)
{
	Evas_Object *padding = NULL;

	padding = evas_object_data_get(box, PADDING_START);

	if (padding == NULL) {
		return;
	}
	elm_box_pack_after(box, item, padding);
}

/*
 * @brief: Add a circular index viewable to layout
 * @param[idx] : Number of index and it's style
 * @param[item_count] : Number of items
 */
void view_add_circular_index(int idx, int item_count)
{
	int is_odd = item_count % 2;
	Elm_Object_Item *it = NULL;

	it = elm_index_item_append(s_info.index, NULL, NULL, (void *)idx);
	if (is_odd) {
		elm_object_item_style_set(it, style_odd[9 - (item_count / 2) + idx]);
	} else {
		elm_object_item_style_set(it, style_even[10 - (item_count / 2) + idx]);
	}
	elm_index_level_go(s_info.index, 0);
}

/*
 * @brief: Synchronize index highlight with item is shown at the moment
 * @param[idx] : Number of item is showing
 */
void view_sync_index_with_item(int idx)
{
	Elm_Object_Item *it = NULL;

	it = elm_index_item_find(s_info.index, (void*)idx);

	if (it) {
		elm_index_item_selected_set(it, EINA_TRUE);
	} else {
	}
}

/*
 * @brief: Synchronize index highlight with page is shown at the moment
 */
void view_sync_index_with_page_on_scroll(void)
{
	int cur_page = -1;
	Elm_Object_Item *it = NULL;

	elm_scroller_current_page_get(s_info.scroller, &cur_page, NULL);

	it = elm_index_item_find(s_info.index, (void*)cur_page);

	if (it) {
		elm_index_item_selected_set(it, EINA_TRUE);
	} else {
	}
}

/*
 * @brief: Update index on both situation page is changed and item is removed or added
 */
void view_update_index(void)
{
	int num_item = 0;
	Eina_List *list = NULL;
	int i;

	elm_index_item_clear(s_info.index);

	list = elm_box_children_get(s_info.box);
	num_item = eina_list_count(list) - 2;
	if (num_item >= 1) {
		for (i = 0; i < num_item; i++) {
			view_add_circular_index(i, num_item);
		}
	} else {
		elm_object_signal_emit(s_info.layout, "del.all.item", "item.list");
		view_set_bottom_button_disabled(s_info.layout, EINA_TRUE);
		elm_object_signal_emit(s_info.layout, "noapps", "show,noapps");
	}
}

/*
 * @brief: Create item has 1text, 1image, and delete button at once
 * @param[parent] : Object that this item will be added
 * @param[file_path] : The name of the EDJ file will be used
 * @param[group_name] : The group name in the EDJ file
 * You can customize the position of text and image inside this function
 */
Evas_Object *view_create_1text_1image_delbutton_item(Evas_Object *parent, char *file_path, char *group_name)
{
	Evas_Object *item_layout = NULL;

	item_layout = view_create_layout(parent, file_path, group_name, NULL, NULL);

	evas_object_show(item_layout);

	return item_layout;
}

/*
 * @brief: Display or not display the bottom button
 * @param[parent] : Object that this button will be added
 * @param[is_disabled] : Whether bottom button show or not
 */
void view_set_bottom_button_disabled(Evas_Object *parent, Eina_Bool is_disabled)
{
	if (is_disabled) {
		elm_object_signal_emit(parent, "del,btn,hide", "del.all.btn");
	} else {
		elm_object_signal_emit(parent, "del,btn,show", "del.all.btn");
	}
}

/*
 * @brief: The function will be called when the window is deleted
 * @param[user_data] : The data passed to the function
 * @param[obj] : The object, normally window
 * @param[event_info] : Information regarding with the event
 */
static void _win_delete_request_cb(void *user_data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

/*
 * @brief: The function will be called when the scroll event is occurred
 * @param[user_data] : The data passed to the function
 * @param[scroller] : The scroller that make this app can be scrolled
 * @param[event_info] : Information regarding with the event
 */
static void _scrolled_cb(void *user_data, Evas_Object *scroller, void *event_info)
{
	view_sync_index_with_page_on_scroll();
}

/*
 * @brief: The function will be called when the back event is detected
 * @param[user_data] : The data passed to the function
 * @param[obj] : The object, normally layout
 * @param[event_info] : Information regarding with the event
 */
static void _layout_back_cb(void *user_data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

