/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <tizen.h>
#include "$(appName).h"
#include <widget_service.h>
#include <widget_errno.h>
#include <bundle.h>

typedef struct widget_instance_data {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *label;
} widget_instance_data_s;

/**
 *  @brief Take necessary actions
 *         when widget instance is created.
 */
static int widget_instance_create(widget_context_h context, bundle *content, int w, int h, void *user_data)
{
	widget_instance_data_s *wid = (widget_instance_data_s*) malloc(sizeof(widget_instance_data_s));
	int ret;

	if (content != NULL) {
		/**
		 *  Recover the previous status with the bundle object.
		 */
	}

	/* Window */
	ret = widget_app_get_elm_win(context, &wid->win);
	if (ret != WIDGET_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to get window. err = %d", ret);
		return WIDGET_ERROR_FAULT;
	}

	evas_object_resize(wid->win, w, h);

	/* Conformant */
	wid->conform = elm_conformant_add(wid->win);
	evas_object_size_hint_weight_set(wid->conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(wid->win, wid->conform);
	evas_object_show(wid->conform);

	Evas_Object *rect;
	rect = evas_object_rectangle_add(evas_object_evas_get(wid->win));
	evas_object_resize(rect, w, h);
	evas_object_move(rect, 0, 0);
	evas_object_color_set(rect, 100, 100, 100, 255);
	evas_object_show(rect);

	/* Label */
	wid->label = elm_label_add(wid->conform);
	evas_object_resize(wid->label, w, h / 3);
	evas_object_move(wid->label, 0, h / 3);
	evas_object_show(wid->label);
	elm_object_text_set(wid->label, "<align=center>Widget Service</align>");

	/* Show window after base gui is set up */
	evas_object_show(wid->win);

	widget_app_context_set_tag(context, wid);
	return WIDGET_ERROR_NONE;
}

/**
 *  @brief Take necessary actions
 *         when widget instance is destroyed.
 */
static int widget_instance_destroy(widget_context_h context, widget_app_destroy_type_e reason, bundle *content, void *user_data)
{
	if (reason != WIDGET_APP_DESTROY_TYPE_PERMANENT) {
		/**
		 *   Save the current status at the bundle object.
		 */
	}

	widget_instance_data_s *wid = NULL;
	widget_app_context_get_tag(context, (void**)&wid);

	if (wid->win)
		evas_object_del(wid->win);

	free(wid);

	return WIDGET_ERROR_NONE;
}

/**
 *  @brief Take necessary actions
 *         when widget instance becomes invisible.
 */
static int widget_instance_pause(widget_context_h context, void *user_data)
{
	return WIDGET_ERROR_NONE;
}

/**
 *  @brief Take necessary actions
 *         when widget instance becomes visible.
 */
static int widget_instance_resume(widget_context_h context, void *user_data)
{
	return WIDGET_ERROR_NONE;
}

/**
 *  @brife Take necessary actions
 *         when widget instance is updated.
 */
static int widget_instance_update(widget_context_h context, bundle *content, int force, void *user_data)
{
	const char* data_str = NULL;
	const char* default_str = "Something went wrong";

	/**
	 *  Get a data for key 'string'
	 *  to display it on the screen of widget.
	 */
	int ret = bundle_get_str(content, "string", (char**)&data_str);
	if (ret != BUNDLE_ERROR_NONE) {
		/**
		 *  An error has occured,
		 *  we need to set data_str by default string,
		 *  "Something went wrong".
		 */
		data_str = default_str;
	}

	/**
	 *  Get widget instance
	 *  to display the data_str on the screen of widget
	 */
	widget_instance_data_s *wid = NULL;
	ret = widget_app_context_get_tag(context, (void**)&wid);
	if (ret != WIDGET_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get widget instance data from context: %d", ret);
		return ret;
	}

	const char* format = "<align=center>%s</align>";
	int len = snprintf(NULL, 0, format, data_str) + 1;
	char* str = (char*)malloc(len);
	snprintf(str, len, format, data_str);
	elm_object_text_set(wid->label, str);
	free(str);

	return WIDGET_ERROR_NONE;
}

/**
 *  @brief Take necessary actions
 *         when the size of widget instance was changed.
 */
static int widget_instance_resize(widget_context_h context, int w, int h, void *user_data)
{
	return WIDGET_ERROR_NONE;
}

/**
 *  @brief Take necessary actions
 *         when the language was changed.
 */
static void widget_app_lang_changed(app_event_info_h event_info, void *user_data)
{

}

/**
 *  @brief Take necessary actions
 *         when the region format was changed.
 */
static void widget_app_region_changed(app_event_info_h event_info, void *user_data)
{

}

/**
 *  @brief Take necessary actions before main event loop starts.
 */
static widget_class_h widget_app_create(void *user_data)
{
	app_event_handler_h handlers[5] = {NULL, };

	widget_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED],
									APP_EVENT_LANGUAGE_CHANGED, widget_app_lang_changed, user_data);
	widget_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED],
									APP_EVENT_REGION_FORMAT_CHANGED, widget_app_region_changed, user_data);

	widget_instance_lifecycle_callback_s obj_callback = {0,};
	obj_callback.create = widget_instance_create;
	obj_callback.destroy = widget_instance_destroy;
	obj_callback.pause = widget_instance_pause;
	obj_callback.resume = widget_instance_resume;
	obj_callback.resize = widget_instance_resize;
	obj_callback.update = widget_instance_update;

	return widget_app_class_create(obj_callback, user_data);
}

/**
 *  @brief Take necessary actions before widget terminate.
 */
static void widget_app_terminate(void *user_data)
{

}

/**
 *  @brief Main function of widget.
 */
int main(int argc, char *argv[])
{
	widget_app_lifecycle_callback_s ops = {0,};
	int ret;

	ops.create = widget_app_create;
	ops.terminate = widget_app_terminate;

	ret = widget_app_main(argc, argv, &ops, NULL);
	if (ret != WIDGET_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "widget_app_main() is failed. err = %d", ret);

	return ret;
}