/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <app_manager.h>
#include <app_common.h>
#include "$(appName).h"
#include "data.h"

typedef int (*get_info_func)(char **);
typedef char* (*get_path_func)(void);

static struct data_info {
	int paths_count;
	path_type_t *paths;
	app_info_t app_info;
} s_info = {
	.paths_count = 0,
	.paths = NULL,
	.app_info = {0,},
};

static void _get_info(get_info_func func, char **info);
static void _get_app_icon_path(char **icon_path);
static char *_get_path(get_path_func func);
static bool _add_path(char *path, const char *name);
static bool _is_same_string(const char *s1, const char *s2);

/**
 * @brief Initialization function for data module.
 * This function acquires application's related information: appId,
 * name, version, application's icon path and a set of paths to the
 * application's resource directories: data, cache, resource, shared data,
 * shared resource, shared trusted, external data, external cache,
 * external shared data, tep resource.
 */
void data_initialize(void)
{
	_get_info(app_get_id, &s_info.app_info.id);
	_get_info(app_get_name, &s_info.app_info.name);
	_get_info(app_get_version, &s_info.app_info.version);

	_get_app_icon_path(&s_info.app_info.icon_path);

	_add_path(_get_path(app_get_data_path), "Data");
	_add_path(_get_path(app_get_cache_path), "Cache");
	_add_path(_get_path(app_get_resource_path), "Resource");
	_add_path(_get_path(app_get_shared_data_path), "Shared data");
	_add_path(_get_path(app_get_shared_resource_path), "Shared resource");
	_add_path(_get_path(app_get_shared_trusted_path), "Shared trusted");
	_add_path(_get_path(app_get_external_data_path), "External data");
	_add_path(_get_path(app_get_external_cache_path), "External cache");
	_add_path(_get_path(app_get_tep_resource_path), "Tep resource");
}

/**
 * @brief Data module finalization function.
 * All the memory allocated during application run time is freed.
 */
void data_finalize(void)
{
	int i;

	for (i = 0; i < s_info.paths_count; i++)
		free(s_info.paths[i].path);

	free(s_info.paths);

	free(s_info.app_info.id);
	free(s_info.app_info.name);
	free(s_info.app_info.version);
	free(s_info.app_info.icon_path);
}

/**
 * @brief This function gets fully qualified path and custom name for its identification based on provided index.
 * @param[in] path_index The index of the path entry to be returned.
 * @param[out] path The path structure.
 * @return This function returns 'true' if the path structure was successfully obtained
 * for given path_index, otherwise 'false' is returned.
 */
bool data_get_path(int path_index, path_type_t **path)
{
	if (path_index < 0 || path_index >= s_info.paths_count) {
		dlog_print(DLOG_WARN, LOG_TAG, "data_get_path() failed. Index out of range.");
		return false;
	}

	*path = &s_info.paths[path_index];

	return true;
}

/**
 * @brief This function gets the application's information structure.
 * @param[out] app_info The structure containing the application's information (identifier, name and version).
 */
void data_get_app_info(app_info_t **app_info)
{
	*app_info = &s_info.app_info;
}

/**
 * @brief Function gets the number of files stored in the given directory and all its sub-directories.
 * @param[in] path The directory path to be searched.
 * @param[out] files_count The number of files found in given directory including sub-directories.
 * @return This function returns 'true' if no failure occurred, otherwise 'false' is returned.
 */
bool data_get_files_count(const char *path, int *files_count)
{
	DIR *dir_h;
	struct dirent *dir;
	int subdir_len;
	char *subdir;

	dir_h = opendir(path);
	if (!dir_h) {
		dlog_print(DLOG_ERROR, LOG_TAG, "opendir() failed");
		return false;
	}

	while ((dir = readdir(dir_h)) != NULL)
		if (dir->d_type == DT_REG) {
			(*files_count)++;
		} else if (dir->d_type == DT_DIR) {
			if (!_is_same_string(dir->d_name, ".") &&
				!_is_same_string(dir->d_name, "..")) {
				subdir_len = strlen(path) + strlen(dir->d_name) + 2;
				subdir = (char *)calloc(1, subdir_len);
				snprintf(subdir, subdir_len, "%s%s/", path, dir->d_name);
				data_get_files_count(subdir, files_count);
				free(subdir);
			}
		}

	closedir(dir_h);

	return true;
}

/**
 * @brief Internal wrapper function for application's information acquisition.
 * @param[in] func The pointer to the function to be invoked.
 * @param[out] info The information acquired by function 'func'.
 */
static void _get_info(get_info_func func, char **info)
{
	int ret;

	ret = func(info);
	if (ret != APP_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to get valid information. Err = %d", ret);
}

/**
 * @brief Internal function which obtains the absolute path to the application's
 * icon file.
 * @param[out] icon_path The path to the application's icon file.
 */
static void _get_app_icon_path(char **icon_path)
{
	int ret;
	char *app_id = NULL;
	app_info_h app_info = NULL;

	ret = app_get_id(&app_id);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_get_id() failed. Err = %d.", ret);
		return;
	}

	ret = app_info_create(app_id, &app_info);
	if (ret != APP_MANAGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_info_create() failed. Err = %d.", ret);
		free(app_id);
		return;
	}

	ret = app_info_get_icon(app_info, icon_path);
	if (ret != APP_MANAGER_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "app_info_get_icon() failed. Err = %d.", ret);

	free(app_id);
	app_info_destroy(app_info);
}

/**
 * @brief Internal wrapper function for application's path acquisition.
 * @param[in] func The pointer to the function to be invoked.
 * @return The acquired path is returned. If an error occurs, the NULL value is returned.
 */
static char *_get_path(get_path_func func)
{
	char *path;

	path = func();
	if (!path) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to get valid path.");
		return NULL;
	}

	return strdup(path);
}

/**
 * @brief Adds and stores new path entry.
 * The path entry consists of fully qualified path and custom name for path identification.
 * @param[in] path The fully qualified path to be added.
 * @param[in] name The custom name for path identification.
 * @return The function returns 'true' if the path entry was successfully added,
 * otherwise 'false' is returned.
 */
static bool _add_path(char *path, const char *name)
{
	path_type_t *temp = NULL;

	if (!path || !name) {
		dlog_print(DLOG_ERROR, LOG_TAG, "data_add_path() failed. Invalid input arguments.");
		return false;
	}

	temp = realloc(s_info.paths, sizeof(path_type_t) * (s_info.paths_count + 1));
	if (!temp) {
		dlog_print(DLOG_ERROR, LOG_TAG, "memory allocation failed for '%s' path entry.", name);
		return false;
	}

	s_info.paths = temp;
	s_info.paths[s_info.paths_count].name = name;
	s_info.paths[s_info.paths_count].path = path;

	s_info.paths_count++;

	return true;
}

/**
 * @brief Internal function which checks whether two strings are the same.
 * @param[in] s1 The first string.
 * @param[in] s2 The second string.
 * @return This function returns 'true' is both given strings are the same,
 * otherwise 'false' is returned.
 */
static bool _is_same_string(const char *s1, const char *s2)
{
	if (!s1 || !s2)
		return false;

	return (strncmp(s1, s2, strlen(s1)) == 0 && strlen(s1) == strlen(s2));
}
