/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <app.h>
#include <Elementary.h>
#include <system_settings.h>
#include <device/battery.h>
#include <utils_i18n.h>
#include "$(appName).h"
#include "data.h"
#include "view.h"
#include "types.h"

static struct _app_info {
	Ecore_Timer *battery_timer;
} s_info = {
	.battery_timer = NULL,
};

static Eina_Bool _battery_status_timer_cb(void *data);
static int _path_files_query_cb(const char *path);
static void _app_info_query_cb(app_info_t **app_info);
static bool _path_query_cb(path_type_t **path);
static void _get_display_name(char *localization, char **display_name);
static void _init_regional_settings(void);
static void _update_orientation(int angle);

/**
 * @brief Hook to take necessary actions before main event loop starts.
 * Initialize UI resources and application's data.
 * If this function returns true, the main loop of application starts.
 * If this function returns false, the application is terminated.
 */
static bool app_create(void *user_data)
{
	/* Hook to take necessary actions before main event loop starts
	   Initialize UI resources and application's data
	   If this function returns true, the main loop of application starts
	   If this function returns false, the application is terminated */

	data_initialize();

	view_set_callbacks(_app_info_query_cb, _path_query_cb, _path_files_query_cb);

	if (!view_create(NULL))
		return false;

	_init_regional_settings();
	_update_orientation(view_get_rotation());

	return true;
}

/**
 * @brief This callback function is called when another application
 * sends a launch request to the application.
 */
static void app_control(app_control_h app_control, void *user_data)
{
	/* Handle the launch request. */
}

/**
 * @brief This callback function is called each time
 * the application is completely obscured by another application
 * and becomes invisible to the user.
 */
static void app_pause(void *user_data)
{
	/* Take necessary actions when application becomes invisible. */
}

/**
 * @brief This callback function is called each time
 * the application becomes visible to the user.
 */
static void app_resume(void *user_data)
{
	/* Take necessary actions when application becomes visible. */
}

/**
 * @brief This callback function is called once after the main loop of the application exits.
 */
static void app_terminate(void *user_data)
{
	/* Release all resources. */
	app_event_handler_h *handlers = (app_event_handler_h *)user_data;

	data_finalize();

	if (s_info.battery_timer)
		ecore_timer_del(s_info.battery_timer);

	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_BATTERY]);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);
	ui_app_remove_event_handler(handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED]);
	ui_app_remove_event_handler(handlers[APP_EVENT_LANGUAGE_CHANGED]);
	ui_app_remove_event_handler(handlers[APP_EVENT_REGION_FORMAT_CHANGED]);

	view_destroy();
}

/**
 * @brief This function will be called when the battery level gets low.
 * The battery status is stored in event_info structure. In case the battery
 * status drops below the critical level, the timer is invoked to monitor
 * critical level exit.
 * @param[in] event_info The structure containing battery level status,
 * which may be acquired with app_event_get_low_battery_status() function.
 * @param[in] user_data The user data passed to the ui_app_add_event_handler() function.
 */
static void ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	int ret;
	app_event_low_battery_status_e status;

	ret = app_event_get_low_battery_status(event_info, &status);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_event_get_low_battery_status() failed. Err = %d", ret);
		return;
	}

	view_update_battery_status(status);

	if (!s_info.battery_timer)
		s_info.battery_timer = ecore_timer_add(1.0, _battery_status_timer_cb, NULL);
}

/**
 * @brief This function will be called when the memory level gets low.
 * The memory status is stored in event_info structure.
 * @param[in] event_info The structure containing memory level status,
 * which may be acquired with app_event_get_low_memory_status() function.
 * @param[in] user_data The user data passed to the ui_app_add_event_handler() function.
 */
static void ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	int ret;
	app_event_low_memory_status_e status;

	ret = app_event_get_low_memory_status(event_info, &status);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_event_get_low_memory_status() failed. Err = %d.", ret);
		return;
	}

	view_update_memory_status(status);
}

/**
 * @brief This function will be called when the device orientation is changed.
 * The device orientation angle is stored in event_info structure.
 * @param[in] event_info The structure containing device orientation angle,
 * which may be acquired with app_event_get_low_memory_status() function.
 * @param[in] user_data The user data passed to the ui_app_add_event_handler() function.
 */
static void ui_app_device_orientation_changed(app_event_info_h event_info, void *user_data)
{
	int ret;
	app_device_orientation_e orientation;

	ret = app_event_get_device_orientation(event_info, &orientation);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_event_get_device_orientation() failed. Err = %d.", ret);
		return;
	}

	_update_orientation((int)orientation);
}

/**
 * @brief This function will be called when the language is changed.
 * The current language code is stored in event_info structure, which is
 * then converted to the readable language name using i18n functions.
 * @param[in] event_info The structure containing current language name,
 * which may be acquired with app_event_get_language() function.
 * @param[in] user_data The user data passed to the ui_app_add_event_handler() function.
 */
static void ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	int ret;
	char *locale;
	char *lang_name = NULL;

	ret = app_event_get_language(event_info, &locale);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_event_get_language() failed. Err = %d.", ret);
		return;
	}

	if (locale != NULL) {
		elm_language_set(locale);
		_get_display_name(locale, &lang_name);
		view_update_language(lang_name);
		free(locale);
		free(lang_name);
	}

	return;
}

/**
 * @brief This function will be called when the region formatting is changed.
 * The current region format setting is stored in event_info structure, which is
 * then converted to the readable country name using i18n functions.
 * @param[in] event_info The structure containing current region format setting,
 * which may be acquired with app_event_get_region_format() function.
 * @param[in] user_data The user data passed to the ui_app_add_event_handler() function.
 */
static void ui_app_region_format_changed(app_event_info_h event_info, void *user_data)
{
	int ret;
	char *region = NULL;
	char *region_name = NULL;

	ret = app_event_get_region_format(event_info, &region);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "model_app_event_get_region_format() failed. Err = %d", ret);
		return;
	}

	if (region != NULL) {
		_get_display_name(region, &region_name);
		view_update_region_format(region_name);
		free(region);
		free(region_name);
	}
}

/**
 * @brief Main function of the application.
 */
int main(int argc, char *argv[])
{
	int ret;

	ui_app_lifecycle_callback_s event_callback = {0, };
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	/*
	 * If you want to handle more events,
	 * please check the application life cycle guide.
	 */

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, NULL);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, NULL);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_device_orientation_changed, NULL);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, NULL);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_format_changed, NULL);

	ret = ui_app_main(argc, argv, &event_callback, (void *)handlers);
	if (ret != APP_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() failed. err = %d", ret);

	return ret;
}

/**
 * @brief Internal callback function invoked with battery timer interval.
 * This function is responsible for monitoring the battery level once it gets
 * below the critical level. After the battery exists the critical level,
 * the UI state is updated with custom define value.
 * @param[in] data The user data passed to the callback attachment function.
 * @return This function returns ECORE_CALLBACK_RENEW to continue the timer,
 * or ECORE_CALLBACK_CANCEL to stop it.
 */
static Eina_Bool _battery_status_timer_cb(void *data)
{
	device_battery_level_e status;

	int ret = device_battery_get_level_status(&status);
	if (ret != DEVICE_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function device_battery_get_level_status() failed. Err = %d.", ret);
		return ECORE_CALLBACK_RENEW;
	}

	if (status != DEVICE_BATTERY_LEVEL_CRITICAL &&
		status != DEVICE_BATTERY_LEVEL_EMPTY)
		view_update_battery_status(DEVICE_BATTERY_LEVEL_NORMAL);
	else
		return ECORE_CALLBACK_RENEW;

	s_info.battery_timer = NULL;

	return ECORE_CALLBACK_CANCEL;
}

/**
 * @brief Internal callback function invoked when the user taps the path item in the paths tab.
 * This function obtains the files list stored in the directory pointed by the 'path' argument
 * and finally updates the popup window's content with the acquired files list.
 * @param[in] path The full path.
 */
static int _path_files_query_cb(const char *path)
{
	int files_count = 0;

	data_get_files_count(path, &files_count);

	return files_count;
}

/**
 * @brief Internal callback function invoked when the view module needs to fill the 'Application' tab's content.
 * @param[out] app_info The structure containing application's related information (id, name, version).
 */
static void _app_info_query_cb(app_info_t **app_info)
{
	data_get_app_info(app_info);
}

/**
 * @brief Internal callback function invoked when the view module needs to fill the 'Paths' tab's content.
 * @param[out] path The path information structure.
 * @return This function returns 'true' to continue paths query or 'false' to
 * stop the query procedure.
 */
static bool _path_query_cb(path_type_t **path)
{
	static int path_index = 0;
	bool ret;

	ret = data_get_path(path_index, path);

	path_index++;
	if (!ret)
		path_index = 0;

	return ret;
}

/**
 * @brief Internal function which obtains the readable name for provided
 * localization string.
 * @param[in] localization The localization string.
 * @param[out] display name The readable name corresponding to the provided
 * localization string.
 */
static void _get_display_name(char *localization, char **display_name)
{
	int ret;
	int disp_name_size = 0;
	i18n_uchar wide_disp_name[256] = {0,};

	if (!display_name) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_get_display_name() failed. Invalid input argument.");
		return;
	}

	i18n_ulocale_get_display_name(localization, "en_US", wide_disp_name, 256, &disp_name_size);
	ret = get_last_result();
	if (ret == I18N_ERROR_NONE) {
		*display_name = (char *)calloc(sizeof(char), disp_name_size + 1);
		i18n_ustring_copy_au_n(*display_name, wide_disp_name, disp_name_size);
	} else {
		dlog_print(DLOG_ERROR, LOG_TAG, "i18n_ulocale_get_display_name() failed. Err = %d.", ret);
		*display_name = (char *)calloc(sizeof(char), strlen(localization) + 1);
		strncpy(*display_name, localization, strlen(localization));
	}
}

/**
 * @brief Internal function which initializes the regional settings display
 * (language and region format).
 */
static void _init_regional_settings(void)
{
	int ret;
	char *code = NULL;
	char *disp_name = NULL;

	ret = system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &code);
	if (ret == SYSTEM_SETTINGS_ERROR_NONE) {
		_get_display_name(code, &disp_name);
		view_update_language(disp_name);
		free(disp_name);
		free(code);
	} else {
		dlog_print(DLOG_ERROR, LOG_TAG, "system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE) failed. Err = %d.", ret);
	}

	ret = system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_COUNTRY, &code);
	if (ret == SYSTEM_SETTINGS_ERROR_NONE) {
		_get_display_name(code, &disp_name);
		view_update_region_format(disp_name);
		free(disp_name);
		free(code);
	} else {
		dlog_print(DLOG_ERROR, LOG_TAG, "system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_COUNTRY) failed. Err = %d.", ret);
	}
}

/**
 * @brief Internal function which interprets the window rotation angle and displays
 * the relevant information.
 * @param[in] angle The window rotation angle.
 */
static void _update_orientation(int angle)
{
	switch (angle) {
	case APP_DEVICE_ORIENTATION_0:
		view_update_orientation("Natural position", angle);
		break;
	case APP_DEVICE_ORIENTATION_90:
		view_update_orientation("Left-side up", angle);
		break;
	case APP_DEVICE_ORIENTATION_180:
		view_update_orientation("Up-side down", angle);
		break;
	case APP_DEVICE_ORIENTATION_270:
		view_update_orientation("Right-side up", angle);
		break;
	default:
		view_update_orientation("Unknown orientation", angle);
	}
}
